/* GNOME DB library
 * Copyright (C) 1999-2001 The Free Software Foundation
 *
 * AUTHORS:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-connection-properties.h"
#include <libgnomedb/gnome-db-util.h>
#include <bonobo/bonobo-i18n.h>
#include <glade/glade.h>

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbConnectionPropertiesPrivate {
	GladeXML *xml;
	GdaConnection *cnc;

	/* widgets */
	GtkWidget *main_widget;
};

static void gnome_db_connection_properties_class_init (GnomeDbConnectionPropertiesClass *klass);
static void gnome_db_connection_properties_init       (GnomeDbConnectionProperties *props,
						       GnomeDbConnectionPropertiesClass *klass);
static void gnome_db_connection_properties_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static gboolean
get_widgets (GnomeDbConnectionProperties *props)
{
	g_return_val_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props), FALSE);

#define GW(name) glade_xml_get_widget (props->priv->xml, name)

	/* get the main widget */
	props->priv->main_widget = GW ("connection-properties");
	if (!props->priv->main_widget)
		return FALSE;

	gtk_widget_ref (props->priv->main_widget);
	gtk_widget_unparent (props->priv->main_widget);

	/* add loaded UI to the GnomeDbConnectionProperties widget */
	gtk_box_pack_start (GTK_BOX (props), props->priv->main_widget, TRUE, TRUE, 0);

#undef GW

	return (props->priv->main_widget != NULL);
}

static void
refresh_widget (GnomeDbConnectionProperties *props)
{
	g_return_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props));
}

/*
 * GnomeDbConnectionProperties class implementation
 */

static void
gnome_db_connection_properties_class_init (GnomeDbConnectionPropertiesClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_connection_properties_finalize;
}

static void
gnome_db_connection_properties_init (GnomeDbConnectionProperties *props,
				     GnomeDbConnectionPropertiesClass *klass)
{
	GtkWidget *table;

	g_return_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props));

	/* allocate internal structure */
	props->priv = g_new0 (GnomeDbConnectionPropertiesPrivate, 1);
	props->priv->cnc = NULL;
	props->priv->xml = NULL;
}

static void
gnome_db_connection_properties_finalize (GObject *object)
{
	GnomeDbConnectionProperties *props = (GnomeDbConnectionProperties *) object;

	g_return_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props));

	/* free memory */
	if (GDA_IS_CONNECTION (props->priv->cnc)) {
		g_object_unref (G_OBJECT (props->priv->cnc));
		props->priv->cnc = NULL;
	}

	if (GLADE_IS_XML (props->priv->xml)) {
		g_object_unref (props->priv->xml);
		props->priv->xml = NULL;
	}

	g_free (props->priv);
	props->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_connection_properties_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbConnectionPropertiesClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_connection_properties_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbConnectionProperties),
			0,
			(GInstanceInitFunc) gnome_db_connection_properties_init
		};
		type = g_type_register_static (PARENT_TYPE,
					       "GnomeDbConnectionProperties",
					       &info, 0);
	}
	return type;
}

/**
 * gnome_db_connection_properties_new
 * @cnc: a #GdaConnection object.
 *
 * Create a new #GnomeDbConnectionProperties widget.
 */
GtkWidget *
gnome_db_connection_properties_new (GdaConnection *cnc)
{
	GnomeDbConnectionProperties *props;

	props = g_object_new (GNOME_DB_TYPE_CONNECTION_PROPERTIES, NULL);

	/* load widgets from Glade file */
	props->priv->xml = glade_xml_new (GNOME_DB_GLADEDIR "/connection-properties.glade",
					  NULL, NULL);
	if (!props->priv->xml) {
		gda_log_error (_("Could not load the Glade XML file"));
		g_object_unref (G_OBJECT (props));
		return NULL;
	}

	if (!get_widgets (props)) {
		gda_log_error (_("Could not find all widgets in Glade XML file"));
		g_object_unref (G_OBJECT (props));
		return NULL;
	}

	/* refresh the widget if we've got a valid connection */
	if (GDA_IS_CONNECTION (cnc)) {
		g_object_ref (G_OBJECT (cnc));
		props->priv->cnc = cnc;
		refresh_widget (props);
	}

	return GTK_WIDGET (props);
}

/**
 * gnome_db_connection_properties_get_connection
 */
GdaConnection *
gnome_db_connection_properties_get_connection (GnomeDbConnectionProperties *props)
{
	g_return_val_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props), NULL);
	return props->priv->cnc;
}

/**
 * gnome_db_connection_properties_set_connection
 */
void
gnome_db_connection_properties_set_connection (GnomeDbConnectionProperties *props,
					       GdaConnection *cnc)
{
	g_return_if_fail (GNOME_DB_IS_CONNECTION_PROPERTIES (props));

	if (GDA_IS_CONNECTION (props->priv->cnc))
		g_object_unref (G_OBJECT (props->priv->cnc));

	props->priv->cnc = cnc;
	if (GDA_IS_CONNECTION (props->priv->cnc)) {
		g_object_ref (G_OBJECT (props->priv->cnc));
		refresh_widget (props);
	}
}
