/* GNOME-DB Components
 * Copyrigth (C) 2000-2001 The Free Software Foundation
 *
 * AUTHORS:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtkdialog.h>
#include <gtk/gtklabel.h>
#include <gtk/gtknotebook.h>
#include <bonobo/bonobo-i18n.h>
#include "gnome-db-connection-properties.h"
#include "components-connection.h"
#include "components-sql.h"
#include "components-sql-view.h"

#define PARENT_TYPE GNOME_DB_TYPE_SHELL_COMPONENT

static void components_sql_class_init (ComponentsSqlClass *klass);
static void components_sql_init       (ComponentsSql *comp, ComponentsSqlClass *klass);
static void components_sql_finalize   (GObject *object);
static gboolean components_sql_select_connection (GnomeDbShellComponent *comp,
						  const gchar *name,
						  const gchar *username,
						  const gchar *password);

static void verb_DatabaseConnect (BonoboUIComponent *uic, void *user_data, const char *path);
static void verb_DatabaseDisconnect (BonoboUIComponent *uic, void *user_data, const char *path);
static void verb_DatabaseOpenCommand (BonoboUIComponent *uic, void *user_data, const char *path);
static void verb_DatabaseProperties (BonoboUIComponent *uic, void *user_data, const char *path);
static void verb_DatabaseRun (BonoboUIComponent *uic, void *user_data, const char *path);
static void verb_EditCopy (BonoboUIComponent *uic, void *user_data, const char *path);
static void verb_EditCut (BonoboUIComponent *uic, void *user_data, const char *path);
static void verb_EditPaste (BonoboUIComponent *uic, void *user_data, const char *path);

static BonoboUIVerb sql_verbs[] = {
	BONOBO_UI_VERB ("EditCopy", verb_EditCopy),
	BONOBO_UI_VERB ("EditCut", verb_EditCut),
	BONOBO_UI_VERB ("EditPaste", verb_EditPaste),
	BONOBO_UI_VERB ("DatabaseConnect", verb_DatabaseConnect),
	BONOBO_UI_VERB ("DatabaseDisconnect", verb_DatabaseDisconnect),
	BONOBO_UI_VERB ("DatabaseOpenCommand", verb_DatabaseOpenCommand),
	BONOBO_UI_VERB ("DatabaseProperties", verb_DatabaseProperties),
	BONOBO_UI_VERB ("DatabaseRun", verb_DatabaseRun),
	BONOBO_UI_VERB_END
};
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
verb_DatabaseConnect (BonoboUIComponent *uic, void *user_data, const char *path)
{
	GdaConnection *cnc;
	gchar *str;
	gchar *dsn_name;
	GtkWidget *sql_view;
	ComponentsSql *sql = (ComponentsSql *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	cnc = components_connection_open_dialog (&dsn_name);
	if (!GDA_IS_CONNECTION (cnc))
		return;

	str = g_strdup_printf ("%s@%s", dsn_name, gda_connection_get_username (cnc));
	sql_view = components_sql_view_new (cnc);
	gtk_widget_show (sql_view);

	sql->connections = g_list_append (sql->connections, sql_view);
	gtk_notebook_append_page (GTK_NOTEBOOK (sql->notebook),
				  sql_view,
				  gtk_label_new (str));

	gtk_notebook_set_current_page (GTK_NOTEBOOK (sql->notebook),
				       gtk_notebook_page_num (GTK_NOTEBOOK (sql->notebook),
							      sql_view));

	/* free memory */
	g_free (str);
	g_free (dsn_name);
	g_object_unref (G_OBJECT (cnc)); /* needed, since ComponentsSqlView keeps a ref */
}

static void
verb_DatabaseDisconnect (BonoboUIComponent *uic, void *user_data, const char *path)
{
	gint current;
	GtkWidget *sql_view;
	ComponentsSql *sql = (ComponentsSql *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (sql->notebook));
	sql_view = gtk_notebook_get_nth_page (GTK_NOTEBOOK (sql->notebook), current);

	if (COMPONENTS_IS_SQL_VIEW (sql_view)) {
		components_connection_close (
			components_sql_view_get_connection (COMPONENTS_SQL_VIEW (sql_view)));
		gtk_notebook_remove_page (GTK_NOTEBOOK (sql->notebook), current);
	}
}

static void
verb_DatabaseOpenCommand (BonoboUIComponent *uic, void *user_data, const char *path)
{
	gint current;
	GtkWidget *sql_view;
	ComponentsSql *sql = (ComponentsSql *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (sql->notebook));
	sql_view = gtk_notebook_get_nth_page (GTK_NOTEBOOK (sql->notebook), current);

	if (COMPONENTS_IS_SQL_VIEW (sql_view))
		components_sql_view_open_command (sql_view);
}

static void
verb_DatabaseProperties (BonoboUIComponent *uic, void *user_data, const char *path)
{
	gint current;
	GtkWidget *sql_view;
	ComponentsSql *sql = (ComponentsSql *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (sql->notebook));
	sql_view = gtk_notebook_get_nth_page (GTK_NOTEBOOK (sql->notebook), current);

	if (COMPONENTS_IS_SQL_VIEW (sql_view)) {
		GtkWidget *dialog;
		GtkWidget *props;

		dialog = gtk_dialog_new ();
		gtk_window_set_title (GTK_WINDOW (dialog), _("Connection properties"));
		gtk_dialog_add_button (GTK_DIALOG (dialog), _("Close"), GTK_RESPONSE_CLOSE);

		props = gnome_db_connection_properties_new (
			components_sql_view_get_connection (COMPONENTS_SQL_VIEW (sql_view)));
		gtk_widget_show (props);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), props, TRUE, TRUE, 0);

		gtk_dialog_run (GTK_DIALOG (dialog));
	}
}

static void
verb_DatabaseRun (BonoboUIComponent *uic, void *user_data, const char *path)
{
	gint current;
	GtkWidget *sql_view;
	ComponentsSql *sql = (ComponentsSql *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (sql->notebook));
	sql_view = gtk_notebook_get_nth_page (GTK_NOTEBOOK (sql->notebook), current);

	if (COMPONENTS_IS_SQL_VIEW (sql_view))
		components_sql_view_run_current (COMPONENTS_SQL_VIEW (sql_view));
}

static void
verb_EditCopy (BonoboUIComponent *uic, void *user_data, const char *path)
{
	gint current;
	GtkWidget *sql_view;
	ComponentsSql *sql = (ComponentsSql *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (sql->notebook));
	sql_view = gtk_notebook_get_nth_page (GTK_NOTEBOOK (sql->notebook), current);

	if (COMPONENTS_IS_SQL_VIEW (sql_view)) {
		components_sql_view_copy_clipboard (
			COMPONENTS_SQL_VIEW (sql_view));
	}
}

static void
verb_EditCut (BonoboUIComponent *uic, void *user_data, const char *path)
{
	gint current;
	GtkWidget *sql_view;
	ComponentsSql *sql = (ComponentsSql *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (sql->notebook));
	sql_view = gtk_notebook_get_nth_page (GTK_NOTEBOOK (sql->notebook), current);

	if (COMPONENTS_IS_SQL_VIEW (sql_view)) {
		components_sql_view_cut_clipboard (
			COMPONENTS_SQL_VIEW (sql_view));
	}
}

static void
verb_EditPaste (BonoboUIComponent *uic, void *user_data, const char *path)
{
	gint current;
	GtkWidget *sql_view;
	ComponentsSql *sql = (ComponentsSql *) user_data;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (sql->notebook));
	sql_view = gtk_notebook_get_nth_page (GTK_NOTEBOOK (sql->notebook), current);

	if (COMPONENTS_IS_SQL_VIEW (sql_view)) {
		components_sql_view_paste_clipboard (
			COMPONENTS_SQL_VIEW (sql_view));
	}
}

/*
 * ComponentsSql class implementation
 */

static void
components_sql_class_init (ComponentsSqlClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GnomeDbShellComponentClass *comp_class = GNOME_DB_SHELL_COMPONENT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = components_sql_finalize;
	comp_class->select_connection = components_sql_select_connection;
}

static void
components_sql_init (ComponentsSql *comp, ComponentsSqlClass *klass)
{
	GnomeDbShellComponent *shell_comp;
	GnomeDbControl *control;

	/* create the notebook widget, where all open connections will be added */
	comp->notebook = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (comp->notebook), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (comp->notebook), FALSE);
	comp->connections = NULL;

	/* create the shell component */
	shell_comp = gnome_db_shell_component_construct (
		GNOME_DB_SHELL_COMPONENT (comp),
		comp->notebook);
	if (!GNOME_DB_IS_SHELL_COMPONENT (shell_comp)) {
		g_object_unref (G_OBJECT (comp));
		return;
	}

	control = gnome_db_shell_component_get_control (shell_comp);

	g_assert (GNOME_DB_IS_CONTROL (control));

	gnome_db_control_set_ui (control,
				 GNOMEDB_DATADIR,
				 "gnome-db-sql.xml",
				 sql_verbs,
				 shell_comp);
}

static void
components_sql_finalize (GObject *object)
{
	ComponentsSql *sql = (ComponentsSql *) object;

	g_return_if_fail (COMPONENTS_IS_SQL (sql));

	/* free memory */
	g_list_free (sql->connections);
	sql->connections = NULL;

	parent_class->finalize (object);
}

GType
components_sql_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (ComponentsSqlClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) components_sql_class_init,
			NULL,
			NULL,
			sizeof (ComponentsSql),
			0,
			(GInstanceInitFunc) components_sql_init
		};
		type = g_type_register_static (PARENT_TYPE, "ComponentsSql", &info, 0);
	}
	return type;
}

GnomeDbShellComponent *
components_sql_new (void)
{
	GnomeDbShellComponent *comp;

	comp = g_object_new (COMPONENTS_TYPE_SQL, NULL);
	return comp;
}

static gboolean
components_sql_select_connection (GnomeDbShellComponent *comp,
				  const gchar *name,
				  const gchar *username,
				  const gchar *password)
{
	GdaConnection *cnc;
	GList *l;
	ComponentsSql *sql = (ComponentsSql *) comp;

	g_return_val_if_fail (COMPONENTS_IS_SQL (sql), FALSE);

	cnc = components_connection_open (name, username, password, TRUE, TRUE);
	if (!GDA_IS_CONNECTION (cnc))
		return FALSE;

	/* activate the selected connection view */
	for (l = sql->connections; l; l = l->next) {
		ComponentsSqlView *sql_view = (ComponentsSqlView *) l->data;

		g_assert (COMPONENTS_IS_SQL_VIEW (sql_view));

		if (cnc == components_sql_view_get_connection (sql_view)) {
			gtk_notebook_set_current_page (
				GTK_NOTEBOOK (sql->notebook),
				gtk_notebook_page_num (GTK_NOTEBOOK (sql->notebook),
						       GTK_WIDGET (sql_view)));
			break;
		}
	}

	return TRUE;
}
