/* GNOME DB library
 * Copyright (C) 1999-2001 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-dsn-config-druid.h"
#include "gnome-db-util.h"
#include <gdk_imlib.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-druid-page-start.h>
#include <libgnomeui/gnome-druid-page-standard.h>
#include <libgnomeui/gnome-druid-page-finish.h>

struct _GnomeDbDsnConfigDruidPrivate {
	GdaDsn *dsn;
	GnomeDruidPageStart *start_page;
	GnomeDruidPageStandard *general_page;
	GnomeDruidPageStandard *provider_page;
	GnomeDruidPageFinish *finish_page;

	GtkWidget *general_gdaname;
	GtkWidget *general_provider;
	GtkWidget *general_description;
	GtkWidget *general_username;

	GtkWidget *provider_container;
	GList *provider_params;
	GList *provider_entries;
};

enum {
	FINISH,
	LAST_SIGNAL
};

static gint config_druid_signals[LAST_SIGNAL] = { 0, };

/*
 * Private functions
 */
static void
free_param_in_list (gpointer data, gpointer user_data)
{
	gchar *str;

	str = (gchar *) data;
	if (str)
		g_free((gpointer) str);
}

/*
 * Callbacks
 */
static void
cancel_druid_cb (GnomeDruid *gnome_druid, gpointer user_data)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_if_fail(GNOME_DB_IS_DSN_CONFIG_DRUID(druid));

	if (druid->priv->dsn) {
		gda_dsn_free(druid->priv->dsn);
		druid->priv->dsn = NULL;
	}
	gtk_signal_emit(GTK_OBJECT(druid),
					config_druid_signals[FINISH],
					TRUE);
}

static void
druid_finished_cb (GnomeDruidPage *druid_page, gpointer arg1, gpointer user_data)
{
	GdaDsn *dsn;
	GString *dsn_str = NULL;
	GList *l;
	gint n;
	GtkWidget *menu_item;
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_if_fail(GNOME_DB_IS_DSN_CONFIG_DRUID(druid));

	dsn = gda_dsn_new();
	gda_dsn_set_name(dsn, gtk_entry_get_text(GTK_ENTRY(druid->priv->general_gdaname)));
	gda_dsn_set_description(dsn, gtk_entry_get_text(GTK_ENTRY(druid->priv->general_description)));
	gda_dsn_set_username(dsn, gtk_entry_get_text(GTK_ENTRY(druid->priv->general_username)));

	menu_item = GTK_OPTION_MENU(druid->priv->general_provider)->menu_item;
	if (GTK_IS_MENU_ITEM(menu_item)) {
		gchar *str = NULL;

		str = (gchar *) gtk_object_get_data(GTK_OBJECT(menu_item), "GNOMEDB_MenuItemLabel");
		gda_dsn_set_provider(dsn, str);
	}
	else
		gda_dsn_set_provider(dsn, NULL);

	/* build the DSN string from the list of paramaters */
	for (l = g_list_first(druid->priv->provider_params), n = 0; l != NULL; l = g_list_next(l), n++) {
		GtkWidget *w;
		GList *entry;

		entry = g_list_nth(druid->priv->provider_entries, n);
		w = entry ? GTK_WIDGET(entry->data) : NULL;
		if (GTK_IS_ENTRY(w)) {
			gchar *text = gtk_entry_get_text(GTK_ENTRY(w));

			if (text != NULL && strlen(text) >= 1) {
				if (!dsn_str) 
					dsn_str = g_string_new("");
				else
					dsn_str = g_string_append(dsn_str, ";");
				dsn_str = g_string_append(dsn_str, (gchar *) l->data);
				dsn_str = g_string_append(dsn_str, "=");
				dsn_str = g_string_append(dsn_str, text);
			}
		}
	}
	if (dsn_str) {
		gda_dsn_set_dsn(dsn, dsn_str->str);
		g_string_free(dsn_str, TRUE);
	}
	else
		gda_dsn_set_dsn(dsn, NULL);

	if (druid->priv->dsn)
		gda_dsn_free(druid->priv->dsn);
	druid->priv->dsn = dsn;

	gtk_signal_emit(GTK_OBJECT(druid),
					config_druid_signals[FINISH],
					FALSE);
}

static gboolean
general_page_next_cb (GnomeDruidPage *druid_page, gpointer arg1, gpointer user_data)
{
	GtkWidget *menu_item;
	gchar *text;
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) user_data;

	g_return_val_if_fail(GNOME_DB_IS_DSN_CONFIG_DRUID(druid), TRUE);

	/* check that the user entered a name */
	text = gtk_entry_get_text(GTK_ENTRY(druid->priv->general_gdaname));
	if (!text || strlen(text) <= 0) {
		gnome_db_show_error(_("You must enter a name for the new data source"));
		gtk_widget_grab_focus(druid->priv->general_gdaname);
		return TRUE;
	}
	
	/* free previous parameters, if any */
	g_list_foreach(druid->priv->provider_params, (GFunc) free_param_in_list, NULL);
	g_list_free(druid->priv->provider_params);
	druid->priv->provider_params = NULL;

	g_list_free(druid->priv->provider_entries);
	druid->priv->provider_entries = NULL;

	if (GTK_IS_WIDGET(druid->priv->provider_container))
		gtk_widget_destroy(druid->priv->provider_container);
	druid->priv->provider_container = NULL;

	/* build the entries for the selected parameter */
	menu_item = GTK_OPTION_MENU(druid->priv->general_provider)->menu_item;
	if (GTK_IS_MENU_ITEM(menu_item)) {
		gchar *str = NULL;
		GdaProvider *provider;

		str = (gchar *) gtk_object_get_data(GTK_OBJECT(menu_item), "GNOMEDB_MenuItemLabel");
		provider = gda_provider_find_by_name(str);
		if (provider) {
			gint n;
			GList *l;
			GtkWidget *label;
			GtkWidget *entry;
			GList *dsn_params = GDA_PROVIDER_DSN_PARAMS(provider);

			/* create the entry container */
			if (dsn_params) {
				druid->priv->provider_container = gnome_db_new_table_widget(g_list_length(dsn_params), 2, FALSE);
				gtk_box_pack_start(GTK_BOX(GNOME_DRUID_PAGE_STANDARD(druid->priv->provider_page)->vbox),
								   druid->priv->provider_container, 1, 1, 0);

				/* add all parameters to our internal list */
				for (l = g_list_first(dsn_params), n = 0; l != NULL; l = g_list_next(l), n++) {
					druid->priv->provider_params = g_list_append(druid->priv->provider_params,
																 g_strdup((gchar *) l->data));

					label = gnome_db_new_label_widget((gchar *) l->data);
					gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
					gtk_table_attach(GTK_TABLE(druid->priv->provider_container), label,
									 0, 1, n, n + 1,
									 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND,
									 3, 3);
					entry = gnome_db_new_entry_widget(0, TRUE);
					gtk_table_attach(GTK_TABLE(druid->priv->provider_container), entry,
									 1, 2, n, n + 1,
									 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND,
									 3, 3);
					druid->priv->provider_entries = g_list_append(druid->priv->provider_entries, entry);
				}
			}
			else gnome_db_show_error("no parameters for provider %s", str);
			gda_provider_free(provider);
		}
	}

	return FALSE;
}

/*
 * GnomeDbDsnConfigDruid interface
 */
static void
gnome_db_dsn_config_druid_class_init (GnomeDbDsnConfigDruidClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	config_druid_signals[FINISH] =
		gtk_signal_new("finish",
					   GTK_RUN_LAST,
					   object_class->type,
					   GTK_SIGNAL_OFFSET(GnomeDbDsnConfigDruidClass, finish),
					   gtk_marshal_NONE__INT,
					   GTK_TYPE_NONE, 1, GTK_TYPE_INT);
	gtk_object_class_add_signals(object_class, config_druid_signals, LAST_SIGNAL);

	klass->finish = NULL;
}

static void
gnome_db_dsn_config_druid_destroy (GtkObject *object, gpointer user_data)
{
	GnomeDbDsnConfigDruid *druid = (GnomeDbDsnConfigDruid *) object;

	g_return_if_fail(GNOME_DB_IS_DSN_CONFIG_DRUID(druid));

	/* free memory */
	if (druid->priv) {
		if (druid->priv->dsn)
			gda_dsn_free(druid->priv->dsn);

		/* free parameter list */
		g_list_foreach(druid->priv->provider_params, (GFunc) free_param_in_list, NULL);
		g_list_free(druid->priv->provider_params);
		druid->priv->provider_params = NULL;

		/* free entry list */
		g_list_free(druid->priv->provider_entries);
		druid->priv->provider_entries = NULL;
		
		g_free((gpointer) druid->priv);
	}
}

static void
gnome_db_dsn_config_druid_init (GnomeDbDsnConfigDruid *druid)
{
	GdkImlibImage *logo = NULL;
	gchar *pathname;
	GtkWidget *table;
	GtkWidget *label;
	GList *providerlist;
	GtkWidget *provider_menu;
	GList *l;

	/* allocate private structure */
	druid->priv = g_new0(GnomeDbDsnConfigDruidPrivate, 1);

	pathname = gnome_pixmap_file("gnome-db.png");
	if (pathname) {
		logo = gdk_imlib_load_image(pathname);
		g_free((gpointer) pathname);
	}

	/* create the start page */
	druid->priv->start_page = gnome_druid_page_start_new();
	gnome_druid_page_start_set_logo(GNOME_DRUID_PAGE_START(druid->priv->start_page), logo);
	gnome_druid_page_start_set_title(GNOME_DRUID_PAGE_START(druid->priv->start_page),
									 _("Add a new data source..."));
	gnome_druid_page_start_set_text(GNOME_DRUID_PAGE_START(druid->priv->start_page),
									_("This wizard will guide you through the process of\n"
									  "creating a new data source for your GNOME-DB\n"
									  "installation. Just follow the steps!"));
	gtk_widget_show_all(GTK_WIDGET(druid->priv->start_page));

	/* create the general info page */
	druid->priv->general_page = gnome_druid_page_standard_new_with_vals(_("General Information"),
																		logo);
	table = gnome_db_new_table_widget(5, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(GNOME_DRUID_PAGE_STANDARD(druid->priv->general_page)->vbox), table, 1, 1, 0);

	label = gnome_db_new_label_widget(_("The following fields represent the basic information\n"
										"items for your new data source. Some of them (description,\n"
										"username) are optional, whereas the others are mandatory.\n"
										"If you don't know which provider to use, just select\n"
										"the default one"));
	gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
	gtk_table_attach(GTK_TABLE(table), label, 0, 2, 0, 1,
					 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
					 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
					 3, 3);

	label = gnome_db_new_label_widget(_("Name"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
					 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 3, 3);
	druid->priv->general_gdaname = gnome_db_new_entry_widget(0, TRUE);
	gnome_db_set_widget_tooltip(druid->priv->general_gdaname,
								_("Enter the name you want to use to refer to this data source"));
	gtk_table_attach(GTK_TABLE(table), druid->priv->general_gdaname, 1, 2, 1, 2,
					 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 3, 3);

	label = gnome_db_new_label_widget(_("Provider"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
					 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 3, 3);
	druid->priv->general_provider = gnome_db_new_option_menu_widget();
	gnome_db_set_widget_tooltip(druid->priv->general_provider,
								_("Select the provider for the backend you want to use "
								  "in your new data source. If you don't know which one "
								  "to use, just select the **default** provider"));
	gtk_table_attach(GTK_TABLE(table), druid->priv->general_provider, 1, 2, 2, 3,
					 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 3, 3);

	provider_menu = gtk_menu_new();
	providerlist = gda_provider_list();
	for (l = g_list_first(providerlist); l != NULL; l = g_list_next(l)) {
		GtkWidget *menu_item;
		GdaProvider *prov;

		prov = (GdaProvider *) l->data;
		if (prov) {
			gchar *tmp_str;

			tmp_str = g_strdup(GDA_PROVIDER_NAME(prov));
			menu_item = gtk_menu_item_new_with_label(tmp_str);
			gtk_object_set_data_full(GTK_OBJECT(menu_item),
									 "GNOMEDB_MenuItemLabel",
									 (gpointer) tmp_str,
									 (GtkDestroyNotify) g_free);
			gtk_menu_append(GTK_MENU(provider_menu), menu_item);
		}
	}
	gda_provider_free_list(providerlist);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(druid->priv->general_provider), provider_menu);

	label = gnome_db_new_label_widget(_("Description"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4,
					 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 3, 3);
	druid->priv->general_description = gnome_db_new_entry_widget(0, TRUE);
	gnome_db_set_widget_tooltip(druid->priv->general_description,
								_("Enter a meaningful description of this data source (Optional)"));
	gtk_table_attach(GTK_TABLE(table), druid->priv->general_description, 1, 2, 3, 4,
					 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 3, 3);

	label = gnome_db_new_label_widget(_("Username"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5,
					 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 3, 3);
	druid->priv->general_username = gnome_db_new_entry_widget(0, TRUE);
	gnome_db_set_widget_tooltip(druid->priv->general_username,
								_("Enter the username you'll be using for connecting to "
								  "this data source (Optional)"));
	gtk_table_attach(GTK_TABLE(table), druid->priv->general_username, 1, 2, 4, 5,
					 GTK_FILL, GTK_FILL, 3, 3);

	gtk_widget_show_all(GTK_WIDGET(druid->priv->general_page));

	/* create the provider-specific page */
	druid->priv->provider_page = gnome_druid_page_standard_new_with_vals(_("Connection String"),
																		 logo);
	gtk_widget_show_all(GTK_WIDGET(druid->priv->provider_page));

	/* create the finish page */
	druid->priv->finish_page = gnome_druid_page_finish_new();
	gnome_druid_page_finish_set_logo(GNOME_DRUID_PAGE_FINISH(druid->priv->finish_page), logo);
	gnome_druid_page_finish_set_title(GNOME_DRUID_PAGE_FINISH(druid->priv->finish_page),
									  _("All information retrieved"));
	gnome_druid_page_finish_set_text(GNOME_DRUID_PAGE_FINISH(druid->priv->finish_page),
									 _("All information needed to create a new data source \n"
										 "has been retrieved. Now, press 'Finish' to close \n"
										 "this dialog. To open your newly created data source, \n"
										 "click on the 'Database' Icon on the left bar, and \n"
										 "then the 'Connect' button, which will open a \n"
										 "dialog box to let you open a new connection"));
	gtk_widget_show_all(GTK_WIDGET(druid->priv->finish_page));

	/* append all pages to the druid */
	gnome_druid_append_page(GNOME_DRUID(druid), GNOME_DRUID_PAGE(druid->priv->start_page));
	gnome_druid_append_page(GNOME_DRUID(druid), GNOME_DRUID_PAGE(druid->priv->general_page));
	gnome_druid_append_page(GNOME_DRUID(druid), GNOME_DRUID_PAGE(druid->priv->provider_page));
	gnome_druid_append_page(GNOME_DRUID(druid), GNOME_DRUID_PAGE(druid->priv->finish_page));
	gnome_druid_set_page(GNOME_DRUID(druid), GNOME_DRUID_PAGE(druid->priv->start_page));

	/* connect to signals */
	gtk_signal_connect(GTK_OBJECT(druid),
					   "cancel",
					   GTK_SIGNAL_FUNC(cancel_druid_cb),
					   druid);
	gtk_signal_connect(GTK_OBJECT(druid),
					   "destroy",
					   GTK_SIGNAL_FUNC(gnome_db_dsn_config_druid_destroy),
					   NULL);
	gtk_signal_connect(GTK_OBJECT(druid->priv->general_page),
					   "next",
					   GTK_SIGNAL_FUNC(general_page_next_cb),
					   (gpointer) druid);
	gtk_signal_connect(GTK_OBJECT(druid->priv->finish_page),
					   "finish",
					   GTK_SIGNAL_FUNC(druid_finished_cb),
					   (gpointer) druid);
}

GtkType
gnome_db_dsn_config_druid_get_type (void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"GnomeDbDsnConfigDruid",
			sizeof(GnomeDbDsnConfigDruid),
			sizeof(GnomeDbDsnConfigDruidClass),
			(GtkClassInitFunc) gnome_db_dsn_config_druid_class_init,
			(GtkObjectInitFunc) gnome_db_dsn_config_druid_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		type = gtk_type_unique(gnome_druid_get_type(), &info);
	}
	return type;
}

/**
 * gnome_db_dsn_config_druid_new
 *
 * Create a new #GnomeDbDsnConfigDruid widget, which you can insert in any
 * GTK container-like widget. Usually, you'll want to add it to a dialog box.
 * Once you create it, the druid runs until the user completes all the
 * steps in the process or cancels the operation. In either case, once
 * it's finished, the #GnomeDbDsnConfigDruid widget emits its
 * "finish" signal, at which time you can get the resulting GDA
 * data source set up by the user by calling
 * #gnome_db_dsn_config_druid_get_dsn
 */
GtkWidget *
gnome_db_dsn_config_druid_new (void)
{
	GnomeDbDsnConfigDruid *druid;

	druid = GNOME_DB_DSN_CONFIG_DRUID(gtk_type_new(GNOME_DB_TYPE_DSN_CONFIG_DRUID));
	return GTK_WIDGET(druid);
}

/**
 * gnome_db_dsn_config_druid_get_dsn
 * @druid: a #GnomeDbDsnConfigDruid widget
 *
 * Returns the GDA data source resulted from the interaction between the
 * #GnomeDbDsnConfigDruid widget and the user
 */
GdaDsn *
gnome_db_dsn_config_druid_get_dsn (GnomeDbDsnConfigDruid *druid)
{
	g_return_val_if_fail(GNOME_DB_IS_DSN_CONFIG_DRUID(druid), NULL);
	return druid->priv->dsn;
}
