/* GNOME DB library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <liboaf/liboaf.h>
#include <gda-config.h>
#include <gda-corba.h>
#include <gnome-db-config.h>

/**
 * gnome_db_config_get_saved_sql_queries
 *
 * Returns the names of all the saved SQL queries for the current user.
 * To free the returned GList, use gda_config_free_list
 *
 * Returns: a GList containing the names of all saved SQL queries.
 */
GList *
gnome_db_config_get_saved_sql_queries (void)
{
	GList* keys;

	keys = gda_config_list_keys(GNOME_DB_CONFIG_SECTION_SAVED_SQL_QUERIES);
	return keys;
}

/**
 * gnome_db_config_get_saved_xml_queries
 *
 * Returns the names of all the saved XML queries for the current user.
 * To free the returned GList, use gda_config_free_list
 *
 * Returns: a GList containing the names of all saved XML queries.
 */
GList *
gnome_db_config_get_saved_xml_queries (void)
{
	GList* keys;

	keys = gda_config_list_keys(GNOME_DB_CONFIG_SECTION_SAVED_XML_QUERIES);
	return keys;
}

/**
 * gnome_db_config_component_new
 *
 * Creates a new #GnomeDbConfigComponent structure. This structure is used
 * to represent the information stored in the configuration database about
 * a component.
 *
 * You would almost never want to use this routine directly, since the
 * #GnomeDbConfigComponent are created internally in the GNOME-DB library,
 * and just returned to the programmer.
 *
 * Returns: the newly-created structure
 */
GnomeDbConfigComponent *
gnome_db_config_component_new (void)
{
	return g_new0(GnomeDbConfigComponent, 1);
}

/**
 * gnome_db_config_component_free
 * @components: a #GnomeDbConfigComponent structure
 *
 * Frees all memory used by the given #GnomeDbConfigComponent structure.
 */
void
gnome_db_config_component_free (GnomeDbConfigComponent *component)
{
	g_return_if_fail(component != NULL);
  
	if (component->name) g_free((gpointer) component->name);
	if (component->location) g_free((gpointer) component->location);
	if (component->description) g_free((gpointer) component->description);
	if (component->repo_ids) g_free((gpointer) component->repo_ids);
	if (component->type) g_free((gpointer) component->type);
  
	g_free((gpointer) component);
}

/**
 * gnome_db_config_component_set_name
 * @component: a #GnomeDbConfigComponent structure
 * name: the new name for the component
 *
 * Sets the 'name' field in the given #GnomeDbConfigComponent structure
 */
void
gnome_db_config_component_set_name (GnomeDbConfigComponent *component, const gchar *name)
{
	g_return_if_fail(component != NULL);
	g_return_if_fail(name != NULL);
  
	if (component->name) g_free((gpointer) component->name);
	component->name = g_strdup(name);
}

/**
 * gnome_db_config_component_set_location
 * @component: a #GnomeDbConfigComponent structure
 * @location: new location for the component
 *
 * Sets the 'location' field in the given #GnomeDbConfigComponent structure
 */
void
gnome_db_config_component_set_location (GnomeDbConfigComponent *component, const gchar *location)
{
	g_return_if_fail(component != NULL);
	g_return_if_fail(location != NULL);
  
	if (component->location) g_free((gpointer) component->location);
	component->location = g_strdup(location);
}

/**
 * gnome_db_config_component_set_description
 * @component: a #GnomeDbConfigComponent structure
 * @description: new description for the component
 *
 * Sets the 'description' field in the given #GnomeDbConfigComponent structure
 */
void
gnome_db_config_component_set_description (GnomeDbConfigComponent *component, const gchar *desc)
{
	g_return_if_fail(component != NULL);
	g_return_if_fail(desc != NULL);
  
	if (component->description) g_free((gpointer) component->description);
	component->description = g_strdup(desc);
}

/**
 * gnome_db_config_component_set_repo_ids
 * @component: a #GnomeDbConfigComponent structure
 * @repo_ids: new repository IDs for the component
 *
 * Sets the 'repo_ids' field in the given #GnomeDbConfigComponent structure
 */
void
gnome_db_config_component_set_repo_ids (GnomeDbConfigComponent *component, const gchar *repo_ids)
{
	g_return_if_fail(component != NULL);
	g_return_if_fail(repo_ids != NULL);
  
	if (component->repo_ids) g_free((gpointer) component->repo_ids);
	component->repo_ids = g_strdup(repo_ids);
}

/**
 * gnome_db_config_component_set_type
 * @component: a #GnomeDbConfigComponent structure
 * @type: new type for the component
 *
 * Sets the 'type' field in the given #GnomeDbConfigComponent structure
 */
void
gnome_db_config_component_set_type (GnomeDbConfigComponent *component, const gchar *type)
{
	g_return_if_fail(component != NULL);
	g_return_if_fail(type != NULL);

	if (component->type) g_free((gpointer) component->type);
	component->type = g_strdup(type);
}

/**
 * gnome_db_config_component_set_icon
 * @component: a #GnomeDbConfigComponent structure
 * @icon: new icon for the component
 *
 * Sets the 'icon' field in the given #GnomeDbConfigComponent structure
 */
void
gnome_db_config_component_set_icon (GnomeDbConfigComponent *component, const gchar *icon)
{
	g_return_if_fail(component != NULL);
	g_return_if_fail(icon != NULL);
  
	if (component->icon) g_free((gpointer) component->icon);
	component->icon = g_strdup(icon);
}

/**
 * gnome_db_config_get_shell_components
 *
 * Return a list of all registered components that can be used in the
 * gnome-db frontend. It's up to the programmer to free the returned list,
 * which can be easily done by calling #gnome_db_config_component_free_list.
 *
 * Returns: a GList of #GnomeDbConfigComponent structures.
 */
GList *
gnome_db_config_get_shell_components (void)
{
	OAF_ServerInfoList* servlist;
	CORBA_Environment   ev;
	GList*              ret = NULL;
	gint                i;
  
	CORBA_exception_init(&ev);
	servlist = oaf_query("repo_ids.has('IDL:GNOME/Database/UIShellComponent:1.0')", NULL, &ev);
	if (servlist) {
		for (i = 0; i < servlist->_length; i++) {
			GnomeDbConfigComponent* component = gnome_db_config_component_new();
			gnome_db_config_component_set_name(component, servlist->_buffer[i].iid);
			gnome_db_config_component_set_location(component, servlist->_buffer[i].location_info);
			gnome_db_config_component_set_description (
				component,
				gda_corba_get_oaf_attribute(servlist->_buffer[i].props, "description"));
			gnome_db_config_component_set_repo_ids (
				component,
				gda_corba_get_oaf_attribute(servlist->_buffer[i].props, "repo_ids"));
			gnome_db_config_component_set_type(component, servlist->_buffer[i].server_type);
			gnome_db_config_component_set_icon(
				component,
				gda_corba_get_oaf_attribute(servlist->_buffer[i].props, "icon"));
          
			ret = g_list_append(ret, (gpointer) component);
		}

		CORBA_free (servlist);
	}
	CORBA_exception_free(&ev);
  
	return ret;
}

/**
 * gnome_db_config_component_free_list
 * @list: the list to be freed
 *
 * Frees the given list, which should be a GList of #GnomeDbConfigComponent
 * structures, as returned by #gnome_db_config_get_shell_components or
 * #gnome_db_config_get_components.
 */
void
gnome_db_config_component_free_list (GList *list)
{
	GList* node;
  
	while ((node = g_list_first(list))) {
		GnomeDbConfigComponent* component = (GnomeDbConfigComponent *) node->data;
		list = g_list_remove(list, node->data);
		gnome_db_config_component_free(component);
	}
}
