/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-config.h"
#include "gnome-db-login.h"
#include "gnome-db-util.h"
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>
#include <gda-config.h>

enum {
	LAST_SIGNAL
};

static gint   gnome_db_login_signals[LAST_SIGNAL] = {0,};

static void gnome_db_login_class_init      (GnomeDbLoginClass* klass);
static void gnome_db_login_init            (GnomeDbLogin* login);

static void
fill_last_connection_list (GtkWidget *cnc_list) {
	gint cnt;
	GList *last_connections = NULL;
	GList *node;

	g_return_if_fail (GTK_IS_CLIST (cnc_list));

	gnome_db_clear_clist (GTK_CLIST (cnc_list));

	/* read last successful connections */
	for (cnt = 1;
	     cnt <= gda_config_get_int (GDA_CONFIG_KEY_MAX_LAST_CONNECTIONS);
	     cnt++) {
		gchar *str, *data;

		str = g_strdup_printf("%s/Connection%d", GDA_CONFIG_SECTION_LAST_CONNECTIONS, cnt);
		data = gda_config_get_string(str);
		g_free((gpointer) str);
		if (data != 0)
			last_connections = g_list_append(last_connections, g_strdup (data));
        }

	/* fill in the connection list */
	node = g_list_first(last_connections);
	while (node != 0) {
		gchar* row[2];
		gchar  resource[128];
		gchar* user_name;
		gchar* str = (gchar *) node->data;

		if (str != 0) {
			g_snprintf(resource, sizeof(resource), "%s/%s/Username", GDA_CONFIG_SECTION_DATASOURCES, str);
			user_name = gda_config_get_string(resource);
			row[0] = str;
			row[1] = user_name;
			gtk_clist_append(GTK_CLIST(cnc_list), (gchar **) row);
		}
		node = g_list_next(node);
	}

	g_list_foreach (last_connections, (GFunc) g_free, NULL);
	g_list_free (last_connections);
}

static void
select_last_connection_cb (GtkCList *clist,
                           gint row,
                           gint column,
                           GdkEvent *event,
                           gpointer data)
{
	GList*        sel;
	GnomeDbLogin* login = GNOME_DB_LOGIN(data);
  
	g_return_if_fail(GTK_IS_CLIST(clist));
	g_return_if_fail(GNOME_DB_IS_LOGIN(login));
  
	sel = GTK_CLIST(clist)->selection;
	if (sel != 0) {
		gchar* str = 0;
		gchar* empty_string = "";

		gtk_clist_get_text(clist, GPOINTER_TO_UINT(sel->data), 0, &str);
		if (str) {
			gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(login->gda_dbname)->entry), str);
			str = 0;
			gtk_clist_get_text(clist, GPOINTER_TO_UINT(sel->data), 1, &str);
			if (str) {
				gtk_entry_set_text(GTK_ENTRY(login->username_entry), str);
            }
			else gtk_entry_set_text(GTK_ENTRY(login->username_entry), empty_string);
        }
    }
}

GtkType
gnome_db_login_get_type (void)
{
	static GtkType db_login_type = 0;

	if (!db_login_type) {
		GtkTypeInfo db_login_info = {
			"GnomeDbLogin",
			sizeof (GnomeDbLogin),
			sizeof(GnomeDbLoginClass),
			(GtkClassInitFunc)gnome_db_login_class_init,
			(GtkObjectInitFunc)gnome_db_login_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL,
		};
		db_login_type = gtk_type_unique(gtk_vbox_get_type(), &db_login_info);
    }
	return db_login_type;
}

static void
gnome_db_login_class_init (GnomeDbLoginClass* klass)
{
	GtkObjectClass* object_class;
	register gint   cnt;

	object_class = (GtkObjectClass*) klass;
}

static void
gnome_db_login_init (GnomeDbLogin* login)
{
	GtkWidget* label;
	GtkWidget* table;
	GtkWidget* frame;
	GtkWidget* scroll;
	GList*     node;
	gchar*     titles[] = { N_("GDA Name"), N_("User Name") };

	frame = gnome_db_new_frame_widget(NULL);
	gtk_box_pack_start(GTK_BOX(login), frame, TRUE, TRUE, 0);

	table = gnome_db_new_table_widget(3, 4, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	label = gnome_db_new_label_widget(_("GDA Datasource"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
			 GTK_FILL, 0, 3, 3);

	login->gda_dbname = gtk_combo_new();
	//gtk_combo_set_value_in_list(GTK_COMBO(login->gda_dbname), TRUE, FALSE);
	gtk_table_attach(GTK_TABLE(table), login->gda_dbname, 1, 2, 0, 1, GTK_FILL, 0, 3, 3);
	gtk_widget_show(login->gda_dbname);

	label = gnome_db_new_label_widget(_("Username"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, 0, 3, 3);

	login->username_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), login->username_entry, 1, 2, 1, 2, GTK_FILL, 0, 3, 3);
	gtk_widget_show(login->username_entry);

	label = gnome_db_new_label_widget(_("Password"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, 0, 3, 3);

	login->password_entry = gtk_entry_new();
	gtk_widget_show(login->password_entry);
	gtk_table_attach(GTK_TABLE(table), login->password_entry, 1, 2, 2, 3, GTK_FILL, 0, 3, 3);
	gtk_entry_set_visibility(GTK_ENTRY(login->password_entry), FALSE);

	frame = gnome_db_new_frame_widget(_("Last Connections"));
	gtk_table_attach(GTK_TABLE(table), frame, 0, 3, 3, 4,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 3, 3);
                   
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_container_add(GTK_CONTAINER(frame), scroll);
	login->connection_list = gnome_db_new_clist_widget(titles, sizeof(titles) / sizeof(titles[0]));
	fill_last_connection_list (login->connection_list);
	gtk_signal_connect(GTK_OBJECT(login->connection_list),
			   "select_row",
			   GTK_SIGNAL_FUNC(select_last_connection_cb),
			   (gpointer) login);
	gtk_container_add(GTK_CONTAINER(scroll), login->connection_list);
}

/**
 * gnome_db_login_new
 * @gda_dbname: GDA data source name
 *
 * Create a new GnomeDbLogin widget. This widget contains all the entry fields
 * required for connecting to a GDA data source, as well as a history list,
 * which allows the user to select from previously successful connections
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget*
gnome_db_login_new(const gchar* gda_dbname)
{
	GtkWidget*    widget;
	GnomeDbLogin* login;
	GList*        db_names;

	widget = gtk_type_new(gnome_db_login_get_type());
	login = GNOME_DB_LOGIN(widget);

	db_names = gda_list_datasources();
	if (db_names) {
		gtk_combo_set_popdown_strings(GTK_COMBO(login->gda_dbname), db_names);
		if (gda_dbname)
			gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(login->gda_dbname)->entry), gda_dbname);
		g_list_free (db_names);
	}

	return widget;
}

/**
 * gnome_db_login_get_gda_name
 *
 * Return the currently selected GDA DSN name
 */
const gchar *
gnome_db_login_get_gda_name (GnomeDbLogin *login)
{
	g_return_val_if_fail(GNOME_DB_IS_LOGIN(login), NULL);
	g_return_val_if_fail(GTK_IS_COMBO(login->gda_dbname), NULL);
	g_return_val_if_fail(GTK_IS_ENTRY(GTK_COMBO(login->gda_dbname)->entry), NULL);

	return gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(login->gda_dbname)->entry));
}

/**
 * gnome_db_login_get_username
 */
const gchar *
gnome_db_login_get_username (GnomeDbLogin *login)
{
	g_return_val_if_fail(GNOME_DB_IS_LOGIN(login), NULL);
	g_return_val_if_fail(GTK_IS_ENTRY(login->username_entry), NULL);

	return gtk_entry_get_text(GTK_ENTRY(login->username_entry));
}

/**
 * gnome_db_login_get_password
 */
const gchar *
gnome_db_login_get_password (GnomeDbLogin *login)
{
	g_return_val_if_fail(GNOME_DB_IS_LOGIN(login), NULL);
	g_return_val_if_fail(GTK_IS_ENTRY(login->password_entry), NULL);

	return gtk_entry_get_text(GTK_ENTRY(login->password_entry));
}
 
