/* GNOME DB components libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-control.h"
#include "gnome-db-control-corba.h"
#include "bonobo-ui-compat.h"

#define PARENT_TYPE BONOBO_X_OBJECT_TYPE

struct _GnomeDbControlPrivate {
	BonoboControl*     bonobo_control;
	GtkWidget*         widget;
	BonoboPropertyBag* properties;
	BonoboUIComponent* ui_component;
	BonoboUIHandler*   ui_handler;
	GnomeUIInfo*       menubar;
	GnomeUIInfo*       toolbar;
	gpointer           user_data;
};

enum {
	GET_PROPERTY,
	SET_PROPERTY,
	DO_PRINT,
	DO_CUT_CLIPBOARD,
	DO_COPY_CLIPBOARD,
	DO_PASTE_CLIPBOARD,
	LAST_SIGNAL
};

static guint control_signals[LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
static void
control_activated_cb (BonoboControl *bonobo_control,
                      gboolean activate,
                      GnomeDbControl *control)
{
	if (activate) gnome_db_control_activate(control);
	else gnome_db_control_deactivate(control);
}

static void
control_destroyed_cb (GnomeDbControl *control, gpointer data)
{
	CORBA_Environment ev;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	g_free((gpointer) control->priv);
}

static void
get_prop_cb (BonoboPropertyBag *bag, BonoboArg *arg, guint arg_id, gpointer user_data)
{
}

static void
set_prop_cb (BonoboPropertyBag *bag, const BonoboArg *arg, guint arg_id, gpointer user_data)
{
}

static void
widget_destroyed_cb (GtkWidget *w, GnomeDbControl *control)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	gtk_object_unref(GTK_OBJECT(control));
}

/*
 * GnomeDbControl object
 */
static void
gnome_db_control_class_init (GnomeDbControlClass *klass)
{
	GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);
	POA_GNOME_DB_Control__epv *epv = &klass->epv;
	
	control_signals[GET_PROPERTY] =
		gtk_signal_new("get_property",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, get_property),
		               gtk_marshal_NONE__POINTER_POINTER,
		               GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_STRING);
	control_signals[SET_PROPERTY] =
		gtk_signal_new("set_property",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, set_property),
		               gtk_marshal_NONE__POINTER_POINTER_POINTER,
		               GTK_TYPE_NONE, 3, GTK_TYPE_POINTER, GTK_TYPE_STRING,
		               GTK_TYPE_POINTER);
	control_signals[DO_PRINT] =
		gtk_signal_new("do_print",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, do_print),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);
	control_signals[DO_CUT_CLIPBOARD] =
		gtk_signal_new("do_cut_clipboard",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, do_cut_clipboard),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);
	control_signals[DO_COPY_CLIPBOARD] =
		gtk_signal_new("do_copy_clipboard",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, do_copy_clipboard),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);
	control_signals[DO_PASTE_CLIPBOARD] =
		gtk_signal_new("do_paste_clipboard",
		               GTK_RUN_FIRST,
		               object_class->type,
		               GTK_SIGNAL_OFFSET(GnomeDbControlClass, do_paste_clipboard),
		               gtk_signal_default_marshaller,
		               GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals(object_class, control_signals, LAST_SIGNAL);
	
	klass->get_property = NULL;
	klass->set_property = NULL;
	klass->do_print = NULL;
	klass->do_cut_clipboard = NULL;
	klass->do_copy_clipboard = NULL;
	klass->do_paste_clipboard = NULL;

	epv->getVersion = impl_GNOME_DB_Control_getVersion;
	epv->getAuthors = impl_GNOME_DB_Control_getAuthors;
	epv->run = impl_GNOME_DB_Control_run;
	epv->doPrint = impl_GNOME_DB_Control_doPrint;
	epv->doCutClipboard = impl_GNOME_DB_Control_doCutClipboard;
	epv->doCopyClipboard = impl_GNOME_DB_Control_doCopyClipboard;
	epv->doPasteClipboard = impl_GNOME_DB_Control_doPasteClipboard;

}

static void
gnome_db_control_init (GnomeDbControl *control)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	control->priv = g_new0(GnomeDbControlPrivate, 1);
	
	control->priv->bonobo_control = NULL;
	control->priv->widget = NULL;
	control->priv->properties = NULL;
	control->priv->ui_component = NULL;
	control->priv->menubar = NULL;
	control->priv->toolbar = NULL;
	control->priv->user_data = NULL;
}

GtkType
gnome_db_control_get_type (void)
{
	static GtkType db_control_type = 0;
	
	if (!db_control_type) {
		GtkTypeInfo db_control_info = {
			"GnomeDbControl",
			sizeof (GnomeDbControl),
			sizeof (GnomeDbControlClass),
			(GtkClassInitFunc) gnome_db_control_class_init,
			(GtkObjectInitFunc) gnome_db_control_init,
			(GtkArgSetFunc) NULL,
			(GtkArgSetFunc) NULL
		};
		db_control_type = bonobo_x_type_unique(
			PARENT_TYPE,
			POA_GNOME_DB_Control__init, NULL,
			GTK_STRUCT_OFFSET (GnomeDbControlClass, epv),
			&db_control_info);
	}
	return db_control_type;
}

/**
 * gnome_db_control_construct
 */
GnomeDbControl *
gnome_db_control_construct (GnomeDbControl *control, GtkWidget *w)
{
	CORBA_Environment ev;
	
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	g_return_val_if_fail(GTK_IS_WIDGET(w), NULL);
	
	gtk_signal_connect(GTK_OBJECT(control),
	                   "destroy",
	                   GTK_SIGNAL_FUNC(control_destroyed_cb),
	                   NULL);

	control->priv->widget = w;
	gtk_signal_connect(GTK_OBJECT(control->priv->widget),
	                   "destroy",
	                   GTK_SIGNAL_FUNC(widget_destroyed_cb),
	                   (gpointer) control);

	control->priv->bonobo_control = bonobo_control_new(control->priv->widget);
	gtk_signal_connect(GTK_OBJECT(control->priv->bonobo_control),
	                   "activate",
	                   GTK_SIGNAL_FUNC(control_activated_cb),
	                   (gpointer) control);
	bonobo_object_add_interface(BONOBO_OBJECT(control),
	                            BONOBO_OBJECT(control->priv->bonobo_control));

	/* create property bag */
	control->priv->properties = bonobo_property_bag_new((BonoboPropertyGetFn) get_prop_cb,
	                                                          (BonoboPropertySetFn) set_prop_cb,
	                                                          (gpointer) control);
	bonobo_control_set_properties(control->priv->bonobo_control,
	                              control->priv->properties);

	/* create UI component */
	control->priv->ui_component = bonobo_ui_component_new_default();
	bonobo_control_set_ui_component(control->priv->bonobo_control,
	                                control->priv->ui_component);
	control->priv->ui_handler = bonobo_ui_handler_new_from_component(control->priv->ui_component);

	return control;
}

/**
 * gnome_db_control_new
 */
GnomeDbControl *
gnome_db_control_new (GtkWidget *w)
{
	GnomeDbControl* control;

	g_return_val_if_fail(GTK_IS_WIDGET(w), NULL);
	
	control = GNOME_DB_CONTROL(gtk_type_new(gnome_db_control_get_type()));

	return gnome_db_control_construct(control, w);
}

/**
 * gnome_db_control_set_menubar
 */
void
gnome_db_control_set_menubar (GnomeDbControl *control, GnomeUIInfo *menubar)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	control->priv->menubar = menubar;
}

/**
 * gnome_db_control_set_toolbar
 */
void
gnome_db_control_set_toolbar (GnomeDbControl *control, GnomeUIInfo *toolbar)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	control->priv->toolbar = toolbar;
}

/**
 * gnome_db_control_activate
 */
void
gnome_db_control_activate (GnomeDbControl *control)
{
	BonoboUIComponent* uic;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	uic = bonobo_control_get_ui_component(control->priv->bonobo_control);
	if (uic) {
		Bonobo_UIContainer remote_ui_container;
		
		remote_ui_container = bonobo_control_get_remote_ui_container(control->priv->bonobo_control);
		bonobo_ui_component_set_container(uic, remote_ui_container);
		bonobo_object_release_unref(remote_ui_container, NULL);

		bonobo_ui_component_freeze(uic, NULL);
		
		/* add menubar */
		if (control->priv->menubar) {
			BonoboUIHandlerMenuItem* menu_list;
			
			bonobo_ui_handler_create_menubar(control->priv->ui_handler);
			menu_list = bonobo_ui_handler_menu_parse_uiinfo_list_with_data(control->priv->menubar,
			                                                               control);
			bonobo_ui_handler_menu_add_list(control->priv->ui_handler, "/", menu_list);
			bonobo_ui_handler_menu_free_list(menu_list);
		}
		
		/* add toolbar */
		if (control->priv->toolbar) {
			BonoboUIHandlerToolbarItem* toolbar_list;
			
			bonobo_ui_handler_create_toolbar(control->priv->ui_handler, "Toolbar");
			toolbar_list = bonobo_ui_handler_toolbar_parse_uiinfo_list_with_data(control->priv->toolbar,
			                                                                     control);
			bonobo_ui_handler_toolbar_add_list(control->priv->ui_handler, "/Toolbar", toolbar_list);
			bonobo_ui_handler_toolbar_free_list(toolbar_list);
		}
		
		bonobo_ui_component_thaw(uic, NULL);
	}
}

/**
 * gnome_db_control_deactivate
 */
void
gnome_db_control_deactivate (GnomeDbControl *control)
{
	BonoboUIComponent* uic;
	
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	
	uic = bonobo_control_get_ui_component(control->priv->bonobo_control);
	if (uic)
		bonobo_ui_component_unset_container(uic);
}

/**
 * gnome_db_control_set_status
 */
void
gnome_db_control_set_status (GnomeDbControl *control, const gchar *msg)
{
	BonoboUIComponent* uic;
	
	g_return_if_fail(GNOME_DB_CONTROL(control));
	
	uic = bonobo_control_get_ui_component(control->priv->bonobo_control);
	if (uic) bonobo_ui_component_set_status(uic, msg, NULL);
}

/**
 * gnome_db_control_get_user_data
 */
gpointer
gnome_db_control_get_user_data (GnomeDbControl *control)
{
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	return control->priv->user_data;
}

/**
 * gnome_db_control_set_user_data
 */
void
gnome_db_control_set_user_data (GnomeDbControl *control, gpointer user_data)
{
	g_return_if_fail(GNOME_DB_IS_CONTROL(control));
	control->priv->user_data = user_data;
}

/**
 * gnome_db_control_get_widget
 */
GtkWidget *
gnome_db_control_get_widget (GnomeDbControl *control)
{
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	return control->priv->widget;
}

/**
 * gnome_db_control_get_bonobo_control
 */
BonoboControl *
gnome_db_control_get_bonobo_control (GnomeDbControl *control)
{
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
	return control->priv->bonobo_control;
}

/**
 * gnome_db_control_get_ui_container
 */
Bonobo_UIContainer
gnome_db_control_get_ui_container (GnomeDbControl *control)
{
	g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), CORBA_OBJECT_NIL);
	return bonobo_ui_component_get_container(control->priv->ui_component);
}
