/* GNOME DB library
 * Copyright (C) 1999-2001 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-browser.h"
#include "gnome-db-dataset.h"
#include "gnome-db-grid.h"
#include "gnome-db-list.h"
#include "gnome-db-util.h"
#include <libgnome/gnome-i18n.h>
#include <gal/e-paned/e-hpaned.h>

struct _GnomeDbBrowserPrivate {
	GdaConnection* cnc;
	GtkWidget*     container;
	GtkWidget*     table;
	GtkWidget*     object_selector;
	GtkWidget*     detail_toolbar;
	GtkWidget*     detail_container;
	GtkWidget*     detail;
};

static void gnome_db_browser_class_init (GnomeDbBrowserClass *klass);
static void gnome_db_browser_init (GnomeDbBrowser *browser);
static void gnome_db_browser_destroy (GnomeDbBrowser *browser, gpointer user_data);

static GtkWidget *create_table_detail (GnomeDbBrowser *browser, const gchar *name);

static void refresh_list_cb (GtkWidget *w, gpointer user_data);
static void show_extra_info_cb (GtkWidget *w, gpointer user_data);
static void show_table_data_cb (GtkWidget *w, gpointer user_data);
static void switch_notebook_page_cb (GtkNotebook *notebook,
									 GtkNotebookPage *page,
									 guint page_num,
									 gpointer user_data);

enum {
	CONNECTION_CHANGED,
	LAST_SIGNAL
};

static gint db_browser_signals[LAST_SIGNAL] = { 0, };
static GnomeUIInfo tables_toolbar[] = {
	{ GNOME_APP_UI_ITEM, N_("Data"), N_("Show table data"),
	  show_table_data_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_NEW, 0, 0, NULL },
	GNOMEUIINFO_END
};
static struct {
	gchar *label;
	GDA_Connection_QType qtype;
	gchar *pixmap_file;
	gchar *big_pixmap;
	GnomeUIInfo *toolbar;
	GtkWidget * (*create_detail)(GnomeDbBrowser *browser, const gchar *name);
} object_managers[] = {
	{ N_("Tables"), GDA_Connection_GDCN_SCHEMA_TABLES, GNOME_STOCK_MENU_BOOK_RED,
	  GNOME_STOCK_PIXMAP_BOOK_RED, tables_toolbar, create_table_detail },
	{ N_("Views"), GDA_Connection_GDCN_SCHEMA_VIEWS, GNOME_STOCK_MENU_SEARCH,
	  GNOME_STOCK_PIXMAP_SEARCH, NULL, NULL },
	{ N_("Procedures"), GDA_Connection_GDCN_SCHEMA_PROCS, GNOME_STOCK_MENU_EXEC,
	  GNOME_STOCK_PIXMAP_EXEC, NULL, NULL },
	{ N_("Types"), GDA_Connection_GDCN_SCHEMA_PROV_TYPES, GNOME_STOCK_MENU_FONT,
	  GNOME_STOCK_PIXMAP_FONT, NULL, NULL }
};
static GnomeUIInfo actions_toolbar[] = {
	{ GNOME_APP_UI_ITEM, N_("New"), N_("Create new object of selected type"),
	  NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_NEW, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Remove"), N_("Remove selected object"),
	  NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh current list of objects"),
	  refresh_list_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_REFRESH, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Info"), N_("Show extra information about selected object"),
	  show_extra_info_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_PROP, 0, 0, NULL },
	GNOMEUIINFO_END
};

static GtkVBoxClass *parent_class = NULL;

#define NUMBER_OF_OBJECTS (sizeof(object_managers) / sizeof(object_managers[0]))

/*
 * Private functions
 */
static GtkWidget *
create_table_detail (GnomeDbBrowser *browser, const gchar *name)
{
	GtkWidget *grid;
	GdaRecordset *recset;

	g_return_val_if_fail(GNOME_DB_IS_BROWSER(browser), NULL);

	grid = gnome_db_grid_new(NULL);
	gtk_widget_show(grid);

	if (gda_connection_is_open(browser->priv->cnc)) {
		/* get list of columns from database */
		recset = gda_connection_open_schema(browser->priv->cnc,
											GDA_Connection_GDCN_SCHEMA_COLS,
											GDA_Connection_OBJECT_NAME,
											name,
											GDA_Connection_no_CONSTRAINT);
		gnome_db_grid_set_recordset(GNOME_DB_GRID(grid), recset);
	}

	return grid;
}

static void
fill_object_list (GnomeDbBrowser *browser, GnomeDbList *list, GDA_Connection_QType qtype)
{
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(GNOME_DB_IS_LIST(list));
	g_return_if_fail(IS_GDA_CONNECTION(browser->priv->cnc));

	if (gda_connection_is_open(browser->priv->cnc)) {
		GdaRecordset *recset;

		recset = gda_connection_open_schema(browser->priv->cnc, qtype, GDA_Connection_no_CONSTRAINT);
		gnome_db_list_set_recordset(list, recset, 0);
	}
	else gnome_db_list_set_recordset(list, NULL, 0);
}

/**
 * Callbacks
 */
static void
connection_closed_cb (GdaConnection *cnc, gpointer user_data)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail(IS_GDA_CONNECTION(cnc));
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(browser->priv->cnc == cnc);

	/* FIXME: clean up all widgets */
	browser->priv->cnc = NULL;
	gnome_db_browser_refresh(browser);

	gtk_signal_emit(GTK_OBJECT(browser), db_browser_signals[CONNECTION_CHANGED]);
}

static void
object_selected_cb (GnomeDbList *list, gpointer user_data)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;
	gint current_page;

	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));

	current_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(browser->priv->object_selector));
	if (current_page >= 0 && current_page < NUMBER_OF_OBJECTS) {
		/* destroy previous widgets */
		if (browser->priv->detail != NULL) {
			gtk_widget_destroy(browser->priv->detail);
			browser->priv->detail = NULL;
		}
		if (GTK_IS_WIDGET(browser->priv->detail_toolbar)) {
			gtk_widget_destroy(browser->priv->detail_toolbar);
			browser->priv->detail_toolbar = NULL;
		}

		/* create detail for this object */
		if (object_managers[current_page].create_detail) {
			browser->priv->detail =
				object_managers[current_page].create_detail(browser,
															gnome_db_list_get_string(list));
			gtk_widget_show(browser->priv->detail);
			gtk_container_add(GTK_CONTAINER(browser->priv->detail_container), browser->priv->detail);
		}

		if (object_managers[current_page].toolbar) {
			browser->priv->detail_toolbar = gnome_db_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
																		GTK_TOOLBAR_ICONS,
																		tables_toolbar,
																		(gpointer) browser);
			gtk_table_attach(GTK_TABLE(browser->priv->table), browser->priv->detail_toolbar,
							 1, 2, 0, 1, 0, 0, 3, 3);
		}
	}
}

static void
row_added_cb (GnomeDbList *list, gint row, gpointer user_data)
{
	GnomePixmap *pixmap;
	gchar *pixmap_file = (gchar *) user_data;

	g_return_if_fail(GNOME_DB_IS_LIST(list));
	g_return_if_fail(pixmap_file != NULL);

	pixmap = gnome_db_get_pixmap(pixmap_file);
	if (pixmap) {
		gnome_db_list_set_pixmap(list, row, pixmap->pixmap, pixmap->mask);
	}
}

static void
refresh_list_cb (GtkWidget *w, gpointer user_data)
{
	GnomeDbList *list;
	gint page_num;
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));

	page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(browser->priv->object_selector));
	list = (GnomeDbList *) gtk_notebook_get_nth_page(GTK_NOTEBOOK(browser->priv->object_selector),
													 page_num);
	if (GNOME_DB_IS_LIST(list)) {
		fill_object_list(browser, list, object_managers[page_num].qtype);
	}
}

static void
show_extra_info_cb (GtkWidget *w, gpointer user_data)
{
	GnomeDbList *list;
	gint page_num;
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));

	page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(browser->priv->object_selector));
	list = (GnomeDbList *) gtk_notebook_get_nth_page(GTK_NOTEBOOK(browser->priv->object_selector),
													 page_num);
	if (GNOME_DB_IS_LIST(list)) {
		gchar *object_name;
		GdaRecordset *recset;
		GtkWidget *dialog;
		GtkWidget *table;
		GtkWidget *grid;
		GnomePixmap *pixmap;
		GtkWidget *pixmap_widget;

		object_name = gnome_db_list_get_string(list);
		if (object_name) {
			recset = gda_connection_open_schema(browser->priv->cnc,
												object_managers[page_num].qtype,
												GDA_Connection_EXTRA_INFO,
												"",
												GDA_Connection_OBJECT_NAME,
												object_name,
												GDA_Connection_no_CONSTRAINT);
			if (recset) {
				/* build the dialog */
				dialog = gnome_dialog_new(_("Extra information"),
										  GNOME_STOCK_BUTTON_CLOSE,
										  NULL);
				table = gnome_db_new_table_widget(3, 2, FALSE);
				gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, 0);

				pixmap = gnome_db_get_pixmap(object_managers[page_num].big_pixmap);
				if (pixmap) {
					pixmap_widget = GTK_WIDGET(gnome_pixmap_new_from_gnome_pixmap(pixmap));
					gtk_widget_show(pixmap_widget);
					gtk_table_attach(GTK_TABLE(table), pixmap_widget, 0, 1, 0, 1,
									 GTK_FILL, GTK_FILL, 3, 3);
				}

				grid = gnome_db_dataset_new(recset);
				gnome_db_dataset_set_show_buttons(GNOME_DB_DATASET(grid), FALSE);
				gtk_widget_show(grid);
				gtk_table_attach(GTK_TABLE(table), grid, 0, 2, 1, 3,
								 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
								 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
								 3, 3);

				/* run the dialog */
				gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
				gda_recordset_free(recset);
			}
		}
	}
}

static void
show_table_data_cb (GtkWidget *w, gpointer user_data)
{
	gint page_num;
	GtkWidget *list;
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));

	page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(browser->priv->object_selector));
	list = (GnomeDbList *) gtk_notebook_get_nth_page(GTK_NOTEBOOK(browser->priv->object_selector),
													 page_num);
	if (GNOME_DB_IS_LIST(list) && IS_GDA_CONNECTION(browser->priv->cnc)) {
		gchar *object_name;

		object_name = gnome_db_list_get_string(list);
		if (object_name) {
			GdaCommand *cmd;
			gulong reccount;
			GdaRecordset *recset;
			GtkWidget *dialog;
			GtkWidget *frame;
			GtkWidget *grid;

			/* create the command object */
			cmd = gda_command_new();
			gda_command_set_connection(cmd, browser->priv->cnc);
			gda_command_set_cmd_type(cmd, GDA_COMMAND_TYPE_TABLE);
			gda_command_set_text(cmd, object_name);

			recset = gda_command_execute(cmd, &reccount, NULL);

			/* create dialog */
			dialog = gnome_dialog_new(_("Table data"), GNOME_STOCK_BUTTON_CLOSE, NULL);
			gtk_widget_set_usize(dialog, 450, 350);
			frame = gnome_db_new_frame_widget(NULL);
			gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);

			grid = gnome_db_grid_new(recset);
			gtk_widget_show(grid);
			gtk_container_add(GTK_CONTAINER(frame), grid);

			gnome_dialog_run_and_close(GNOME_DIALOG(dialog));

			/* free memory */
			gda_recordset_free(recset);
			gda_command_free(cmd);
		}
	}
}

static void
switch_notebook_page_cb (GtkNotebook *notebook,
						 GtkNotebookPage *page,
						 guint page_num,
						 gpointer user_data)
{
	GnomeDbList *list;
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(browser->priv != NULL);
	g_return_if_fail(browser->priv->cnc != NULL);
	g_return_if_fail(IS_GDA_CONNECTION(browser->priv->cnc));

	list = (GnomeDbList *) gtk_notebook_get_nth_page(notebook, page_num);
	if (GNOME_DB_IS_LIST(list)) {
		if (!gnome_db_list_get_recordset(list))
			fill_object_list(browser, list, object_managers[page_num].qtype);

		/* destroy previous widgets */
		if (browser->priv->detail != NULL) {
			gtk_widget_destroy(browser->priv->detail);
			browser->priv->detail = NULL;
		}
		if (GTK_IS_WIDGET(browser->priv->detail_toolbar)) {
			gtk_widget_destroy(browser->priv->detail_toolbar);
			browser->priv->detail_toolbar = NULL;
		}
	}
}

/*
 * GnomeDbBrowser interface
 */
static void
gnome_db_browser_class_init (GnomeDbBrowserClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class(gtk_vbox_get_type());

	db_browser_signals[CONNECTION_CHANGED] =
		gtk_signal_new("connection_changed",
					   GTK_RUN_LAST,
					   object_class->type,
					   GTK_SIGNAL_OFFSET(GnomeDbBrowserClass, connection_changed),
					   gtk_signal_default_marshaller,
					   GTK_TYPE_NONE, 0);
	gtk_object_class_add_signals(object_class, db_browser_signals, LAST_SIGNAL);

	object_class->destroy = gnome_db_browser_destroy;
	klass->connection_changed = NULL;
}

static void
gnome_db_browser_destroy (GnomeDbBrowser *browser, gpointer data)
{
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));

	gtk_signal_disconnect_by_func(GTK_OBJECT(browser->priv->object_selector),
								  GTK_SIGNAL_FUNC(switch_notebook_page_cb),
								  (gpointer) browser);
	if (IS_GDA_CONNECTION(browser->priv->cnc)) {
		gtk_signal_disconnect_by_func(GTK_OBJECT(browser->priv->cnc),
		                              GTK_SIGNAL_FUNC(connection_closed_cb),
		                              (gpointer) browser);
	}
	browser->priv->cnc = NULL;
	g_free((gpointer) browser->priv);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(*GTK_OBJECT_CLASS(parent_class)->destroy)(GTK_OBJECT(browser));
}

static void
gnome_db_browser_init (GnomeDbBrowser *browser)
{
	gint n;
	GtkWidget *toolbar;

	/* allocate private structure */
	browser->priv = g_new0(GnomeDbBrowserPrivate, 1);

	/* create container */
	browser->priv->container = e_hpaned_new();
	gtk_widget_show(browser->priv->container);
	gtk_box_pack_start(GTK_BOX(browser), browser->priv->container, 1, 1, 0);
	browser->priv->object_selector = gnome_db_new_notebook_widget();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(browser->priv->object_selector), GTK_POS_TOP);
	e_paned_add1(E_PANED(browser->priv->container), browser->priv->object_selector);
	for (n = 0; n < NUMBER_OF_OBJECTS; n++) {
		GtkWidget *list;

		list = gnome_db_list_new(NULL, 0);
		gtk_signal_connect(GTK_OBJECT(list),
						   "select_row",
						   GTK_SIGNAL_FUNC(object_selected_cb),
						   (gpointer) browser);
		gtk_signal_connect(GTK_OBJECT(list),
						   "add_row",
						   GTK_SIGNAL_FUNC(row_added_cb),
						   (gpointer) object_managers[n].pixmap_file);
		gtk_widget_show(list);
		gtk_notebook_append_page(GTK_NOTEBOOK(browser->priv->object_selector),
								 list,
								 gtk_label_new(object_managers[n].label));
	}

	/* create detail container */
	browser->priv->table = gnome_db_new_table_widget(2, 6, FALSE);
	toolbar = gnome_db_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
										  GTK_TOOLBAR_ICONS,
										  actions_toolbar,
										  (gpointer) browser);
	gtk_table_attach(GTK_TABLE(browser->priv->table), toolbar, 0, 1, 0, 1, 0, 0, 3, 3);

	browser->priv->detail_container = gnome_db_new_frame_widget(NULL);
	gtk_table_attach(GTK_TABLE(browser->priv->table), browser->priv->detail_container, 0, 6, 1, 2,
					 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
					 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
					 3, 3);
	e_paned_add2(E_PANED(browser->priv->container), browser->priv->table);
	e_paned_set_position(E_PANED(browser->priv->container), 200);

	/* connect signals */
	gtk_signal_connect(GTK_OBJECT(browser->priv->object_selector),
					   "switch_page",
					   GTK_SIGNAL_FUNC(switch_notebook_page_cb),
					   (gpointer) browser);
}

GtkType
gnome_db_browser_get_type (void)
{
	static GtkType db_browser_type = 0;
	
	if (!db_browser_type) {
		GtkTypeInfo db_browser_info = {
			"GnomeDbBrowser",
			sizeof (GnomeDbBrowser),
			sizeof (GnomeDbBrowserClass),
			(GtkClassInitFunc) gnome_db_browser_class_init,
			(GtkObjectInitFunc) gnome_db_browser_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		db_browser_type = gtk_type_unique(gtk_vbox_get_type(), &db_browser_info);
	}
	return (db_browser_type);
}

/**
 * gnome_db_browser_new
 */
GtkWidget *
gnome_db_browser_new (GdaConnection *cnc)
{
	GnomeDbBrowser* browser;
	
	browser = GNOME_DB_BROWSER(gtk_type_new(gnome_db_browser_get_type()));
	if (IS_GDA_CONNECTION(cnc)) {
		gnome_db_browser_set_connection(browser, cnc);
	}
	return GTK_WIDGET(browser);
}

/**
 * gnome_db_browser_get_connection
 */
GdaConnection *
gnome_db_browser_get_connection (GnomeDbBrowser *browser)
{
	g_return_val_if_fail(GNOME_DB_IS_BROWSER(browser), NULL);
	g_return_val_if_fail(browser->priv != NULL, NULL);
	return browser->priv->cnc;
}

/**
 * gnome_db_browser_set_connection
 */
void
gnome_db_browser_set_connection (GnomeDbBrowser *browser, GdaConnection *cnc)
{
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(browser->priv != NULL);

	browser->priv->cnc = cnc;
	gnome_db_browser_refresh(browser);

	/* connect callback to "close" signal */
	if (IS_GDA_CONNECTION(cnc)) {
		gtk_signal_connect(GTK_OBJECT(cnc),
		                   "close",
		                   GTK_SIGNAL_FUNC(connection_closed_cb),
		                   (gpointer) browser);
	}

	gtk_signal_emit(GTK_OBJECT(browser), db_browser_signals[CONNECTION_CHANGED]);
}

/**
 * gnome_db_browser_refresh
 */
void
gnome_db_browser_refresh (GnomeDbBrowser *browser)
{
	gint page_num;
	gint n;
	GnomeDbList *list;

	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(browser->priv != NULL);

	page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(browser->priv->object_selector));

	for (n = 0; n < NUMBER_OF_OBJECTS; n++) {
		list = (GnomeDbList *) gtk_notebook_get_nth_page(GTK_NOTEBOOK(browser->priv->object_selector),
														 n);
		if (GNOME_DB_IS_LIST(list)) {
			/* refresh only current list */
			if (n == page_num)
				fill_object_list(browser, list, object_managers[n].qtype);
			else
				gnome_db_list_set_recordset(list, NULL, 0);
		}
	}
	gtk_widget_destroy(browser->priv->detail);
}

