/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome-db.h>
#include <config.h>

static void gnome_db_designer_init       (GnomeDbDesigner *designer);
static void gnome_db_designer_class_init (GnomeDbDesignerClass *klass);

static void gnome_db_designer_destroy    (GnomeDbDesigner *designer);

/*
 * Private functions
 */
static void
add_object_nodes (GnomeDbDesigner *designer)
{
  gchar*       ctree_row[2] = { "", "" };

  ctree_row[1] = _("Tables");
  designer->tables_node = gtk_ctree_insert_node(GTK_CTREE(designer->object_tree),
                                                NULL, NULL,
                                                ctree_row, 0,
                                                NULL,
                                                NULL,
                                                NULL,
                                                NULL,
                                                FALSE, TRUE);
  gtk_ctree_expand(GTK_CTREE(designer->object_tree), designer->tables_node);
  ctree_row[1] = _("Views");
  designer->views_node = gtk_ctree_insert_node(GTK_CTREE(designer->object_tree),
                                               NULL, NULL,
                                               ctree_row, 0,
                                               NULL,
                                               NULL,
                                               NULL,
                                               NULL,
                                               FALSE, TRUE);
  gtk_ctree_expand(GTK_CTREE(designer->object_tree), designer->views_node);
}

static GtkWidget *
show_table_detail (GnomeDbDesigner *designer, xmlNodePtr xmlnode)
{
  GtkWidget* container;
  GtkWidget* frame;
  GtkWidget* table;
  GtkWidget* label;
  GtkWidget* entry;
  GtkWidget* scroll;
  GtkWidget* clist;
  gint       field_count;
  gint       cnt;
  gchar*     clist_titles[] = { N_("Name"), N_("GDA Type"), N_("Size"), N_("Scale") };
  
  g_return_val_if_fail(GNOME_DB_IS_DESIGNER(designer), NULL);
  g_return_val_if_fail(xmlnode != NULL, NULL);
  
  /* create container */
  container = gnome_db_new_table_widget(1, 4, FALSE);
  
  /* detail pane */
  frame = gnome_db_new_frame_widget(NULL);
  gtk_table_attach(GTK_TABLE(container), frame, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  table = gnome_db_new_table_widget(3, 3, FALSE);
  gtk_container_add(GTK_CONTAINER(frame), table);
  
  label = gnome_db_new_label_widget(_("Table name"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  entry = gnome_db_new_entry_widget(0, FALSE);
  gtk_entry_set_text(GTK_ENTRY(entry),
                     gda_xml_database_table_get_name(designer->xmldb, xmlnode));
  gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

  label = gnome_db_new_label_widget(_("Owner"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  entry = gnome_db_new_entry_widget(0, FALSE);
  gtk_entry_set_text(GTK_ENTRY(entry),
                     gda_xml_database_table_get_owner(designer->xmldb, xmlnode));
  gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  
  /* create description area */
  scroll = gnome_db_new_scrolled_window_widget();
  gtk_table_attach(GTK_TABLE(container), scroll, 0, 1, 1, 4,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  clist = gnome_db_new_clist_widget(clist_titles,
                                    sizeof(clist_titles) / sizeof(clist_titles[0]));
  gtk_container_add(GTK_CONTAINER(scroll), clist);
  
  field_count = gda_xml_database_table_field_count(designer->xmldb, xmlnode);
  gtk_clist_freeze(GTK_CLIST(clist));
  for (cnt = 0; cnt < field_count; cnt++)
    {
      gchar*     row[4];
      xmlNodePtr field;
      
      field = gda_xml_database_table_get_field(designer->xmldb, xmlnode, cnt);
      if (field)
        {
          row[0] = gda_xml_database_field_get_name(designer->xmldb, field);
          row[1] = gda_xml_database_field_get_gdatype(designer->xmldb, field);
          row[2] = g_strdup_printf("%d", gda_xml_database_field_get_size(designer->xmldb, field));
          row[3] = g_strdup_printf("%d", gda_xml_database_field_get_scale(designer->xmldb, field));
          gtk_clist_append(GTK_CLIST(clist), row);
          g_free((gpointer) row[2]);
          g_free((gpointer) row[3]);
        }
    }
  gtk_clist_thaw(GTK_CLIST(clist));

  return container;
}

/*
 * Callbacks
 */
static void
select_tree_row_cb (GtkCTree *ctree, GtkCTreeNode *row, gint column, GnomeDbDesigner *designer)
{
  xmlNodePtr xmlnode;

  g_return_if_fail(GTK_IS_CTREE(ctree));
  g_return_if_fail(row != NULL);
  g_return_if_fail(GNOME_DB_IS_DESIGNER(designer));
  
  /* clear the current detail */
  if (GTK_IS_WIDGET(designer->detail)) gtk_widget_destroy(designer->detail);
  designer->detail = NULL;

  xmlnode = (xmlNodePtr) gtk_ctree_node_get_row_data(ctree, row);
  if (xmlnode)
    {
      if (xmlnode->parent == gda_xml_database_get_tables(designer->xmldb))
        {
          designer->detail = show_table_detail(designer, xmlnode);
        }

      if (GTK_IS_WIDGET(designer->detail))
        {
          gtk_widget_show(designer->detail);
          gtk_container_add(GTK_CONTAINER(designer->detail_container), designer->detail);
        }
    }
}

/*
 * GnomeDbDesigner widget interface
 */
static void
gnome_db_designer_class_init (GnomeDbDesignerClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);

  object_class->destroy = gnome_db_designer_destroy;
}

static void
gnome_db_designer_init (GnomeDbDesigner *designer)
{
  GtkWidget* table;
  GtkWidget* frame;
  GtkWidget* scroll;

  /* create main container */
  table = gnome_db_new_table_widget(4, 1, TRUE);
  gtk_box_pack_start(GTK_BOX(designer), table, 1, 1, 0);

  frame = gnome_db_new_frame_widget(NULL);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  scroll = gnome_db_new_scrolled_window_widget();
  gtk_container_add(GTK_CONTAINER(frame), scroll);
  designer->object_tree = gnome_db_new_ctree_widget(NULL, 2);
  add_object_nodes(designer);
  gtk_container_add(GTK_CONTAINER(scroll), designer->object_tree);
  gtk_signal_connect_after(GTK_OBJECT(designer->object_tree),
                           "tree_select_row",
                           GTK_SIGNAL_FUNC(select_tree_row_cb),
                           (gpointer) designer);

  /* create detail pane */
  designer->detail_container = gnome_db_new_frame_widget(NULL);
  gtk_table_attach(GTK_TABLE(table), designer->detail_container, 1, 4, 0, 1,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   3, 3);

  /* initialize fields */
  designer->detail = NULL;
  designer->xmldb = NULL;
}

GtkType
gnome_db_designer_get_type (void)
{
  static guint db_designer_type = 0;

  if (!db_designer_type)
    {
      GtkTypeInfo db_designer_info =
      {
        "GnomeDbDesigner",
        sizeof (GnomeDbDesigner),
        sizeof (GnomeDbDesignerClass),
        (GtkClassInitFunc) gnome_db_designer_class_init,
        (GtkObjectInitFunc) gnome_db_designer_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_designer_type = gtk_type_unique(gtk_vbox_get_type(), &db_designer_info);
    }
  return (db_designer_type);
}

/**
 * gnome_db_designer_new
 *
 * Create a new #GnomeDbDesigner widget. This widget lets you visually
 * manage a XML database
 */
GtkWidget *
gnome_db_designer_new (void)
{
  GnomeDbDesigner* designer;

  designer = GNOME_DB_DESIGNER(gtk_type_new(gnome_db_designer_get_type()));
  designer->xmldb = gda_xml_database_new();

  return GTK_WIDGET(designer);
}

/**
 * gnome_db_designer_new_from_file
 */
GtkWidget *
gnome_db_designer_new_from_file (const gchar *filename)
{
  GnomeDbDesigner* designer;

  designer = GNOME_DB_DESIGNER(gnome_db_designer_new());
  gnome_db_designer_load_file(designer, filename);

  return GTK_WIDGET(designer);
}

/**
 * gnome_db_designer_new_from_xml
 * @xmldb: XML database
 *
 * Create a new #GnomeDbDesigner widget from the given XML database
 */
GtkWidget *
gnome_db_designer_new_from_xml (Gda_XmlDatabase *xmldb)
{
  GnomeDbDesigner* designer;

  designer = GNOME_DB_DESIGNER(gtk_type_new(gnome_db_designer_get_type()));
  designer->xmldb = xmldb;
  gnome_db_designer_refresh(designer);

  return GTK_WIDGET(designer);
}

static void
gnome_db_designer_destroy (GnomeDbDesigner *designer)
{
  g_return_if_fail(GNOME_DB_IS_DESIGNER(designer));

  gda_xml_database_free(designer->xmldb);
}

/**
 * gnome_db_designer_load_file
 */
void
gnome_db_designer_load_file (GnomeDbDesigner *designer, const gchar *filename)
{
  Gda_XmlDatabase *xmldb;
  
  g_return_if_fail(GNOME_DB_IS_DESIGNER(designer));
  g_return_if_fail(filename != NULL);
  
  xmldb = gda_xml_database_new_from_file(filename);
  if (GDA_IS_XML_DATABASE(xmldb))
    {
      if (GDA_IS_XML_DATABASE(designer->xmldb))
        gda_xml_database_free(designer->xmldb);
      designer->xmldb = xmldb;
      gnome_db_designer_refresh(designer);
    }
  else gnome_db_show_error(_("File %s is not a valid XML file"), filename);
}

/**
 * gnome_db_designer_refresh
 */
void
gnome_db_designer_refresh (GnomeDbDesigner *designer)
{
  xmlNodePtr   xmlnode;
  xmlNodePtr   xmlchild;
  GnomePixmap* pixmap;
  
  g_return_if_fail(GNOME_DB_IS_DESIGNER(designer));
  
  gtk_ctree_remove_node(GTK_CTREE(designer->object_tree), designer->tables_node);
  gtk_ctree_remove_node(GTK_CTREE(designer->object_tree), designer->views_node);
  add_object_nodes(designer);
  
  /* get pixmaps */
  pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_BOOK_RED);
  
  /* refresh list of tables */
  if ((xmlnode = gda_xml_database_get_tables(designer->xmldb)))
    {
      for (xmlchild = xmlnode->childs;
           xmlchild != NULL;
           xmlchild = xmlchild->next)
        {
          gchar*        ctree_row[2] = { "", "" };
          GtkCTreeNode* new_node;

          ctree_row[1] = gda_xml_database_table_get_name(designer->xmldb, xmlchild);
          new_node = gtk_ctree_insert_node(GTK_CTREE(designer->object_tree),
                                           designer->tables_node, NULL,
                                           ctree_row, 0,
                                           GNOME_PIXMAP(pixmap)->pixmap,
                                           GNOME_PIXMAP(pixmap)->mask,
                                           GNOME_PIXMAP(pixmap)->pixmap,
                                           GNOME_PIXMAP(pixmap)->mask,
                                           TRUE, FALSE);
          gtk_ctree_node_set_row_data(GTK_CTREE(designer->object_tree),
                                      new_node,
                                      (gpointer) xmlchild);
        }
    }
}

/**
 * gnome_db_designer_get_database
 */
Gda_XmlDatabase *
gnome_db_designer_get_database (GnomeDbDesigner *designer)
{
  g_return_val_if_fail(GNOME_DB_IS_DESIGNER(designer), NULL);
  return designer->xmldb;
}
