/* GNOME DB library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome-db-moniker.h>

struct _GnomeDbMonikerPrivate {
	gchar*                    oafiid;
	GnomeDbMonikerResolveFunc resolve_func;
};

/*
 * Private functions
 */
static void
moniker_destroy_cb (GnomeDbMoniker *moniker, gpointer data) {
	g_return_if_fail(GNOME_DB_IS_MONIKER(moniker));
	
	if (moniker->priv) {
		g_free((gpointer) moniker->priv->oafiid);
		g_free((gpointer) moniker->priv);
	}
}

static Bonobo_Moniker
moniker_parse_display_name (BonoboMoniker *moniker,
                            Bonobo_Moniker parent,
                            const CORBA_char *name,
                            CORBA_Environment *ev) {
}

static Bonobo_Unknown
moniker_resolve (BonoboMoniker *moniker,
                 const Bonobo_ResolveOptions *options,
                 const CORBA_char *requested_interface,
                 CORBA_Environment *ev) {
    BonoboObject*   new_client;
    GnomeDbMoniker* db_moniker = GNOME_DB_MONIKER(moniker);
    
	g_return_val_if_fail(GNOME_DB_IS_MONIKER(db_moniker), CORBA_OBJECT_NIL);
	g_return_val_if_fail(requested_interface != NULL, CORBA_OBJECT_NIL);
	g_return_val_if_fail(db_moniker->priv != NULL, CORBA_OBJECT_NIL);
	g_return_val_if_fail(db_moniker->priv->resolve_func != NULL, CORBA_OBJECT_NIL);
	
	new_client = db_moniker->priv->resolve_func(GNOME_DB_MONIKER(db_moniker),
	                                            (const gchar *) requested_interface);
	if (BONOBO_IS_OBJECT(new_client)) {
		Bonobo_Unknown object;
		
		object = bonobo_object_corba_objref(new_client);
		if (ev->_major != CORBA_NO_EXCEPTION) return CORBA_OBJECT_NIL;
		if (object == CORBA_OBJECT_NIL) {
			g_warning ("Can't find object satisfying requirements");
			CORBA_exception_set(ev, CORBA_USER_EXCEPTION, ex_Bonobo_Moniker_InterfaceNotFound, NULL);
			return CORBA_OBJECT_NIL;
		}
		
		return bonobo_moniker_util_qi_return(object, requested_interface, ev);
	}
	return CORBA_OBJECT_NIL;
}

/*
 * GnomeDbMoniker object interface
 */
static void
gnome_db_moniker_class_init (GnomeDbMonikerClass *klass) {
	BonoboMonikerClass* mclass = (BonoboMonikerClass *) klass;
	
	mclass->parse_display_name = moniker_parse_display_name;
	mclass->resolve = moniker_resolve;
}

static void
gnome_db_moniker_init (GnomeDbMoniker *moniker) {
	g_return_if_fail(GNOME_DB_IS_MONIKER(moniker));
	
	moniker->priv = g_new0(GnomeDbMonikerPrivate, 1);
	gtk_signal_connect(GTK_OBJECT(moniker),
	                   "destroy",
	                   GTK_SIGNAL_FUNC(moniker_destroy_cb),
	                   NULL);
}

GtkType
gnome_db_moniker_get_type (void) {
	static GtkType type = 0;
	
	if (!type) {
		GtkTypeInfo info = {
			"GnomeDbMoniker",
			sizeof(GnomeDbMoniker),
			sizeof(GnomeDbMonikerClass),
			(GtkClassInitFunc) gnome_db_moniker_class_init,
			(GtkObjectInitFunc) gnome_db_moniker_init,
			NULL,
			NULL,
			(GtkClassInitFunc) NULL
		};
		type = gtk_type_unique(bonobo_moniker_get_type(), &info);
	}
	return type;
}

/**
 * gnome_db_moniker_new
 */
GnomeDbMoniker *
gnome_db_moniker_new (const gchar *oafiid,
                      const gchar *prefix,
                      GnomeDbMonikerResolveFunc resolve_func) {
	GnomeDbMoniker* moniker;
	
	//g_return_val_if_fail(oafiid != NULL, NULL);
	g_return_val_if_fail(prefix != NULL, NULL);
	
	//moniker = GNOME_DB_MONIKER(bonobo_moniker_construct(gtk_type_new(gnome_db_moniker_get_type()),
	//                                                    CORBA_OBJECT_NIL,
	//                                                    prefix));
	if (!GNOME_DB_IS_MONIKER(moniker)) {
		g_warning(_("Could not construct '%s' moniker"), prefix);
		return NULL;
	}
	
	moniker->priv->oafiid = g_strdup(oafiid);
	moniker->priv->resolve_func = resolve_func;

	return moniker;
}

/**
 * gnome_db_moniker_get_oafiid
 */
const gchar *
gnome_db_moniker_get_oafiid (GnomeDbMoniker *moniker) {
	g_return_val_if_fail(GNOME_DB_IS_MONIKER(moniker), NULL);
	g_return_val_if_fail(moniker->priv != NULL, NULL);
	
	return (const gchar *) moniker->priv->oafiid;
}

