/* GNOME DB components libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-control.h"
#include "gnome-db-control-corba.h"
#include "bonobo-ui-compat.h"

struct _GnomeDbControlInstancePrivate
{
  GnomeDbControl*    control;
  BonoboControl*     bonobo_control;
  GtkWidget*         widget;
  BonoboPropertyBag* properties;
  BonoboUIComponent* ui_component;
  BonoboUIHandler*   ui_handler;
  GnomeUIInfo*       menubar;
  GnomeUIInfo*       toolbar;
  gpointer           user_data;
};

enum
{
  GET_PROPERTY,
  SET_PROPERTY,
  DO_PRINT,
  DO_CUT_CLIPBOARD,
  DO_COPY_CLIPBOARD,
  DO_PASTE_CLIPBOARD,
  LAST_SIGNAL
};

static guint control_instance_signals[LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
static void
control_activated_cb (BonoboControl *bonobo_control,
                      gboolean activate,
                      GnomeDbControlInstance *control_instance)
{
  if (activate) gnome_db_control_instance_activate(control_instance);
  else gnome_db_control_instance_deactivate(control_instance);
}

static void
control_destroyed_cb (GnomeDbControlInstance *control_instance, gpointer data)
{
  CORBA_Environment ev;

  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));  
  g_free((gpointer) control_instance->priv);
}

static void
get_prop_cb (BonoboPropertyBag *bag, BonoboArg *arg, guint arg_id, gpointer user_data)
{
}

static void
set_prop_cb (BonoboPropertyBag *bag, const BonoboArg *arg, guint arg_id, gpointer user_data)
{
}

static void
widget_destroyed_cb (GtkWidget *w, GnomeDbControlInstance *control_instance)
{
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  gtk_object_unref(GTK_OBJECT(control_instance));
}

/*
 * GnomeDbControlInstance object
 */
static void
gnome_db_control_instance_class_init (GnomeDbControlInstanceClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);

  control_instance_signals[GET_PROPERTY] =
    gtk_signal_new("get_property",
                   GTK_RUN_FIRST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET(GnomeDbControlInstanceClass, get_property),
                   gtk_marshal_NONE__POINTER_POINTER,
                   GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_STRING);
  control_instance_signals[SET_PROPERTY] =
    gtk_signal_new("set_property",
                   GTK_RUN_FIRST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET(GnomeDbControlInstanceClass, set_property),
                   gtk_marshal_NONE__POINTER_POINTER_POINTER,
                   GTK_TYPE_NONE, 3, GTK_TYPE_POINTER, GTK_TYPE_STRING,
                   GTK_TYPE_POINTER);
  control_instance_signals[DO_PRINT] =
    gtk_signal_new("do_print",
                   GTK_RUN_FIRST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET(GnomeDbControlInstanceClass, do_print),
                   gtk_signal_default_marshaller,
                   GTK_TYPE_NONE, 0);
  control_instance_signals[DO_CUT_CLIPBOARD] =
    gtk_signal_new("do_cut_clipboard",
                   GTK_RUN_FIRST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET(GnomeDbControlInstanceClass, do_cut_clipboard),
                   gtk_signal_default_marshaller,
                   GTK_TYPE_NONE, 0);
  control_instance_signals[DO_COPY_CLIPBOARD] =
    gtk_signal_new("do_copy_clipboard",
                   GTK_RUN_FIRST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET(GnomeDbControlInstanceClass, do_copy_clipboard),
                   gtk_signal_default_marshaller,
                   GTK_TYPE_NONE, 0);
  control_instance_signals[DO_PASTE_CLIPBOARD] =
    gtk_signal_new("do_paste_clipboard",
                   GTK_RUN_FIRST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET(GnomeDbControlInstanceClass, do_paste_clipboard),
                   gtk_signal_default_marshaller,
                   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, control_instance_signals, LAST_SIGNAL);

  klass->get_property = NULL;
  klass->set_property = NULL;
}

static void
gnome_db_control_instance_init (GnomeDbControlInstance *control_instance)
{
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  
  control_instance->priv = g_new0(GnomeDbControlInstancePrivate, 1);
  
  control_instance->priv->control = NULL;
  control_instance->priv->bonobo_control = NULL;
  control_instance->priv->widget = NULL;
  control_instance->priv->properties = NULL;
  control_instance->priv->ui_component = NULL;
  control_instance->priv->menubar = NULL;
  control_instance->priv->toolbar = NULL;
  control_instance->priv->user_data = NULL;
}

GtkType
gnome_db_control_instance_get_type (void)
{
  static GtkType db_control_instance_type = 0;

  if (!db_control_instance_type)
    {
      GtkTypeInfo db_control_instance_info =
      {
        "GnomeDbControlInstance",
        sizeof (GnomeDbControlInstance),
        sizeof (GnomeDbControlInstanceClass),
        (GtkClassInitFunc) gnome_db_control_instance_class_init,
        (GtkObjectInitFunc) gnome_db_control_instance_init,
        (GtkArgSetFunc) NULL,
        (GtkArgSetFunc) NULL
      };
      db_control_instance_type = gtk_type_unique(bonobo_object_get_type(), &db_control_instance_info);
    }
  return db_control_instance_type;
}

/**
 * gnome_db_control_instance_construct
 */
GnomeDbControlInstance *
gnome_db_control_instance_construct (GnomeDbControlInstance *control_instance,
                                     GnomeDbControl *control,
                                     GtkWidget *w)
{
  CORBA_Environment       ev;
  GnomeDbControlInstance* real_instance;
  
  g_return_val_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance), NULL);
  g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
  g_return_val_if_fail(GTK_IS_WIDGET(w), NULL);
  
  /* start Bonobo/CORBA stuff */
  CORBA_exception_init(&ev);
  real_instance = bonobo_object_construct(BONOBO_OBJECT(control_instance),
                                          impl_GNOME_DB_Control__create(control_instance, &ev));
  if (!GNOME_DB_IS_CONTROL_INSTANCE(real_instance))
    {
      gtk_object_unref(GTK_OBJECT(control_instance));
      return NULL;
    }
  CORBA_exception_free(&ev);
  
  gtk_signal_connect(GTK_OBJECT(real_instance),
                     "destroy",
                     GTK_SIGNAL_FUNC(control_destroyed_cb),
                     NULL);
  
  real_instance->priv->control = control;
  real_instance->priv->widget = w;
  gtk_signal_connect(GTK_OBJECT(real_instance->priv->widget),
                     "destroy",
                     GTK_SIGNAL_FUNC(widget_destroyed_cb),
                     (gpointer) real_instance);

  real_instance->priv->bonobo_control = bonobo_control_new(real_instance->priv->widget);
  gtk_signal_connect(GTK_OBJECT(real_instance->priv->bonobo_control),
                     "activate",
                     GTK_SIGNAL_FUNC(control_activated_cb),
                     (gpointer) real_instance);
  bonobo_object_add_interface(BONOBO_OBJECT(real_instance),
                              BONOBO_OBJECT(real_instance->priv->bonobo_control));
  
  /* create property bag */
  real_instance->priv->properties = bonobo_property_bag_new(get_prop_cb, set_prop_cb, real_instance);
  bonobo_control_set_properties(real_instance->priv->bonobo_control,
                                  real_instance->priv->properties);
  
  /* create UI component */
  real_instance->priv->ui_component = bonobo_ui_component_new_default();
  bonobo_control_set_ui_component(real_instance->priv->bonobo_control,
                                  real_instance->priv->ui_component);
  real_instance->priv->ui_handler = bonobo_ui_handler_new_from_component(real_instance->priv->ui_component);

  return real_instance;
}

/**
 * gnome_db_control_instance_new
 */
GnomeDbControlInstance *
gnome_db_control_instance_new (GnomeDbControl *control, GtkWidget *w)
{
  GnomeDbControlInstance* control_instance;
  
  g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
  g_return_val_if_fail(GTK_IS_WIDGET(w), NULL);
  
  control_instance = GNOME_DB_CONTROL_INSTANCE(gtk_type_new(gnome_db_control_instance_get_type()));
  return gnome_db_control_instance_construct(control_instance, control, w);
}

/**
 * gnome_db_control_instance_set_menubar
 */
void
gnome_db_control_instance_set_menubar (GnomeDbControlInstance *control_instance,
                                       GnomeUIInfo *menubar)
{
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  control_instance->priv->menubar = menubar;
}

/**
 * gnome_db_control_instance_set_toolbar
 */
void
gnome_db_control_instance_set_toolbar (GnomeDbControlInstance *control_instance,
                                       GnomeUIInfo *toolbar)
{
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  control_instance->priv->toolbar = toolbar;
}

/**
 * gnome_db_control_instance_activate
 */
void
gnome_db_control_instance_activate (GnomeDbControlInstance *control_instance)
{
  BonoboUIComponent* uic;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  
  uic = bonobo_control_get_ui_component(control_instance->priv->bonobo_control);
  if (uic)
    {
      Bonobo_UIContainer remote_ui_container;
      
      remote_ui_container = bonobo_control_get_remote_ui_container(control_instance->priv->bonobo_control);
      bonobo_ui_component_set_container(uic, remote_ui_container);
      bonobo_object_release_unref(remote_ui_container, NULL);
      
      bonobo_ui_component_freeze(uic, NULL);
      
      /* add menubar */
      if (control_instance->priv->menubar)
        {
          BonoboUIHandlerMenuItem* menu_list;

          bonobo_ui_handler_create_menubar(control_instance->priv->ui_handler);
          menu_list = bonobo_ui_handler_menu_parse_uiinfo_list_with_data(control_instance->priv->menubar,
                                                                         control_instance);
          bonobo_ui_handler_menu_add_list(control_instance->priv->ui_handler, "/", menu_list);
          bonobo_ui_handler_menu_free_list(menu_list);
        }
      
      /* add toolbar */
      if (control_instance->priv->toolbar)
        {
          BonoboUIHandlerToolbarItem* toolbar_list;

          bonobo_ui_handler_create_toolbar(control_instance->priv->ui_handler, "Toolbar");
          toolbar_list = bonobo_ui_handler_toolbar_parse_uiinfo_list_with_data(control_instance->priv->toolbar,
                                                                               control_instance);
          bonobo_ui_handler_toolbar_add_list(control_instance->priv->ui_handler, "/Toolbar", toolbar_list);
          bonobo_ui_handler_toolbar_free_list(toolbar_list);
        }
      
      bonobo_ui_component_thaw(uic, NULL);
    }
}

/**
 * gnome_db_control_instance_deactivate
 */
void
gnome_db_control_instance_deactivate (GnomeDbControlInstance *control_instance)
{
  BonoboUIComponent* uic;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  
  uic = bonobo_control_get_ui_component(control_instance->priv->bonobo_control);
  if (uic)
    {
      bonobo_ui_component_unset_container(uic);
    }
}

/**
 * gnome_db_control_instance_set_status
 */
void
gnome_db_control_instance_set_status (GnomeDbControlInstance *control_instance,
                                      const gchar *msg)
{
  BonoboUIComponent* uic;
  
  g_return_if_fail(GNOME_DB_CONTROL_INSTANCE(control_instance));
  
  uic = bonobo_control_get_ui_component(control_instance->priv->bonobo_control);
  if (uic) bonobo_ui_component_set_status(uic, msg, NULL);
}

/**
 * gnome_db_control_instance_get_user_data
 */
gpointer
gnome_db_control_instance_get_user_data (GnomeDbControlInstance *control_instance)
{
  g_return_val_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance), NULL);
  return control_instance->priv->user_data;
}

/**
 * gnome_db_control_instance_set_user_data
 */
void
gnome_db_control_instance_set_user_data (GnomeDbControlInstance *control_instance,
                                         gpointer user_data)
{
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  control_instance->priv->user_data = user_data;
}

/**
 * gnome_db_control_instance_get_widget
 */
GtkWidget *
gnome_db_control_instance_get_widget (GnomeDbControlInstance *control_instance)
{
  g_return_val_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance), NULL);
  return control_instance->priv->widget;
}

/**
 * gnome_db_control_instance_get_bonobo_control
 */
BonoboControl *
gnome_db_control_instance_get_bonobo_control (GnomeDbControlInstance *control_instance)
{
  g_return_val_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance), NULL);
  return control_instance->priv->bonobo_control;
}

