/* GNOME DB components libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome-db.h>
#include "bonobo-ui-compat.h"

struct _GnomeDbWindowPrivate
{
  BonoboUIContainer* ui_container;
  BonoboUIComponent* ui_component;
  GtkWidget*         status_bar;
  GtkWidget*         progress_bar;
};

enum
{
  GNOME_DB_WINDOW_CLOSE,
  GNOME_DB_WINDOW_LAST_SIGNAL
};
static gint db_window_signals[GNOME_DB_WINDOW_LAST_SIGNAL] = { 0, };

/*
 * GnomeDbWindow class interface
 */
static void
gnome_db_window_class_init (GnomeDbWindowClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);

  db_window_signals[GNOME_DB_WINDOW_CLOSE] = gtk_signal_new("close",
                                                            GTK_RUN_FIRST,
                                                            object_class->type,
                                                            GTK_SIGNAL_OFFSET(GnomeDbWindowClass, close),
                                                            gtk_signal_default_marshaller,
                                                            GTK_TYPE_NONE, 0);
  gtk_object_class_add_signals(object_class, db_window_signals, GNOME_DB_WINDOW_LAST_SIGNAL);
  
  klass->close = NULL; // FIXME: add default close handler
}

static void
gnome_db_window_init (GnomeDbWindow *window)
{
  window->priv = g_new0(GnomeDbWindowPrivate, 1);
  
  window->priv->ui_container = NULL;
  window->priv->ui_component = NULL;
  window->priv->progress_bar = NULL;
}

GtkType
gnome_db_window_get_type (void)
{
  static GtkType db_window_type = 0;

  if (!db_window_type)
    {
      GtkTypeInfo db_window_info =
      {
        "GnomeDbWindow",
        sizeof (GnomeDbWindow),
        sizeof (GnomeDbWindowClass),
        (GtkClassInitFunc) gnome_db_window_class_init,
        (GtkObjectInitFunc) gnome_db_window_init,
        (GtkArgSetFunc) NULL,
        (GtkArgSetFunc) NULL
      };
      db_window_type = gtk_type_unique(bonobo_window_get_type(), &db_window_info);
    }
  return db_window_type;
}

static void
gnome_db_window_destroy (GnomeDbWindow *window, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_WINDOW(window));
  
  bonobo_object_unref(BONOBO_OBJECT(window->priv->ui_container));
  bonobo_object_unref(BONOBO_OBJECT(window->priv->ui_component));
  g_free((gpointer) window->priv);
}

GtkWidget *
gnome_db_window_construct (GnomeDbWindow *window,
                           const gchar *name,
                           GnomeUIInfo menus[],
                           GnomeUIInfo toolbar[],
                           gpointer data)
{
  GnomeDbWindow*     window_instance;
  Bonobo_UIContainer corba_container;
  BonoboControl*     status_control;
  BonoboControl*     progress_control;
  BonoboUIHandler*   ui_handler;
  
  window_instance = GNOME_DB_WINDOW(bonobo_window_construct(BONOBO_WINDOW(window), name, name));
  if (!window_instance)
    {
      gtk_object_unref(GTK_OBJECT(window));
      return NULL;
    }

  window_instance->priv->ui_container = bonobo_ui_container_new();
  bonobo_ui_container_set_win(window_instance->priv->ui_container, BONOBO_WINDOW(window_instance));
  
  gtk_signal_connect(GTK_OBJECT(window_instance),
                     "delete_event",
                     GTK_SIGNAL_FUNC(gnome_db_window_close),
                     NULL);
  gtk_signal_connect(GTK_OBJECT(window_instance),
                     "destroy",
                     GTK_SIGNAL_FUNC(gnome_db_window_destroy),
                     NULL);

  window_instance->priv->ui_component = bonobo_ui_component_new_default();
  corba_container = bonobo_object_corba_objref(BONOBO_OBJECT(window_instance->priv->ui_container));
  bonobo_ui_component_set_container(window_instance->priv->ui_component, corba_container);
  
  ui_handler = bonobo_ui_handler_new_from_component(window_instance->priv->ui_component);
  bonobo_ui_component_freeze(window_instance->priv->ui_component, NULL);
  
  /* set menus */
  if (menus)
    {
      BonoboUIHandlerMenuItem* menu_list;
      
      bonobo_ui_handler_create_menubar(ui_handler);
      menu_list = bonobo_ui_handler_menu_parse_uiinfo_list_with_data(menus, data);
      bonobo_ui_handler_menu_add_list(ui_handler, "/", menu_list);
      bonobo_ui_handler_menu_free_list(menu_list);
    }
  
  /* set toolbar */
  if (toolbar)
    {
      BonoboUIHandlerToolbarItem* toolbar_list;

      bonobo_ui_handler_create_toolbar(ui_handler, "Toolbar");
      toolbar_list = bonobo_ui_handler_toolbar_parse_uiinfo_list_with_data(toolbar, window_instance);
      bonobo_ui_handler_toolbar_add_list(ui_handler, "/Toolbar", toolbar_list);
      bonobo_ui_handler_toolbar_free_list(toolbar_list);
    }

  /* set status bar */
  window_instance->priv->status_bar = gnome_db_new_status_bar_widget();
  status_control = bonobo_control_new(window_instance->priv->status_bar);
  if (BONOBO_IS_CONTROL(status_control))
    {
      //bonobo_ui_component_object_set(window_instance->priv->ui_component,
      //                               "/status/main",
      //                               bonobo_object_corba_objref(BONOBO_OBJECT(status_control)),
      //                               NULL);
    }
  window_instance->priv->progress_bar = gnome_db_new_progress_bar_widget();
  progress_control = bonobo_control_new(window_instance->priv->progress_bar);
  if (BONOBO_IS_CONTROL(progress_control))
    {
      //bonobo_ui_component_object_set(window_instance->priv->ui_component,
      //                               "/status/Progress",
      //                               bonobo_object_corba_objref(BONOBO_OBJECT(progress_control)),
      //                               NULL);
    }
    
  bonobo_ui_component_thaw(window_instance->priv->ui_component, NULL);
  return window_instance;
}


/**
 * gnome_db_window_new
 * @name: name of the window 
 * @menus: menu bar items
 * @verbs: UI verbs handlers
 * @data: data to be passed to callbacks
 * 
 * Create a new GnomeDbWindow widget, which is a convenience widget which
 * manages a typical application window, with its menus and toolbar. Apart
 * from this, it implements a full-features Bonobo container, thus letting
 * you easily add Bonobo support to your applications.
 */
GtkWidget *
gnome_db_window_new (const gchar *name, GnomeUIInfo menus[], GnomeUIInfo toolbar[], gpointer data)
{
  GnomeDbWindow*     window;

  /* create the window */
  window = GNOME_DB_WINDOW(gtk_type_new(gnome_db_window_get_type()));

  return gnome_db_window_construct(window, name, menus, toolbar, data);
}

/**
 * gnome_db_window_close
 */
void
gnome_db_window_close (GnomeDbWindow *window)
{
  gtk_signal_emit(GTK_OBJECT(window), db_window_signals[GNOME_DB_WINDOW_CLOSE]);
  gtk_widget_destroy(GTK_WIDGET(window));
}

/**
 * gnome_db_window_show
 */
void
gnome_db_window_show (GnomeDbWindow *window)
{
  g_return_if_fail(GNOME_DB_IS_WINDOW(window));
  gtk_widget_show_all(GTK_WIDGET(window));
}

/**
 * gnome_db_window_get_contents
 */
GtkWidget *
gnome_db_window_get_contents (GnomeDbWindow *window)
{
  g_return_val_if_fail(GNOME_DB_IS_WINDOW(window), NULL);
  return bonobo_window_get_contents(BONOBO_WINDOW(window));
}

/**
 * gnome_db_window_set_contents
 */
void
gnome_db_window_set_contents (GnomeDbWindow *window, GtkWidget *contents)
{
  g_return_if_fail(GNOME_DB_IS_WINDOW(window));
  g_return_if_fail(GTK_IS_WIDGET(contents));

  bonobo_window_set_contents(BONOBO_WINDOW(window), contents);
}

/**
 * gnome_db_window_set_status
 */
void
gnome_db_window_set_status (GnomeDbWindow *window, const gchar *status)
{
  g_return_if_fail(GNOME_DB_IS_WINDOW(window));
  g_return_if_fail(BONOBO_IS_UI_COMPONENT(window->priv->ui_component));
  bonobo_ui_component_set_status(window->priv->ui_component, status, NULL);
}

/**
 * gnome_db_window_get_ui_container
 */
Bonobo_UIContainer
gnome_db_window_get_ui_container (GnomeDbWindow *window)
{
  g_return_val_if_fail(GNOME_DB_IS_WINDOW(window), NULL);
  
  return bonobo_ui_component_get_container(window->priv->ui_component);
}

