/* GNOME DB libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include "gnome-db-report.h"

static void gnome_db_report_class_init (GnomeDbReportClass *);
static void gnome_db_report_init       (GnomeDbReport *);
static void gnome_db_report_destroy    (GnomeDbReport *);

/*
 * GnomeDbReport widget signals
 */
enum
{
  GNOME_DB_REPORT_FILE_LOADED,
  GNOME_DB_REPORT_FILENAME_CHANGED,
  GNOME_DB_REPORT_LAST_SIGNAL
};

static gint gnome_db_report_signals[GNOME_DB_REPORT_LAST_SIGNAL] = { 0, };

/*
 * Private functions
 */
static void
gnome_db_report_destroy (GnomeDbReport *rep)
{
  g_return_if_fail(GNOME_DB_IS_REPORT(rep));
  
  if (rep->xmldoc != NULL) xmlFreeDoc(rep->xmldoc);
  if (rep->file_name != NULL) g_free((gpointer) rep->file_name);
}

/*
 * GnomeDbReport interface
 */
static void
gnome_db_report_class_init (GnomeDbReportClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  
  gnome_db_report_signals[GNOME_DB_REPORT_FILE_LOADED] =
           gtk_signal_new("file_loaded", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbReportClass, file_loaded),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);
  gnome_db_report_signals[GNOME_DB_REPORT_FILENAME_CHANGED] =
           gtk_signal_new("filename_changed", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbReportClass, filename_changed),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, gnome_db_report_signals,
                               GNOME_DB_REPORT_LAST_SIGNAL);
                               
  klass->file_loaded = NULL;
  klass->filename_changed = NULL;
  object_class->destroy = gnome_db_report_destroy;
}

static void
gnome_db_report_init (GnomeDbReport *rep)
{
  rep->xmldoc = xmlNewDoc("1.0");
  rep->file_name = NULL;
}

guint
gnome_db_report_get_type (void)
{
  static guint db_report_type = 0;

  if (!db_report_type)
    {
      GtkTypeInfo db_report_info =
      {
        "GnomeDbReport",
        sizeof (GnomeDbReport),
        sizeof (GnomeDbReportClass),
        (GtkClassInitFunc) gnome_db_report_class_init,
        (GtkObjectInitFunc) gnome_db_report_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_report_type = gtk_type_unique(gtk_vbox_get_type(), &db_report_info);
    }
  return (db_report_type);
}

/**
 * gnome_db_report_new
 *
 * Create a new GnomeDbReport widget, which can be used both to dynamically
 * design a report based on data from a GDA provider, and to be run displaying
 * the output to the user
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_report_new (void)
{
  GtkWidget *rep = gtk_type_new(gnome_db_report_get_type());
  return (rep);
}

/**
 * gnome_db_report_new_from_file
 * @file_name: file to load the report from
 *
 * Create a new GnomeDbReport widget from an existing report definition
 * contained in the @file_name file
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_report_new_from_file (const gchar *file_name)
{
  GnomeDbReport *rep;
  g_return_val_if_fail(file_name != NULL, NULL);
  rep = GNOME_DB_REPORT(gnome_db_report_new());
  gnome_db_report_load(rep, file_name);
  return (GTK_WIDGET(rep));
}

/**
 * gnome_db_report_clear
 * @rep: the GnomeDbReport widget
 *
 * Clears all data associated to the given GnomeDbReport widget
 */
void
gnome_db_report_clear (GnomeDbReport *rep)
{
  g_return_if_fail(GNOME_DB_IS_REPORT(rep));
  
  if (rep->xmldoc)
    {
      xmlFreeDoc(rep->xmldoc);
      rep->xmldoc = xmlNewDoc("1.0");
    }
  if (rep->file_name)
    {
      g_free((gpointer) rep->file_name);
      rep->file_name = NULL;
    }
}

/**
 * gnome_db_report_load
 * @rep: the GnomeDbReport widget
 * @file_name: file to load the report from
 *
 * Load an existing report definition into the given GnomeDbReport
 * widget
 *
 * Returns: TRUE on success, FALSE on error
 */
gboolean
gnome_db_report_load (GnomeDbReport *rep, const gchar *file_name)
{
  xmlDocPtr xmldoc;
  
  g_return_val_if_fail(GNOME_DB_IS_REPORT(rep), FALSE);
  g_return_val_if_fail(file_name != NULL, FALSE);
  
  xmldoc = xmlParseFile(file_name);
  if (xmldoc)
    {
      gnome_db_report_clear(rep);
      rep->xmldoc = xmldoc;
      gnome_db_report_set_file_name(rep, file_name);
      gtk_signal_emit_by_name(GTK_OBJECT(rep), "file_loaded", NULL);
      return (TRUE);
    }
  return (FALSE);
}

/**
 * gnome_db_report_save
 * @rep: the GnomeDbReport widget
 *
 * Save the given report to a disk file
 *
 *  Returns: TRUE on success, FALSE on error
 */
gboolean
gnome_db_report_save (GnomeDbReport *rep)
{
  g_return_val_if_fail(GNOME_DB_IS_REPORT(rep), FALSE);
  g_return_val_if_fail(rep->file_name != NULL, FALSE);
  
  return (!xmlSaveFile(rep->file_name, rep->xmldoc));
}

/**
 * gnome_db_report_set_file_name
 * @rep: the GnomeDbReport widget
 * @file_name: file name
 *
 * Sets the file name to be used when calling gnome_db_report_save
 */
void
gnome_db_report_set_file_name (GnomeDbReport *rep, const gchar *file_name)
{
  g_return_if_fail(GNOME_DB_IS_REPORT(rep));
  g_return_if_fail(file_name != NULL);

  if (rep->file_name != NULL)
    g_free((gpointer) rep->file_name);
  rep->file_name = g_strdup(file_name);
  
  gtk_signal_emit_by_name(GTK_OBJECT(rep), "filename_changed", NULL);
}
