/* GNOME DB libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <gnome-db-help.h>
#if defined(HAVE_GTKHTML)
#  include <gtkhtml/gtkhtml.h>
#endif

static void load_url (GnomeDbHelp *help, const gchar *url);

/*
 * Callbacks
 */
static void
menu_item_selected_cb (GtkMenuItem *menu_item, GnomeDbHelp *help)
{
  gchar* url;
  
  g_return_if_fail(GTK_IS_MENU_ITEM(menu_item));
  g_return_if_fail(GNOME_DB_IS_HELP(help));
  
  url = (gchar *) gtk_object_get_data(GTK_OBJECT(menu_item), "GNOME_DB_Help_MenuItemLabel");
  if (url)
    {
      load_url(help, url);
    }
}

#if defined(HAVE_GTKHTML)
static void
link_clicked_cb (GtkHTML *html, const gchar *url, GnomeDbHelp *help)
{
  g_print("utl = %s\n", url);
}
#endif

/*
 * Private functions
 */
static void
install_doc (GnomeDbHelp *help, const gchar *app_name, const gchar *file, const gchar *title)
{
  GnomePixmap* pixmap;
  gchar*       url;
  gchar*       row[2] = { "", "" };
  GtkWidget*   menu;
  
  g_return_if_fail(GNOME_DB_IS_HELP(help));
  g_return_if_fail(GTK_IS_OPTION_MENU(help->index_list));
  g_return_if_fail(title != NULL);
  
  menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(help->index_list));
  g_return_if_fail(GTK_IS_MENU(menu));
  
  pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_BOOK_RED);
  if ((url = gnome_help_file_find_file(app_name, file)))
    {
      GtkWidget* menu_item;
      
      menu_item = gtk_menu_item_new_with_label(title);
      gtk_object_set_data_full(GTK_OBJECT(menu_item),
                               "GNOME_DB_Help_MenuItemLabel",
                               (gpointer) g_strdup(url),
                               (GtkDestroyNotify) g_free);
      gtk_signal_connect(GTK_OBJECT(menu_item),
                         "activate",
                         GTK_SIGNAL_FUNC(menu_item_selected_cb),
                         (gpointer) help);
      gtk_widget_show(menu_item);
      gtk_menu_append(GTK_MENU(menu), menu_item);
    }
}

static void
fill_table_of_contents (GnomeDbHelp *help)
{
  GtkWidget* menu;

  g_return_if_fail(GNOME_DB_IS_HELP(help));
  g_return_if_fail(GTK_IS_OPTION_MENU(help->index_list));
  
  gtk_option_menu_remove_menu(GTK_OPTION_MENU(help->index_list));
  menu = gtk_menu_new();
  gtk_option_menu_set_menu(GTK_OPTION_MENU(help->index_list), menu);
  
  /* search for installed documentation */
  install_doc(help, "gnome-db", "index.html", _("GNOME-DB Manual"));
  install_doc(help, "libgda", "index.html", _("Libgda Manual"));  
}

static void
load_url (GnomeDbHelp *help, const gchar *url)
{
#if defined(HAVE_GTKHTML)
  GtkHTMLStream* handle;
#endif
  gchar*         contents;
  
  g_return_if_fail(GNOME_DB_IS_HELP(help));
  g_return_if_fail(url != NULL);
  
#if defined(HAVE_GTKHTML)
  handle = gtk_html_begin(GTK_HTML(help->html_viewer));
#endif
  contents = gnome_db_util_load_file(url);
  if (contents)
    {
#if defined(HAVE_GTKHTML)
      gtk_html_write(GTK_HTML(help->html_viewer), handle, contents, strlen(contents));
#endif
      g_free((gpointer) contents);
    }
#if defined(HAVE_GTKHTML)
  gtk_html_end(GTK_HTML(help->html_viewer), handle, GTK_HTML_STREAM_OK);
#endif
}

/*
 * Public functions
 */
static void
gnome_db_help_destroy (GnomeDbHelp *help)
{
}

static void
gnome_db_help_class_init (GnomeDbHelpClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);
}

static void
gnome_db_help_init (GnomeDbHelp *help)
{
  GtkWidget* table;
  GtkWidget* scroll;
  GtkWidget* frame;
  
  gtk_signal_connect(GTK_OBJECT(help),
                     "destroy",
                     GTK_SIGNAL_FUNC(gnome_db_help_destroy),
                     NULL);

  table = gnome_db_new_table_widget(1, 2, FALSE);
  gtk_box_pack_start(GTK_BOX(help), table, 1, 1, 0);
  
  /* add table of contents */
  help->index_list = gnome_db_new_option_menu_widget();
  gtk_table_attach(GTK_TABLE(table), help->index_list, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  fill_table_of_contents(help);
  
  /* add HTML viewer */
  frame = gnome_db_new_frame_widget(NULL);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 1, 1, 2,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   3, 3);
  scroll = gnome_db_new_scrolled_window_widget();
  gtk_container_add(GTK_CONTAINER(frame), scroll);
#if defined(HAVE_GTKHTML)
  help->html_viewer = gtk_html_new();
  //gtk_html_set_editable(GTK_HTML(help->html_viewer), FALSE);
  gtk_html_allow_selection(GTK_HTML(help->html_viewer), TRUE);
  gtk_widget_show(help->html_viewer);
  gtk_container_add(GTK_CONTAINER(scroll), help->html_viewer);
  gtk_signal_connect(GTK_OBJECT(help->html_viewer),
                     "link_clicked",
                     GTK_SIGNAL_FUNC(link_clicked_cb),
                     (gpointer) help);
#endif
}

GtkType
gnome_db_help_get_type (void)
{
  static guint db_help_type = 0;

  if (!db_help_type)
    {
      GtkTypeInfo db_help_info =
      {
        "GnomeDbHelp",
        sizeof (GnomeDbHelp),
        sizeof (GnomeDbHelpClass),
        (GtkClassInitFunc) gnome_db_help_class_init,
        (GtkObjectInitFunc) gnome_db_help_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_help_type = gtk_type_unique(gtk_vbox_get_type(), &db_help_info);
    }
  return (db_help_type);
}

/**
 * gnome_db_help_new
 */
GtkWidget *
gnome_db_help_new (void)
{
  GtkWidget* help;
  
  help = GTK_WIDGET(gtk_type_new(gnome_db_help_get_type()));
  return help;
}

/**
 * gnome_db_help_show_url
 */
void
gnome_db_help_show_url (GnomeDbHelp *help, const gchar *url)
{
  g_return_if_fail(GNOME_DB_IS_HELP(help));
  g_return_if_fail(url != NULL);
}

/**
 * gnome_db_help_go_home
 */
void
gnome_db_help_go_home (GnomeDbHelp *help)
{
}

/**
 * gnome_db_help_go_back
 */
void
gnome_db_help_go_back (GnomeDbHelp *help)
{
  g_return_if_fail(GNOME_DB_IS_HELP(help));
}

/**
 * gnome_db_help_go_forward
 */
void
gnome_db_help_go_forward (GnomeDbHelp *help)
{
}
