/*  GNOME-DB Components
 *  Copyrigth (C) 2000 Rodrigo Moya
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gnomedb-com.h>
#if defined(HAVE_GAL)
#  include <gal/e-paned/e-vpaned.h>
#endif

#define CONNECTION_INSTANCE_DATA "ConnectionInstanceData"

typedef struct
{
  GnomeDbControlInstance* control_instance;
  GtkWidget*              tab;
  GtkWidget*              cnc_selector;
  GtkWidget*              notebook;
  GList*                  open_cncs;
} connection_control_data_t;

typedef struct
{
  connection_control_data_t* control_data;
  Gda_Connection*            cnc;
  GtkWidget*                 tab;
  GtkWidget*                 login_dialog;
  gchar*                     gda_dsn;
  
  /* SQL tab */
  GtkWidget*                 sql_tab;
  GtkWidget*                 sql_cmd;
  GtkWidget*                 sql_grid;
  
  /* browser tab */
  GtkWidget*                 browser_tab;
} connection_instance_data_t;

static void activate_connection_cb (GtkMenuItem *menu_item, connection_instance_data_t *instance_data);

static void activate_browser_cb           (GtkWidget *w, GnomeDbControlInstance *control_instance);
static void activate_sql_cb               (GtkWidget *w, GnomeDbControlInstance *control_instance);
static void disconnect_cb                 (GtkWidget *w, GnomeDbControlInstance *control_instance);
static void open_connection_cb            (GtkWidget *w, GnomeDbControlInstance *control_instance);
static void show_connection_properties_cb (GtkWidget *w, GnomeDbControlInstance *control_instance);

static void do_print_cb (GnomeDbControlInstance *control_instance, connection_control_data_t *instance_data);

static void sql_clear_cb     (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_copy_cb      (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_cut_cb       (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_open_file_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_paste_cb     (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_run_cb       (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_save_file_cb (GtkWidget *w, connection_instance_data_t *instance_data);
static void sql_select_cb    (GtkWidget *w, connection_instance_data_t *instance_data);

static GnomeDbControl* connection_control = NULL;

GnomeUIInfo connection_control_toolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Connect"), N_("Open new database connection"),
    open_connection_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CONVERT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Disconnect"), N_("Close current connection"),
    disconnect_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_STOP, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("SQL"), N_("Activate SQL window"),
    activate_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Browser"), N_("Activate database browser window"),
    activate_browser_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PROPERTIES, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Properties"), N_("View current connection properties"),
    show_connection_properties_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PREFERENCES, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
fill_connection_list (connection_control_data_t *control_data)
{
  GtkWidget* menu;
  GList*     node;
  
  g_return_if_fail(control_data != NULL);
  g_return_if_fail(GTK_IS_OPTION_MENU(control_data->cnc_selector));
  
  gtk_option_menu_remove_menu(GTK_OPTION_MENU(control_data->cnc_selector));
  menu = gtk_menu_new();
  for (node = g_list_first(control_data->open_cncs); node != NULL; node = g_list_next(node))
    {
      connection_instance_data_t* instance_data = (connection_instance_data_t *) node->data;
      if (instance_data != NULL)
        {
          GtkWidget* menu_item;
          gchar* label;
          
          label = g_strdup_printf("%s@%s",
                                  gda_connection_get_user(instance_data->cnc),
                                  instance_data->gda_dsn);
          menu_item = gtk_menu_item_new_with_label(label);
          g_free((gpointer) label);
          gtk_signal_connect(GTK_OBJECT(menu_item),
                             "activate",
                             GTK_SIGNAL_FUNC(activate_connection_cb),
                             (gpointer) instance_data);
          gtk_menu_append(GTK_MENU(menu), menu_item);
        }
    }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(control_data->cnc_selector), menu);
}

static connection_instance_data_t *
get_current_connection_from_control_instance (GnomeDbControlInstance *control_instance)
{
  connection_control_data_t* control_data;
  
  g_return_val_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance), NULL);
  
  control_data = gnome_db_control_instance_get_user_data(control_instance);
  if (control_data)
    {
      gint       page_num;
      GtkWidget* w;
      
      if (GTK_IS_NOTEBOOK(control_data->notebook))
        {
          page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(control_data->notebook));
          w = gtk_notebook_get_nth_page(GTK_NOTEBOOK(control_data->notebook), page_num);
          if (GTK_IS_WIDGET(w))
            return gtk_object_get_data(GTK_OBJECT(w), CONNECTION_INSTANCE_DATA);
        }
    }
  return NULL;
}

static void
init_browser_window (connection_instance_data_t *instance_data)
{
  g_return_if_fail(instance_data != NULL);
  
  instance_data->browser_tab = gnome_db_browser_new(instance_data->cnc);
  gtk_widget_show(instance_data->browser_tab);
  gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->tab),
                           instance_data->browser_tab,
                           gtk_label_new(_("Browser")));
}

static GnomeUIInfo sql_toolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Run"), N_("Execute current command"),
    sql_run_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Select"), N_("Select command from previous"),
    sql_select_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_UP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Open SQL File"),
    sql_open_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save command to file"),
    sql_save_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text"),
    sql_cut_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text"),
    sql_copy_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste text from clipboard"),
    sql_paste_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PASTE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear widget contents"),
    sql_clear_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  GNOMEUIINFO_END
};

static void
init_sql_window (connection_instance_data_t *instance_data)
{
  GtkWidget* toolbar;
  GtkWidget* scroll;
  GtkWidget* paned;
  
  g_return_if_fail(instance_data != NULL);
  
  instance_data->sql_tab = gnome_db_new_table_widget(1, 2, FALSE);
  toolbar = gnome_db_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
                                        GTK_TOOLBAR_ICONS,
                                        sql_toolbar,
                                        (gpointer) instance_data);
  gtk_table_attach(GTK_TABLE(instance_data->sql_tab), toolbar, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

#if defined(HAVE_GAL)
  paned = e_vpaned_new();
#else
  paned = gtk_vpaned_new();
#endif
  gtk_widget_show(paned);
  gtk_table_attach(GTK_TABLE(instance_data->sql_tab), paned, 0, 1, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);

  scroll = gnome_db_new_scrolled_window_widget();
#if defined(HAVE_GAL)
  e_paned_add1(E_PANED(paned), scroll);
#else
  gtk_paned_add1(GTK_PANED(paned), scroll);
#endif
  instance_data->sql_cmd = gnome_db_new_text_widget();
  gtk_container_add(GTK_CONTAINER(scroll), instance_data->sql_cmd);
  
  instance_data->sql_grid = gnome_db_new_grid_widget(NULL);
#if defined(HAVE_GAL)
  e_paned_add2(E_PANED(paned), instance_data->sql_grid);
#else
  gtk_paned_add2(GTK_PANED(paned), instance_data->sql_grid);
#endif
  gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->tab),
                           instance_data->sql_tab,
                           gtk_label_new(_("SQL")));
}

/*
 * Callbacks
 */
static void
activate_browser_cb (GtkWidget *w, GnomeDbControlInstance *control_instance)
{
  connection_instance_data_t* instance_data;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  
  instance_data = get_current_connection_from_control_instance(control_instance);
  if (instance_data)
    {
      gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->tab),
                            gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->tab),
                                                  instance_data->browser_tab));
    }
}

static void
activate_sql_cb (GtkWidget *w, GnomeDbControlInstance *control_instance)
{
  connection_instance_data_t* instance_data;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  
  instance_data = get_current_connection_from_control_instance(control_instance);
  if (instance_data)
    {
      gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->tab),
                            gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->tab),
                                                  instance_data->sql_tab));
    }
}

static void
activate_connection_cb (GtkMenuItem *menu_item, connection_instance_data_t *instance_data)
{
  g_return_if_fail(instance_data != NULL);
  g_return_if_fail(instance_data->control_data != NULL);
  
  gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->control_data->notebook),
                        gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->control_data->notebook),
                                              instance_data->tab));
}

static void
cancel_connecting_cb (GnomeDbLoginDlg *dlg, connection_instance_data_t *instance_data)
{
  g_return_if_fail(instance_data != NULL);
  
  gnome_dialog_close(GNOME_DIALOG(instance_data->login_dialog));
  gda_connection_free(instance_data->cnc);
  g_free((gpointer) instance_data);
}

static GnomeDbControlInstance*
connection_create_instance_cb (GnomeDbControl *control)
{
  GtkWidget*              tab;
  GnomeDbControlInstance* control_instance;
  
  g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), NULL);
  
  tab = gnome_db_new_table_widget(4, 2, FALSE);
  control_instance = gnome_db_control_instance_new(control, tab);
  if (GNOME_DB_IS_CONTROL_INSTANCE(control_instance))
    {
      connection_control_data_t* control_data;
      
      control_data = g_new0(connection_control_data_t, 1);
      control_data->control_instance = control_instance;
      control_data->tab = tab;
      
      control_data->cnc_selector = gnome_db_new_option_menu_widget();
      gtk_table_attach(GTK_TABLE(control_data->tab), control_data->cnc_selector, 0, 1, 0, 1,
                       GTK_FILL, GTK_FILL, 3, 3);

      /* create notebook */
      control_data->notebook = gnome_db_new_notebook_widget();
      gtk_notebook_set_tab_pos(GTK_NOTEBOOK(control_data->notebook), GTK_POS_BOTTOM);
      gtk_table_attach(GTK_TABLE(control_data->tab), control_data->notebook, 0, 4, 1, 2,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       3, 3);

      gnome_db_control_instance_set_toolbar(control_data->control_instance, connection_control_toolbar);
      gnome_db_control_instance_set_user_data(control_data->control_instance, control_data);
      
      gtk_signal_connect(GTK_OBJECT(control_data->control_instance),
                         "do_print",
                         GTK_SIGNAL_FUNC(do_print_cb),
                         (gpointer) control_data);
    }
  else gtk_object_destroy(GTK_OBJECT(tab));
  return control_instance;
}

static void
connection_closed_cb (Gda_Connection *cnc, connection_instance_data_t *instance_data)
{
  g_return_if_fail(instance_data != NULL);
  
  gda_connection_free(instance_data->cnc);
  if (instance_data->control_data)
    {
      instance_data->control_data->open_cncs = g_list_remove(instance_data->control_data->open_cncs,
                                                             (gpointer) instance_data);
      gtk_notebook_remove_page(GTK_NOTEBOOK(instance_data->control_data->notebook),
                               gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->control_data->notebook),
                                                     instance_data->tab));
      fill_connection_list(instance_data->control_data);
    }
  g_free((gpointer) instance_data->gda_dsn);
  g_free((gpointer) instance_data);
}

static void
connection_initiated_cb (Gda_Connection *cnc, connection_instance_data_t *instance_data)
{
  gchar*        label;
  gchar*        str;
  GnomeDbLogin* login;
  
  g_return_if_fail(gda_connection_is_open(cnc));
  g_return_if_fail(instance_data != NULL);
  
  /* get login dialog information */
  login = GNOME_DB_LOGIN(GNOME_DB_LOGINDLG(instance_data->login_dialog)->login);
  if (login)
    str = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(login->gda_dbname)->entry));
  else str = NULL;
  gnome_dialog_close(GNOME_DIALOG(instance_data->login_dialog));
  instance_data->login_dialog = NULL;
  instance_data->gda_dsn = g_strdup(str);
  
  /* create tab widget (a GtkNotebook) */
  instance_data->tab = gnome_db_new_notebook_widget();
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(instance_data->tab), FALSE);
  gtk_notebook_set_show_border(GTK_NOTEBOOK(instance_data->tab), FALSE);
  gtk_notebook_popup_disable(GTK_NOTEBOOK(instance_data->tab));

  init_sql_window(instance_data);
  init_browser_window(instance_data);
  
  /* add this instance to list of open connections */
  if (instance_data->control_data)
    {
      gint page_num;

      instance_data->control_data->open_cncs = g_list_append(instance_data->control_data->open_cncs,
                                                             (gpointer) instance_data);
      label = g_strdup_printf("%s@%s", gda_connection_get_user(instance_data->cnc), str);
      gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->control_data->notebook),
                               instance_data->tab,
                               gtk_label_new(label));
      g_free((gpointer) label);

      gtk_object_set_data(GTK_OBJECT(instance_data->tab), CONNECTION_INSTANCE_DATA, instance_data);
      page_num = gtk_notebook_page_num(GTK_NOTEBOOK(instance_data->control_data->notebook),
                                       instance_data->tab);
      gtk_notebook_set_page(GTK_NOTEBOOK(instance_data->control_data->notebook), page_num);
      fill_connection_list(instance_data->control_data);
    }
}

static void
disconnect_cb (GtkWidget *w, GnomeDbControlInstance *control_instance)
{
  connection_instance_data_t* instance_data;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  
  instance_data = get_current_connection_from_control_instance(control_instance);
  if (instance_data)
    {
      /* just _close the connection, since we've got connection_closed_cb
         responding to the "close" signal, which is emitted when
         the connection is closed */
      gda_connection_close(instance_data->cnc);
    }
}

static
void do_print_cb (GnomeDbControlInstance *control_instance, connection_control_data_t *instance_data)
{
  GtkWidget* dialog;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  g_return_if_fail(instance_data != NULL);
  
  dialog = gnome_dialog_new(_("Do Print"), GNOME_STOCK_BUTTON_CLOSE, NULL);
  gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
}

static void
error_in_connection_cb (Gda_Connection *cnc, connection_instance_data_t *instance_data)
{
  GtkWidget* error_dialog;
  
  error_dialog = gnome_db_errordlg_new(cnc, _("Error Viewer"));
  gnome_db_errordlg_show_errors(GNOME_DB_ERRORDLG(error_dialog));
}

static void
open_connection_cb (GtkWidget *w, GnomeDbControlInstance *control_instance)
{
  connection_instance_data_t* instance_data;
  GnomeDbLogin*               login;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  
  instance_data = g_new0(connection_instance_data_t, 1);
  instance_data->control_data = gnome_db_control_instance_get_user_data(control_instance);
  
  instance_data->cnc = gda_connection_new(gda_corba_get_orb());
  gtk_signal_connect(GTK_OBJECT(instance_data->cnc),
                     "error",
                     GTK_SIGNAL_FUNC(error_in_connection_cb),
                     (gpointer) instance_data);
  gtk_signal_connect(GTK_OBJECT(instance_data->cnc),
                     "open",
                     GTK_SIGNAL_FUNC(connection_initiated_cb),
                     (gpointer) instance_data);
  gtk_signal_connect(GTK_OBJECT(instance_data->cnc),
                     "close",
                     GTK_SIGNAL_FUNC(connection_closed_cb),
                     (gpointer) instance_data);
  
  /* create login dialog box */
  login = gnome_db_login_new(instance_data->cnc, NULL, NULL);
  instance_data->login_dialog = gnome_db_logindlg_new(login, _("Open Connection"));
  gnome_dialog_close_hides(GNOME_DIALOG(instance_data->login_dialog), TRUE);
  gtk_signal_connect(GTK_OBJECT(instance_data->login_dialog),
                     "cancel",
                     GTK_SIGNAL_FUNC(cancel_connecting_cb),
                     (gpointer) instance_data);
  
  /* run dialog */
  while (gtk_events_pending()) gtk_main_iteration();
  gnome_db_logindlg_popup(GNOME_DB_LOGINDLG(instance_data->login_dialog));
}

static void
show_connection_properties_cb (GtkWidget *w, GnomeDbControlInstance *control_instance)
{
  connection_instance_data_t* instance_data;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_INSTANCE(control_instance));
  
  instance_data = get_current_connection_from_control_instance(control_instance);
  if (instance_data)
    {
      GtkWidget* dialog;
      GtkWidget* frame;
      GtkWidget* table;
      GtkWidget* label;
      GtkWidget* entry;
      Gda_Dsn*   dsn_info;
      
      dialog = gnome_dialog_new(_("Connection Properties"),
                                GNOME_STOCK_BUTTON_CLOSE,
                                NULL);
      frame = gnome_db_new_frame_widget(NULL);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
      table = gnome_db_new_table_widget(3, 6, FALSE);
      gtk_container_add(GTK_CONTAINER(frame), table);
      
      label = gnome_db_new_label_widget(_("GDA Name"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      entry = gnome_db_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), instance_data->gda_dsn);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      
      dsn_info = gda_dsn_find_by_name(instance_data->gda_dsn);
      label = gnome_db_new_label_widget(_("Provider"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
      entry = gnome_db_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), GDA_DSN_PROVIDER(dsn_info));
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

      label = gnome_db_new_label_widget(_("User name"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
      entry = gnome_db_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_user(instance_data->cnc));
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
      
      label = gnome_db_new_label_widget(_("Version"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
      entry = gnome_db_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_version(instance_data->cnc));
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
      
      gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
    }
}

static void
sql_clear_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
  g_return_if_fail(instance_data != NULL);
  g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
  
  gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
  gnome_db_grid_set_recordset(GNOME_DB_GRID(instance_data->sql_grid), NULL);
}

static void
sql_copy_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
  g_return_if_fail(instance_data != NULL);
  g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
  gtk_editable_copy_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
}

static void
sql_cut_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
  g_return_if_fail(instance_data != NULL);
  g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
  gtk_editable_cut_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
}

static void
sql_open_file_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
  gchar* filename;
  gchar* text;
  
  g_return_if_fail(instance_data != NULL);
  g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
  
  filename = gnome_db_select_file(_("Open SQL file"));
  if (filename)
    {
      text = gnome_db_util_load_file(filename);
      if (text)
        {
          gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
          gtk_text_insert(GTK_TEXT(instance_data->sql_cmd), NULL, NULL, NULL, text, strlen(text));
          g_free((gpointer) text);
        }
      else gnome_db_show_error(_("Error loading file %s"), filename);
      g_free((gpointer) filename);
    }
}

static void
sql_paste_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
  g_return_if_fail(instance_data != NULL);
  g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
  gtk_editable_paste_clipboard(GTK_EDITABLE(instance_data->sql_cmd));
}

static void
sql_run_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
  gchar* sql;
  
  g_return_if_fail(instance_data != NULL);
  g_return_if_fail(GTK_IS_TEXT(instance_data->sql_cmd));
  
  sql = gtk_editable_get_chars(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
  if (sql)
    {
      gulong         reccount;
      Gda_Recordset* recset;
      
      /* execute command */
      recset = gda_connection_execute(instance_data->cnc, sql, &reccount, 0);
      if (recset)
        {
          /* FIXME: notify user of success in container's status bar */
        }
      gnome_db_grid_set_recordset(GNOME_DB_GRID(instance_data->sql_grid), recset);
      components_config_add_last_command((const gchar *) sql);
    }
}

static void
sql_save_file_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
  gchar* text;
  gchar* filename;
  
  g_return_if_fail(instance_data != NULL);
  g_return_if_fail(GTK_IS_EDITABLE(instance_data->sql_cmd));
  
  filename = gnome_db_select_file(_("Save SQL file"));
  if (filename)
    {
      text = gtk_editable_get_chars(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
      if (text)
        {
          if (!gnome_db_util_save_file(filename, text))
            gnome_db_show_error(_("Could not save file %s"), filename);
        }
      g_free((gpointer) filename);
    }
}

static void
sql_select_cb (GtkWidget *w, connection_instance_data_t *instance_data)
{
  GtkWidget* dialog;
  GtkWidget* frame;
  GtkWidget* scroll;
  GtkWidget* list;
  GList*     commands;

  g_return_if_fail(instance_data != NULL);

  /* create the dialog */
  dialog = gnome_dialog_new(_("Select SQL Command"),
                            GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL,
                            NULL);
  gtk_widget_set_usize(dialog, 350, 250);
  frame = gnome_db_new_frame_widget(NULL);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
  scroll = gnome_db_new_scrolled_window_widget();
  gtk_container_add(GTK_CONTAINER(frame), scroll);

  list = gnome_db_new_clist_widget(NULL, 2);
  commands = components_config_get_last_commands();
  if (commands)
    {
      GList* node;
      gint   cnt;

      gtk_clist_freeze(GTK_CLIST(list));
      for (cnt = 1, node = g_list_first(commands); node; node = g_list_next(node), cnt++)
        {
          gchar* row[2];

          row[0] = g_strdup_printf("%d", cnt);
          row[1] = (gchar *) node->data;
          gtk_clist_append(GTK_CLIST(list), row);
        }
      gtk_clist_thaw(GTK_CLIST(list));
    }
  gtk_container_add(GTK_CONTAINER(scroll), list);

  /* run the dialog */
  if (!gnome_dialog_run(GNOME_DIALOG(dialog)))
    {
      gchar* sql = NULL;
      GList* selection;

      /* retrieve CList selection */
      selection = GTK_CLIST(list)->selection;
      if (selection)
        {
          gint row = GPOINTER_TO_INT(selection->data);
          if (gtk_clist_get_text(GTK_CLIST(list), row, 1, &sql))
            {
              gtk_editable_delete_text(GTK_EDITABLE(instance_data->sql_cmd), 0, -1);
              gtk_text_insert(GTK_TEXT(instance_data->sql_cmd), NULL, NULL, NULL, sql, strlen(sql));
            }
        }
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

/*
 * Public functions
 */
void
components_connection_init (void)
{
  if (!GNOME_DB_IS_CONTROL(connection_control))
    {
      gchar* oafiid;

      oafiid = g_strdup("OAFIID:control-factory:gnome-db-connection:c6967e5e-035e-48ff-9983-87966d81a0e2");
      
      /* register factory */
      connection_control = gnome_db_control_new(oafiid, connection_create_instance_cb);
      if (!GNOME_DB_IS_CONTROL(connection_control))
        g_error(_("Could not initialize %s component"), oafiid);
      g_free((gpointer) oafiid);
    }
}
