/* Sample Rolodex Application
 * Copyright (C) 1998,1999 Michael Lausch
 * Copyright (C) 2000 Alvaro del Castillo 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-rolodex.h"
#include "gda-form.h"
#include "gda-rolodex-nav.h"

static guint prepare_app (void);


void Exception( CORBA_Environment *ev )
{
  switch( ev->_major )
    {
    case CORBA_SYSTEM_EXCEPTION:
      fprintf( stderr, "CORBA system exception %s.\n",
	       CORBA_exception_id(ev));
      exit ( 1 );
    case CORBA_USER_EXCEPTION:
      fprintf( stderr, "CORBA user exception: %s.\n",
	       CORBA_exception_id( ev ) );
      exit ( 1 );
    default:
      break;
    }
}

static void
next_record(GtkWidget* button, gpointer data)
{
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);
  float    value;
  
  value = adj->value;
  value++;
  gtk_adjustment_set_value(adj, value);
}

static void
prev_record(GtkWidget* button, gpointer data)
{
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);
  float    value;

  value = adj->value;
  value--;
  gtk_adjustment_set_value(adj, value);
}

static void
first_record(GtkWidget* button, gpointer data)
{
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);
  
  gtk_adjustment_set_value(adj, 1);
}

static void
last_record(GtkWidget* button, gpointer data)
{
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);
  
  gtk_adjustment_set_value(adj, G_MAXFLOAT);
  g_print ("In last record: %f\n",adj->value);
}



static void
rolodex_update_status_bar_cb (GtkObject* obj, gpointer data)
{
  gchar    bfr[128];
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);

  g_snprintf(bfr, sizeof(bfr), _("Record %8.0f"), adj->value);
  gnome_db_container_set_status (GNOME_DB_CONTAINER(glb_container), bfr);
  gnome_db_container_set_progress (GNOME_DB_CONTAINER(glb_container), adj->value/adj->upper);
}

/*
 * Global variables
 */
GtkWidget* 		glb_container = NULL;
GladeXML*  		xml = NULL;
Gda_Connection*    	cnc = NULL;
CORBA_ORB 		orb = NULL;

gint
main(int argc, char* argv[])
{
  CORBA_Environment ev;
  GnomeClient *client;

  /* initialize the i18n stuff */
  bindtextdomain(PACKAGE, GNOMELOCALEDIR);
  textdomain(PACKAGE);

  /* initialize CORBA/bonobo */
  CORBA_exception_init(&ev);
  orb = gnome_CORBA_init("gda-rolodex-app", VERSION, &argc, argv, 0, &ev);
  CORBA_exception_free(&ev);

  if (bonobo_init(orb, NULL, NULL) == FALSE)
    g_error(_("Could not initialize Bonobo\n"));

  /* Session handling. It is in alpha stage. Only for testing. */
  client = gnome_master_client ();
  gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
                      GTK_SIGNAL_FUNC (rolodex_save_session), argv[0]);
  gtk_signal_connect (GTK_OBJECT (client), "die",
                      GTK_SIGNAL_FUNC (rolodex_session_die), NULL);

  /*
   * We can't make any CORBA calls unless we're in the main
   * loop. So we delay creating the container here.
   */
  gtk_idle_add((GtkFunction) prepare_app, NULL);
  bonobo_main();

  return 0;
}

static guint 
prepare_app(void)
{
  /* Glade */
  glade_gnome_init ();
  /* FIX ME: glade_bonobo_init (); */

  cnc = gda_connection_new(orb);
  gtk_signal_connect(GTK_OBJECT(cnc),"error", GTK_SIGNAL_FUNC(rolodex_error_cb), NULL);
  login_db (cnc);
  return(0);
}

void
login_db (Gda_Connection* cnc) {
  GtkWidget *login;
  GnomeDbLogin *login_aux;

  /* This would be better GDA_IS_CONNECTION */
  g_return_if_fail(IS_GDA_CONNECTION(cnc));

  login_aux=GNOME_DB_LOGIN(gnome_db_login_new(cnc, NULL, NULL));
  login =  gnome_db_logindlg_new(login_aux,"Connection to the database");
  /* Signals that we can receive from the gnome_db_login widget */
  gtk_signal_connect(GTK_OBJECT(cnc),
                         "error",
                         GTK_SIGNAL_FUNC(rolodex_error_cb),
                         NULL);
  gtk_signal_connect_after(GTK_OBJECT(cnc),
                         "open",
                         GTK_SIGNAL_FUNC(rolodex_login_cb),
                         login);
  gtk_signal_connect(GTK_OBJECT(login),
                         "cancel",
                         GTK_SIGNAL_FUNC(rolodex_login_cancel_cb),
                         NULL);
  /* We show the login dialog */
  gnome_db_logindlg_popup(GNOME_DB_LOGINDLG(login));
}

/* This is the principal function. When you have connected to the
database, and selected a table to work with, then you call this
function and start to work with the records of the table */

void show_data(Gda_Connection* cnc, gchar *tablename) {
  GtkWidget*  form;
  gchar  cmd_bfr[1024];
  GtkWidget* hbox;
  GtkWidget* vbox;
  GtkWidget* next;
  GtkWidget* prev;
  GtkWidget* first;
  GtkWidget* last;
  GtkWidget* top_hbox;
  GtkWidget* rnav;
  GtkObject* grs;

  g_return_if_fail(IS_GDA_CONNECTION(cnc));

  grs = gda_recordset_new();
  gda_recordset_set_connection(GDA_RECORDSET(grs), cnc);
  gda_recordset_set_cursorloc(GDA_RECORDSET(grs), GDA_USE_CLIENT_CURSOR);
 
  sprintf(cmd_bfr, "select * from %s", tablename);

  gda_recordset_open_txt(GDA_RECORDSET(grs), cmd_bfr, GDA_OPEN_STATIC, GDA_LOCK_PESSIMISTIC, 0);
  
  hbox = gtk_hbox_new(1,1);
  vbox = gtk_vbox_new(0,1);
  form = gda_form_new();
  gtk_signal_connect(GTK_OBJECT(form),
                     "data_changed",
                     GTK_SIGNAL_FUNC(rolodex_update_status_bar_cb),
                     grs);

  gtk_box_pack_start(GTK_BOX(vbox), form, 1, 1, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, 0, 1, 0);
  next = gtk_button_new_with_label("Next");
  prev = gtk_button_new_with_label("Prev");
  first = gtk_button_new_with_label("First");
  last = gtk_button_new_with_label("Last");
  
  gtk_signal_connect(GTK_OBJECT(next),
		     "clicked",
		     GTK_SIGNAL_FUNC(next_record),
		     grs);

  gtk_signal_connect(GTK_OBJECT(prev),
		     "clicked",
		     GTK_SIGNAL_FUNC(prev_record),
		     grs);

  gtk_signal_connect(GTK_OBJECT(first),
		     "clicked",
		     GTK_SIGNAL_FUNC(first_record),
		     grs);

  gtk_signal_connect(GTK_OBJECT(last),
		     "clicked",
		     GTK_SIGNAL_FUNC(last_record),
		     grs);

  gtk_box_pack_start(GTK_BOX(hbox), first, 1, 1, 0);
  gtk_box_pack_start(GTK_BOX(hbox), prev, 1, 1, 0);
  gtk_box_pack_start(GTK_BOX(hbox), next, 1, 1, 0);
  gtk_box_pack_start(GTK_BOX(hbox), last, 1, 1, 0);
  
  gda_form_set_datasource(GDA_FORM(form), GDA_RECORDSET(grs));

  top_hbox = gtk_hbox_new(0, 0);
  rnav = gda_rolodex_nav_new(GTK_ADJUSTMENT(grs));
  
  gtk_box_pack_start(GTK_BOX(top_hbox), rnav, 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(top_hbox), vbox, 0, 0, 0);

  gnome_db_container_set_contents(glb_container,top_hbox);
  gnome_db_container_show(glb_container); 
  /* Cool: each second one record is shown automatic */
  gtk_timeout_add(1000, rolodex_change_record_cb, grs);
}

void
show_tables (Gda_Connection *cnc)
{
  Gda_Recordset *recset;
  GtkWidget *gnome_app, *menubar, *toolbar, *gnome_app_bar;
  GtkWidget *tablas, *label, *box;
  GladeXML *xml_gnome_app;

  g_return_if_fail(IS_GDA_CONNECTION(cnc));

  xml_gnome_app = glade_xml_new (GLADEDIR "/" ROLODEX_GLADE_FILE, "app1");
  if (!xml_gnome_app)
    {
      g_error (_("We could not load the glade interface description! (%s)"),
               GLADEDIR "/" ROLODEX_GLADE_FILE, "app1");
      return NULL;
    }

  glade_xml_signal_autoconnect (xml_gnome_app);
  gnome_app = glade_xml_get_widget (xml_gnome_app, "app1");
  menubar = glade_xml_get_widget (xml_gnome_app, "menubar1");
  toolbar = glade_xml_get_widget (xml_gnome_app, "toolbar1");
  gnome_app_bar = glade_xml_get_widget (xml_gnome_app, "appbar1");


  glb_container = gnome_db_container_new_glade(GNOME_APP(gnome_app), GTK_MENU_BAR(menubar), GTK_TOOLBAR(toolbar), GNOME_APPBAR(gnome_app_bar), NULL);
  gtk_signal_connect(GTK_OBJECT(glb_container),
                     "close",
                     GTK_SIGNAL_FUNC(rolodex_quit_cb),
                     NULL);

  recset = gda_connection_open_schema(cnc,
                                      GDA_Connection_GDCN_SCHEMA_TABLES,
                                      GDA_Connection_no_CONSTRAINT, 0);
  if (recset != 0) tablas=gnome_db_list_new (recset,0);
  /* FIXME: not elegant */
  else return;
  /* Events for the gnome_db_list */
  gtk_signal_connect(GTK_OBJECT(tablas),
                     "select_row",
                     GTK_SIGNAL_FUNC(rolodex_table_selected_cb),
                     NULL);
  /* Elements to do the layout */
  box = gtk_hbox_new (FALSE, 0);
  label = gtk_label_new(_("Select the table name"));
  gtk_box_pack_start (GTK_BOX (box),
                              label, FALSE, FALSE, 3);
  gtk_box_pack_start (GTK_BOX (box),
                              tablas, TRUE, TRUE, 3);

  gnome_db_container_set_contents(glb_container,box);
  gtk_widget_show(box);
  gtk_widget_show(label);
  gtk_widget_show(tablas);
  gnome_db_container_show(GNOME_DB_CONTAINER(glb_container));
}

/* the gnome session manager may call this function */
void
rolodex_session_die (GnomeClient * client, gpointer client_data)
{
  g_message ("in die");
  gtk_main_quit ();
}                               /* session_die */

/* the gnome session manager may call this function */
gint
rolodex_save_session (GnomeClient * client, gint phase, GnomeSaveStyle save_style,
              gint is_shutdown, GnomeInteractStyle interact_style,
              gint is_fast, gpointer client_data)
{
  gchar **argv;
  guint argc;

  /* allocate 0-filled, so it will be NULL-terminated */
  argv = g_malloc0 (sizeof (gchar *) * 4);
  argc = 1;

  argv[0] = client_data;

  g_message ("In save_session");
#if 0
  if (message)
    {
      argv[1] = "--message";
      argv[2] = message;
      argc = 3;
    }
#endif

  gnome_client_set_clone_command (client, argc, argv);
  gnome_client_set_restart_command (client, argc, argv);

  return TRUE;
}
