/* GNOME DB Postgres Config Component
 * Copyright (C) 2000 Rodrigo Moya
 * Copyright (C) 2000 Vivien Malerba
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-postgres-config-ac.h"

/*
 * Actual work of building the widget in the component. The info
 * struct will be filled at that time
 */
GtkWidget* create_user_ac_config(PostgresPropsInfo *info);


/* to set/reset the contents of the widget */
static void users_ac_start_reset(PostgresPropsInfo *info); 


/* 
 * Prototypes for the CB functions of that component
 */
static void users_ac_clist_sel_cb(GtkWidget *wid, gint row, gint column, 
				  GdkEventButton *event, PostgresPropsInfo *info);
static void users_option_menu_cb(GtkWidget *wid, PostgresPropsInfo *info);
static void users_groups_changed_cb(GtkWidget *mitem, PostgresPropsInfo *info);
static void users_ac_start_reset_rights(PostgresPropsInfo *info);
static void rights_toggled_cb_cb(GtkWidget *wid, PostgresPropsInfo *info);
static void refresh_component(PostgresPropsInfo *info);


/*
 * Implementation 
 */
GtkWidget * create_users_ac_widget_cb (void)
{
  GtkWidget *vb;
  GtkWidget *w;
  PostgresPropsInfo *info;

  vb = gtk_vbox_new(FALSE, GNOME_PAD);

  info = g_new0(PostgresPropsInfo, 1);
  info->ident = TRUE; /* default */
  gtk_object_set_data(GTK_OBJECT(vb), "propinfo", info);
  info->refresh_component = refresh_component;

  /* declaring the widget for updates */
  controls_list = g_slist_append(controls_list, info);

  w = create_user_id_config(info);
  gtk_widget_show(w);
  gtk_box_pack_start(GTK_BOX(vb), w, FALSE, FALSE, 0);
  info->ident_widget = w;

  w = create_user_ac_config(info);
  gtk_widget_show(w);
  gtk_box_pack_start(GTK_BOX(vb), w, TRUE, TRUE, 0);

  g_print("New users access  widget %p (info at %p)\n", vb, info);
  gtk_signal_connect(GTK_OBJECT(vb), "destroy",
		     GTK_SIGNAL_FUNC(component_widget_destroy_cb), NULL);
  return vb;
}

/*
 * Function to set the properties at the creation of the bonobo component
 */
void  users_ac_new_instance_cb(GnomeDbControl *control,
			       GnomeDbControlInstance *control_instance,
			       gpointer data)
{
  PostgresPropsInfo *info;

  /* fetching the info */
  info = gtk_object_get_data(GTK_OBJECT(control_instance->widget), "propinfo");
  /* info is the control instance's data */
  gnome_db_control_set_instance_data(control_instance, info);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_DSN,
				_("DSN for the connection"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_IDENT_NAME,
				_("User identity"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_IDENT_PASS,
				_("User passord"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_IDENT_SHOW,
				_("Show/hide the user id"),
				GNOME_DB_CONTROL_ARG_BOOLEAN, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_REFRESH, 
				_("Give it TRUE to reset/start the component"),
				GNOME_DB_CONTROL_ARG_BOOLEAN, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_USER_NAME, 
				_("User name setting"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_GROUP_NAME,
				_("Group name setting"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_OBJECT_TYPE,
				_("Object type setting"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_OBJECT_NAME, 
				_("Object name setting"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
}


/* 
 * Actual creation of the widgets 
 */
static GtkWidget *create_rights_enum(GtkWidget **sel, GtkWidget **up,
				     GtkWidget **ins, GtkWidget **rule,
				     GtkWidget **selection, gchar *sel_lab);
GtkWidget* create_user_ac_config(PostgresPropsInfo *info)
{
  GtkWidget *frame, *frame2, *vb, *wid, *clist, *sw, *hb, *menu, *label;

  frame = gtk_frame_new(_("Access rigths"));
  
  hb = gtk_hbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(hb), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), hb);

  /* vbox on the left */
  vb = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_box_pack_start(GTK_BOX(hb), vb, TRUE, TRUE, 0);

  wid = gtk_option_menu_new();
  gtk_box_pack_start(GTK_BOX(vb), wid, FALSE, FALSE, 0);
  info->u.users_ac.choice = wid;
  
  menu = gtk_menu_new();
  label = gtk_menu_item_new_with_label(_("Tables"));
  gtk_widget_show(label);
  gtk_signal_connect(GTK_OBJECT(label), "activate",
		     GTK_SIGNAL_FUNC(users_option_menu_cb), info);
  gtk_object_set_data(GTK_OBJECT(label),"order", GINT_TO_POINTER(0));
  gtk_menu_append(GTK_MENU(menu), label);
  label = gtk_menu_item_new_with_label(_("Views"));
  gtk_widget_show(label);
  gtk_signal_connect(GTK_OBJECT(label), "activate",
		     GTK_SIGNAL_FUNC(users_option_menu_cb), info);
  gtk_object_set_data(GTK_OBJECT(label),"order", GINT_TO_POINTER(1));
  gtk_menu_append(GTK_MENU(menu), label);
  label = gtk_menu_item_new_with_label(_("Sequences"));
  gtk_widget_show(label);
  gtk_signal_connect(GTK_OBJECT(label), "activate",
		     GTK_SIGNAL_FUNC(users_option_menu_cb), info);
  gtk_object_set_data(GTK_OBJECT(label),"order", GINT_TO_POINTER(2));
  gtk_menu_append(GTK_MENU(menu), label);
  label = gtk_menu_item_new_with_label(_("Indexes"));
  gtk_widget_show(label);
  gtk_signal_connect(GTK_OBJECT(label), "activate",
		     GTK_SIGNAL_FUNC(users_option_menu_cb), info);
  gtk_object_set_data(GTK_OBJECT(label),"order", GINT_TO_POINTER(3));
  gtk_menu_append(GTK_MENU(menu), label);
  gtk_option_menu_set_menu(GTK_OPTION_MENU(wid), menu);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), 
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(vb), sw, TRUE, TRUE, 0);


  clist = gtk_clist_new(2);
  gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
  gtk_container_add(GTK_CONTAINER(sw), clist);
  gtk_clist_set_sort_column(GTK_CLIST(clist), 0);
  gtk_clist_set_auto_sort(GTK_CLIST(clist), TRUE);
  gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		     GTK_SIGNAL_FUNC(users_ac_clist_sel_cb), info);
  info->u.users_ac.clist = clist;
  gtk_object_set_data(GTK_OBJECT(info->u.users_ac.clist), "selrow", 
		      GINT_TO_POINTER(-1));


  /* right part with the frames */
  vb = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_box_pack_start(GTK_BOX(hb), vb, FALSE, FALSE, 0);

  frame2 = gtk_frame_new(_("All users rights"));
  gtk_box_pack_start(GTK_BOX(vb), frame2, TRUE, FALSE, 0);

  wid = create_rights_enum(&(info->u.users_ac.pub_sel), 
			   &(info->u.users_ac.pub_up),
			   &(info->u.users_ac.pub_ins),
			   &(info->u.users_ac.pub_rule), NULL, NULL);
  gtk_container_add(GTK_CONTAINER(frame2), wid);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.pub_sel), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.pub_up), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.pub_ins), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.pub_rule), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);

  frame2 = gtk_frame_new(_("Groups rights"));
  gtk_box_pack_start(GTK_BOX(vb), frame2, TRUE, FALSE, 0);

  wid = create_rights_enum(&(info->u.users_ac.group_sel), 
			   &(info->u.users_ac.group_up),
			   &(info->u.users_ac.group_ins),
			   &(info->u.users_ac.group_rule),
			   &(info->u.users_ac.group_choice), _("Group:"));
  gtk_container_add(GTK_CONTAINER(frame2), wid);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.group_sel), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.group_up), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.group_ins), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.group_rule), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);

  frame2 = gtk_frame_new(_("Individual users rights"));
  gtk_box_pack_start(GTK_BOX(vb), frame2, TRUE, FALSE, 0);

  wid = create_rights_enum(&(info->u.users_ac.user_sel), 
			   &(info->u.users_ac.user_up),
			   &(info->u.users_ac.user_ins),
			   &(info->u.users_ac.user_rule),
			   &(info->u.users_ac.user_choice), _("User:"));
  gtk_container_add(GTK_CONTAINER(frame2), wid);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.user_sel), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.user_up), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.user_ins), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);
  gtk_signal_connect(GTK_OBJECT(info->u.users_ac.user_rule), "toggled",
		     GTK_SIGNAL_FUNC(rights_toggled_cb_cb), info);

  gtk_widget_show_all(hb);

  /* other settings */
  info->ident = TRUE;
  info->running = FALSE;
  info->dsn = NULL;

  return frame;
}

static GtkWidget *create_rights_enum(GtkWidget **sel, GtkWidget **up,
				     GtkWidget **ins, GtkWidget **rule,
				     GtkWidget **selection, gchar *sel_lab)
{
  GtkWidget *table, *wid;
  gint os = 0;

  if (selection)
    {
      table = gtk_table_new(3, 4, FALSE);
      os = 1; /* offset for the table rows */
      wid = gtk_label_new(sel_lab);
      gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, 0, 1);
      wid = gtk_option_menu_new();
      *selection = wid;
      gtk_table_attach_defaults(GTK_TABLE(table), wid, 1, 4, 0, 1);
    }
  else
    table = gtk_table_new(2, 4, FALSE);
  
  gtk_table_set_row_spacings(GTK_TABLE(table), GNOME_PAD/2);
  gtk_table_set_col_spacings(GTK_TABLE(table), GNOME_PAD/2);
  gtk_container_set_border_width(GTK_CONTAINER(table), GNOME_PAD);
  wid = gtk_label_new(_("Select"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, os, os+1);
  wid = gtk_check_button_new();
  *sel = wid;
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 1, 2, os, os+1);

  wid = gtk_label_new(_("Insert"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 2, 3, os, os+1);
  wid = gtk_check_button_new();
  *ins = wid;
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 3, 4, os, os+1);

  wid = gtk_label_new(_("Update/Delete"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, os+1, os+2);
  wid = gtk_check_button_new();
  *up = wid;
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 1, 2, os+1, os+2);

  wid = gtk_label_new(_("Rule"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 2, 3, os+1, os+2);
  wid = gtk_check_button_new();
  *rule = wid;
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 3, 4, os+1, os+2);
  return table;
}

/* 
 * Get properties replacement function 
 */
void users_ac_get_property_cb(GnomeDbControl *control,
			      GnomeDbControlInstance *control_instance,
			      const gchar *name, 
			      gpointer data)
{
  PostgresPropsInfo *info;
  gchar *str;
  GtkWidget *menu, *mitem;
  gint sysid;
  PGresult *res;
  PGconn *conn;
  gboolean error=FALSE, done=FALSE;

  info = gnome_db_control_get_instance_data(control_instance);
  if (!strcmp(name, GNOME_DB_CONTROL_PROP_DSN))
    {
      if (info->dsn)
	gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_DSN, info->dsn);
      else
	gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_DSN, "");
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_NAME))
    {
      str = gtk_entry_get_text(GTK_ENTRY(info->user_name));
      gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_IDENT_NAME, str);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_PASS))
    {
      str = gtk_entry_get_text(GTK_ENTRY(info->user_passwd));
      gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_IDENT_PASS, str);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_SHOW))
    {
      gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_IDENT_SHOW, 
				    &(info->ident));
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_REFRESH))
    {
      gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_REFRESH, 
				    &(info->running));
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_USER_NAME))
    {
      conn = get_setup_conn(info);
      if (conn)
	{
	  menu = gtk_option_menu_get_menu
	    (GTK_OPTION_MENU(info->u.users_ac.user_choice));
	  if (menu)
	    {
	      mitem = gtk_menu_get_active(GTK_MENU(menu));
	      sysid = GPOINTER_TO_INT
		(gtk_object_get_data(GTK_OBJECT(mitem), "sysid"));
	      str = g_strdup_printf("SELECT usename FROM pg_user "
				    "where usesysid = %d", sysid);
	      res = PQexec(conn, str);
	      g_free(str);
	      if (is_pgresult_valid(res))
		{
		  if (PQntuples(res) > 0)
		    gnome_db_control_set_property(control_instance, 
						  GNOME_DB_CONTROL_PROP_USER_NAME, 
						  PQgetvalue(res, 0, 0));
		  else
		    error = TRUE;
		  PQclear(res);
		}
	      else
		error = TRUE;
	    }
	  else
	    error = TRUE;
	  PQfinish(conn);
	}
      else
	error = TRUE;
      if (error)
	gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_USER_NAME, "");
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_GROUP_NAME))
    {
      conn = get_setup_conn(info);
      if (conn)
	{
	  menu = gtk_option_menu_get_menu
	    (GTK_OPTION_MENU(info->u.users_ac.group_choice));
	  if (menu)
	    {
	      mitem = gtk_menu_get_active(GTK_MENU(menu));
	      sysid = GPOINTER_TO_INT
		(gtk_object_get_data(GTK_OBJECT(mitem), "sysid"));
	      str = g_strdup_printf("SELECT groname FROM pg_group "
				    "where grosysid = %d", sysid);
	      res = PQexec(conn, str);
	      g_free(str);
	      if (is_pgresult_valid(res))
		{
		  if (PQntuples(res) > 0)
		    gnome_db_control_set_property(control_instance, 
						  GNOME_DB_CONTROL_PROP_GROUP_NAME, 
						  PQgetvalue(res, 0, 0));
		  else
		    error = TRUE;
		  PQclear(res);
		}
	      else
		error = TRUE;
	    }
	  else
	    error = TRUE;
	  PQfinish(conn);
	}
      else
	error = TRUE;
      if (error)
	gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_GROUP_NAME, "");
      done = TRUE;
    }

  if (!strcmp(name, GNOME_DB_CONTROL_PROP_OBJECT_TYPE))
    {
      mitem = gtk_menu_get_active
	(GTK_MENU(gtk_option_menu_get_menu
		  (GTK_OPTION_MENU(info->u.users_ac.choice))));
      sysid = GPOINTER_TO_INT
	(gtk_object_get_data(GTK_OBJECT(mitem),"order"));
      switch(sysid)
	{
	case 0:
	  gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_OBJECT_TYPE, 
					"tables");
	  break;
	case 1:
	  gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_OBJECT_TYPE, 
					"views");
	  break;
	case 2:
	  gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_OBJECT_TYPE, 
					"sequences");
	  break;
	case 3:
	  gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_OBJECT_TYPE, 
					"indexes");
	  break;
	}
      done = TRUE;
    }

  if (!strcmp(name, GNOME_DB_CONTROL_PROP_OBJECT_NAME))
    {
      sysid = GPOINTER_TO_INT(gtk_object_get_data
			      (GTK_OBJECT(info->u.users_ac.clist), 
			       "selrow"));
      if (sysid >= 0)
	{
	  gtk_clist_get_text(GTK_CLIST(info->u.users_ac.clist),
			     sysid, 0, &str);
	  gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_OBJECT_NAME, str);
	}
      else
	  gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_OBJECT_NAME, "");
      done = TRUE;
    }
  
  if (!done)
    g_warning("Unknown property '%s'\n", name);
}


/* 
 * Set properties replacement function 
 */
void users_ac_set_property_cb(GnomeDbControl *control,
			      GnomeDbControlInstance *control_instance,
			      const gchar *name,
			      gconstpointer value,
			      gpointer data)
{
  PostgresPropsInfo *info;
  gchar *str;
  GtkWidget *menu, *mitem;
  gint sysid, i;
  PGresult *res;
  PGconn *conn;
  gboolean error=FALSE, done=FALSE;

  info = gnome_db_control_get_instance_data(control_instance);
  if (!strcmp(name, GNOME_DB_CONTROL_PROP_DSN))
    {
      if (info->dsn)
	g_free(info->dsn);
      info->dsn = g_strdup((gchar *) value);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_NAME))
    {
      gtk_entry_set_text(GTK_ENTRY(info->user_name), (gchar *) value);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_PASS))
    {
      gtk_entry_set_text(GTK_ENTRY(info->user_passwd), (gchar *) value);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_SHOW))
    {
      info->ident = *((gboolean *) value);

      if (info->ident)
	gtk_widget_show(info->ident_widget);
      else
	gtk_widget_hide(info->ident_widget);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_REFRESH))
    {
      if (*((gboolean *) value))
	{
	  refresh_component(info);
	  info->running = TRUE;
	}
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_USER_NAME))
    {
      conn = get_setup_conn(info);
      sysid = -1;
      if (conn)
	{
	  str = g_strdup_printf("SELECT usesysid FROM pg_user "
				"where usename = '%s'", (gchar *) value);
	  res = PQexec(conn, str);
	  g_free(str);
	  if (is_pgresult_valid(res))
	    {
	      if (PQntuples(res) > 0)
		sysid = atoi(PQgetvalue(res, 0, 0));
	      else
		error = TRUE;
	      PQclear(res);
	    }
	  else
	    error = TRUE;
	  PQfinish(conn);
	}
      else
	error = TRUE;
      
      if (!error && (sysid>-1))
	{
	  menu = gtk_option_menu_get_menu
	    (GTK_OPTION_MENU(info->u.users_ac.user_choice));	      
	  if (menu)
	    {
	      GList *list;
	      gint index = -1, i=0;
	      list = GTK_MENU_SHELL(menu)->children;
	      while (list && (index < 0))
		{
		  if (sysid == GPOINTER_TO_INT
		      (gtk_object_get_data(GTK_OBJECT(list->data), "sysid")))
		    index = i;
		  else
		    list = g_list_next(list);
		  i++;
		}
	      if (index >= 0)
		{
		  gtk_option_menu_set_history
		    (GTK_OPTION_MENU(info->u.users_ac.user_choice), index);
		  gtk_menu_item_activate(GTK_MENU_ITEM(list->data));
		}
	    }
	}
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_GROUP_NAME))
    {
      conn = get_setup_conn(info);
      sysid = -1;
      if (conn)
	{
	  str = g_strdup_printf("SELECT grosysid FROM pg_group "
				"where groname = '%s'", (gchar *) value);
	  res = PQexec(conn, str);
	  g_free(str);
	  if (is_pgresult_valid(res))
	    {
	      if (PQntuples(res) > 0)
		sysid = atoi(PQgetvalue(res, 0, 0));
	      else
		error = TRUE;
	      PQclear(res);
	    }
	  else
	    error = TRUE;
	  PQfinish(conn);
	}
      else
	error = TRUE;
      
      if (!error && (sysid>-1))
	{
	  menu = gtk_option_menu_get_menu
	    (GTK_OPTION_MENU(info->u.users_ac.group_choice));	      
	  if (menu)
	    {
	      GList *list;
	      gint index = -1, i=0;
	      list = GTK_MENU_SHELL(menu)->children;
	      while (list && (index < 0))
		{
		  if (sysid == GPOINTER_TO_INT
		      (gtk_object_get_data(GTK_OBJECT(list->data), "sysid")))
		    index = i;
		  else
		    list = g_list_next(list);
		  i++;
		}
	      if (index >= 0)
		{
		  gtk_option_menu_set_history
		    (GTK_OPTION_MENU(info->u.users_ac.group_choice), 
		     index);
		  gtk_menu_item_activate(GTK_MENU_ITEM(list->data));
		}
	    }
	}      
      done = TRUE;
    }

  if (!strcmp(name, GNOME_DB_CONTROL_PROP_OBJECT_TYPE))
    {
      str = (gchar *) value;
      sysid = -1;
      if (!strcmp(str, "tables")) sysid = 0;
      if ((sysid<0) && !strcmp(str, "views")) sysid = 1;
      if ((sysid<0) && !strcmp(str, "sequences")) sysid = 2;
      if ((sysid<0) && !strcmp(str, "indexes")) sysid = 3;
      if (sysid > -1)
	{
	  gtk_option_menu_set_history
	    (GTK_OPTION_MENU(info->u.users_ac.choice), 
	     sysid);
	  mitem = gtk_menu_get_active
	    (GTK_MENU(gtk_option_menu_get_menu
		      (GTK_OPTION_MENU(info->u.users_ac.choice))));
	  gtk_menu_item_activate(GTK_MENU_ITEM(mitem));
	}
      done = TRUE;
    }

  if (!strcmp(name, GNOME_DB_CONTROL_PROP_OBJECT_NAME))
    {
      sysid = -1;
      i = 0;
      while ((sysid < 0)  && (i<GTK_CLIST(info->u.users_ac.clist)->rows))
	{
	  gtk_clist_get_text(GTK_CLIST(info->u.users_ac.clist), i, 0, 
			     &str);
	  if (!strcmp(str, (gchar *) value))
	    sysid = i;
	  i++;
	}
      
      if (sysid > -1)
	gtk_clist_select_row(GTK_CLIST(info->u.users_ac.clist), sysid, 0);
      done = TRUE;
    }

  if (!done)
    g_warning("Unknown property '%s'\n", name);
}


/*
 * Refreshment function for the widget
 */
static void refresh_component(PostgresPropsInfo *info)
{
  users_ac_start_reset(info);
}

static void users_ac_start_reset_users_and_groups(PostgresPropsInfo *info)
{
  PGresult *res;
  PGconn *conn;
  gchar *query=NULL;
  GtkWidget *menu, *mitem;
  gint nb, i, hold, index;

  conn = get_setup_conn(info);
  if (conn)
    {
      /* groups */
      hold = -1; /* nothing on hold */
      if ((menu = gtk_option_menu_get_menu
	   (GTK_OPTION_MENU(info->u.users_ac.group_choice))))
	{
	  mitem = gtk_menu_get_active(GTK_MENU(menu));
	  hold = GPOINTER_TO_INT
	    (gtk_object_get_data(GTK_OBJECT(mitem), "sysid"));
	  gtk_option_menu_remove_menu
	    (GTK_OPTION_MENU(info->u.users_ac.group_choice));
	}
      menu = gtk_menu_new();
      query = "SELECT groname, grosysid FROM pg_group order by groname";
      res = PQexec(conn, query);
      index = -1;
      if (is_pgresult_valid(res))
	{
	  nb = PQntuples(res);
	  for (i=0; i<nb; i++)
	    {
	      mitem = gtk_menu_item_new_with_label(PQgetvalue(res, i, 0));
	      gtk_object_set_data(GTK_OBJECT(mitem), "sysid",
				  GINT_TO_POINTER(atoi(PQgetvalue(res, i, 1))));
	      if (atoi(PQgetvalue(res, i, 1)) == hold)
		index = i;
	      gtk_signal_connect(GTK_OBJECT(mitem), "activate",
				 GTK_SIGNAL_FUNC(users_groups_changed_cb),info);
	      gtk_menu_append(GTK_MENU(menu), mitem);
	      gtk_widget_show(mitem);
	    }
	  PQclear(res);
	}
      gtk_option_menu_set_menu
	(GTK_OPTION_MENU(info->u.users_ac.group_choice), menu);
      if (index >= 0)
	gtk_option_menu_set_history
	  (GTK_OPTION_MENU(info->u.users_ac.group_choice), index);

      /* individual users */
      hold = -1;
      if ((menu = gtk_option_menu_get_menu
	   (GTK_OPTION_MENU(info->u.users_ac.user_choice))))
	{	  
	  mitem = gtk_menu_get_active(GTK_MENU(menu));
	  hold = GPOINTER_TO_INT
	    (gtk_object_get_data(GTK_OBJECT(mitem), "sysid"));
	  gtk_option_menu_remove_menu
	    (GTK_OPTION_MENU(info->u.users_ac.user_choice));
	}
      menu = gtk_menu_new();
      query = "SELECT usename, usesysid FROM pg_user order by usename";
      res = PQexec(conn, query);
      index = -1;
      if (is_pgresult_valid(res))
	{
	  nb = PQntuples(res);
	  for (i=0; i<nb; i++)
	    {
	      mitem = gtk_menu_item_new_with_label(PQgetvalue(res, i, 0));
	      gtk_object_set_data(GTK_OBJECT(mitem), "sysid",
				  GINT_TO_POINTER(atoi(PQgetvalue(res, i, 1))));
	      if (atoi(PQgetvalue(res, i, 1)) == hold)
		index = i;
	      gtk_signal_connect(GTK_OBJECT(mitem), "activate",
				 GTK_SIGNAL_FUNC(users_groups_changed_cb),info);
	      gtk_menu_append(GTK_MENU(menu), mitem);
	      gtk_widget_show(mitem);
	    }
	  PQclear(res);
	}
      gtk_option_menu_set_menu
	(GTK_OPTION_MENU(info->u.users_ac.user_choice), menu);
      if (index >= 0)
	gtk_option_menu_set_history
	  (GTK_OPTION_MENU(info->u.users_ac.user_choice), index);

      PQfinish(conn);
    }  
  /* make sure that what is displayed is OK */
  users_ac_start_reset_rights(info);
}

static gboolean is_in(char *str, gchar c)
{
  gchar *ptr=str;
  gboolean retval=FALSE;
  while (*ptr && !retval)
    {
      if (*ptr == c) retval = TRUE;
      ptr++;
    }
  return retval;
}

static gint is_where(char *str, gchar c)
{
  gchar *ptr=str;
  gint retval=0, i=0;
  while (*ptr && !retval)
    {
      if (*ptr == c) retval = i;
      ptr++;
      i++;
    }
  return retval;
}

static void lock_toggles_signals(PostgresPropsInfo *info)
{
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.pub_sel), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.pub_up), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.pub_ins), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.pub_rule), info);

  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.group_sel), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.group_up), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.group_ins), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.group_rule), info);

  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.user_sel), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.user_up), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.user_ins), info);
  gtk_signal_handler_block_by_data(GTK_OBJECT(info->u.users_ac.user_rule), info);
}

static void unlock_toggles_signals(PostgresPropsInfo *info)
{
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.pub_sel), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.pub_up), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.pub_ins), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.pub_rule), info);

  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.group_sel), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.group_up), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.group_ins), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.group_rule), info);

  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.user_sel), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.user_up), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.user_ins), info);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(info->u.users_ac.user_rule), info);
}

static void clear_rights(PostgresPropsInfo *info)
{
  lock_toggles_signals(info);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.pub_sel),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.pub_ins),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.pub_up),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.pub_rule),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.group_sel),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.group_ins),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.group_up),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.group_rule),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.user_sel),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.user_ins),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.user_up),
			       FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->u.users_ac.user_rule),
			       FALSE);
  unlock_toggles_signals(info);
}

static void users_ac_start_reset_rights(PostgresPropsInfo *info)
{
  gint row = -1, order;
  gpointer prow;
  gchar *kind="'r'", *query, *name;
  PGresult *res;
  PGconn *conn;
  GtkWidget *menu, *mitem;

  prow = gtk_object_get_data(GTK_OBJECT(info->u.users_ac.clist), "selrow");
  row = GPOINTER_TO_INT(prow);
  if (row >= 0) /* set all the properties to what's in the DB */
    {
      /* getting the rights string like "=r","dummy=r","debs=rw" */
      menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(info->u.users_ac.choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "order");
      order = GPOINTER_TO_INT(prow);
      switch (order)
	{
	case 0:
	case 1:
	  kind = "'r'";
	  break;
	case 2:
	  kind = "'S'";
	  break;
	case 3:
	  kind = "'i'";
	  break;
	}
      gtk_clist_get_text(GTK_CLIST(info->u.users_ac.clist), row, 0, &name);
      query = g_strdup_printf("SELECT relacl FROM pg_class where relname='%s'"
			      " AND relkind = %s", name, kind);
      conn = get_setup_conn(info);
      if (conn)
	{
	  res = PQexec(conn, query);
	  if (is_pgresult_valid(res))
	    {
	      gchar *acl, *ptr, *ptr2;

	      clear_rights(info);
	      lock_toggles_signals(info);
	      acl = g_strdup(PQgetvalue(res, 0, 0));
	      PQclear(res);
	      if (*acl == '{') ptr2 = acl+1;
	      else ptr2 = acl;
	      if (*(acl+strlen(acl)-1) == '}')
		*(acl+strlen(acl)-1) = 0;
	      ptr = strtok(ptr2, ",");
	      while (ptr)
		{
		  if (*ptr == '"') ptr2 = ptr+1;
		  else ptr2 = ptr;
		  if (*(ptr+strlen(ptr)-1) == '"')
		    *(ptr+strlen(ptr)-1) = 0;
		  if (*ptr2 == '=')
		    { /* these are the rights of the PUBLIC (all users) */
		      gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON(info->u.users_ac.pub_sel), 
			 is_in(ptr2, 'r'));
		      gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON(info->u.users_ac.pub_ins), 
			 is_in(ptr2, 'a'));
		      gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON(info->u.users_ac.pub_up), 
			 is_in(ptr2, 'w'));
		      gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON(info->u.users_ac.pub_rule), 
			 is_in(ptr2, 'R'));
		    }
		  else
		    {
		      if (strstr(ptr2, "group ") == ptr2)
			{ /* this chunck is for a group */
			  gchar *str;

			  ptr2 += 6; /* strlen("group ") */
			  menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(info->u.users_ac.group_choice));
			  mitem = gtk_menu_get_active(GTK_MENU(menu));
			  prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
			  order = GPOINTER_TO_INT(prow);
			  str = g_strdup_printf("SELECT groname FROM pg_group "
						"where grosysid = %d", order);
			  res = PQexec(conn, str);
			  g_free(str);
			  if (is_pgresult_valid(res))
			    {
			      if ((PQntuples(res) > 0) &&
				  (strstr(ptr2, PQgetvalue(res, 0, 0)) == ptr2))
				{ /* the right group is selected here */
				  ptr = ptr2+is_where(ptr2, '=');
				  gtk_toggle_button_set_active
				    (GTK_TOGGLE_BUTTON(info->u.users_ac.group_sel),
				     is_in(ptr, 'r'));
				  gtk_toggle_button_set_active
				    (GTK_TOGGLE_BUTTON(info->u.users_ac.group_ins),
				     is_in(ptr, 'a'));
				  gtk_toggle_button_set_active
				    (GTK_TOGGLE_BUTTON(info->u.users_ac.group_up),
				     is_in(ptr, 'w'));
				  gtk_toggle_button_set_active
				    (GTK_TOGGLE_BUTTON(info->u.users_ac.group_rule),
				     is_in(ptr, 'R'));
				}
			      PQclear(res);
			    }
			}
		      else
			{ /* this chunck is for a user */
			  gchar *str;
			  menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(info->u.users_ac.user_choice));
			  mitem = gtk_menu_get_active(GTK_MENU(menu));
			  prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
			  order = GPOINTER_TO_INT(prow);
			  str = g_strdup_printf("SELECT usename FROM pg_user "
						"where usesysid = %d", order);
			  res = PQexec(conn, str);
			  g_free(str);
			  if (is_pgresult_valid(res))
			    {
			      if ((PQntuples(res) > 0) &&
				  (strstr(ptr2, PQgetvalue(res, 0, 0)) == ptr2))
				{ /* the right user is selected here */
				  ptr = ptr2+is_where(ptr2, '=');
				  gtk_toggle_button_set_active
				    (GTK_TOGGLE_BUTTON(info->u.users_ac.user_sel),
				     is_in(ptr, 'r'));
				  gtk_toggle_button_set_active
				    (GTK_TOGGLE_BUTTON(info->u.users_ac.user_ins),
				     is_in(ptr, 'a'));
				  gtk_toggle_button_set_active
				    (GTK_TOGGLE_BUTTON(info->u.users_ac.user_up),
				     is_in(ptr, 'w'));
				  gtk_toggle_button_set_active
				    (GTK_TOGGLE_BUTTON(info->u.users_ac.user_rule),
				     is_in(ptr, 'R'));
				}
			      PQclear(res);
			    }
			}
		    }
		  ptr = strtok(NULL, ",");
		}
	      g_free(acl);
	      unlock_toggles_signals(info);
	    }
	  else /* there is a problem somewhere => remove selection of clist */
	    {
	      gtk_clist_unselect_row(GTK_CLIST(info->u.users_ac.clist), row, 0);
	      gtk_object_set_data(GTK_OBJECT(info->u.users_ac.clist), "selrow", 
				  GINT_TO_POINTER(-1));	      
	      users_ac_start_reset_rights(info);
	    }
	  PQfinish(conn);
	}
      g_free(query);
    }
  else /* set all the check properties to FALSE */
    {
      clear_rights(info);
    }
}

static void users_groups_changed_cb(GtkWidget *mitem, PostgresPropsInfo *info)
{
  users_ac_start_reset_rights(info); 
}


static void users_ac_start_reset_clist(PostgresPropsInfo *info)
{
  PGresult *res;
  PGconn *conn;
  gchar *query=NULL, *text[2];
  gint nb, i, row;
  GtkWidget *menuitem;

  conn = get_setup_conn(info);
  if (conn)
    {
      menuitem = gtk_menu_get_active
	(GTK_MENU(gtk_option_menu_get_menu
		  (GTK_OPTION_MENU(info->u.users_ac.choice))));
      row = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(menuitem),"order"));
      switch (row)
	{
	case 0:
	  query = "SELECT tablename, tableowner FROM pg_tables WHERE "
	    "tablename !~ 'pg_' ORDER BY tablename";
	  gtk_clist_set_column_title(GTK_CLIST(info->u.users_ac.clist), 0,
				     _("Table"));
	  gtk_clist_set_column_title(GTK_CLIST(info->u.users_ac.clist), 1,
				     _("Owner"));
	  break;
	case 1:
	  query = "SELECT viewname, viewowner FROM pg_views "
	    "WHERE viewname !~ 'pg_' ORDER BY viewname";
	  gtk_clist_set_column_title(GTK_CLIST(info->u.users_ac.clist), 0,
				     _("View"));
	  gtk_clist_set_column_title(GTK_CLIST(info->u.users_ac.clist), 1,
				     _("Owner"));
	  break;
	case 2:
	  query = "SELECT c.relname, pg_get_userbyid(c.relowner) FROM "
	    "pg_class c where c.relkind = 'S' ORDER BY c.relname";
	  gtk_clist_set_column_title(GTK_CLIST(info->u.users_ac.clist), 0,
				     _("Sequence"));
	  gtk_clist_set_column_title(GTK_CLIST(info->u.users_ac.clist), 1,
				     _("Owner"));
	  break;
	case 3:
	  query = "SELECT indexname, tablename FROM pg_indexes "
	    "WHERE indexname !~ 'pg_' AND tablename !~ 'pg_' "
	    "ORDER BY indexname";
	  gtk_clist_set_column_title(GTK_CLIST(info->u.users_ac.clist), 0,
				     _("Index"));
	  gtk_clist_set_column_title(GTK_CLIST(info->u.users_ac.clist), 1,
				     _("Table"));
	  break;
	}
      gtk_clist_column_titles_show(GTK_CLIST(info->u.users_ac.clist));
      res = PQexec(conn, query);
      if (is_pgresult_valid(res))
	{
	  gint hold=-1;
	  gchar *htext=NULL;

	  hold = GPOINTER_TO_INT
	    (gtk_object_get_data(GTK_OBJECT(info->u.users_ac.clist), "selrow"));
	  if (hold >= 0)
	    {
	      gchar *str;
	      gtk_clist_get_text(GTK_CLIST(info->u.users_ac.clist), 
				 hold, 0, &str);
	      htext = g_strdup(str);
	    }
	  gtk_clist_freeze(GTK_CLIST(info->u.users_ac.clist));
	  gtk_clist_clear(GTK_CLIST(info->u.users_ac.clist));
	  gtk_object_set_data(GTK_OBJECT(info->u.users_ac.clist), "selrow", 
			      GINT_TO_POINTER(-1));
	  users_ac_start_reset_rights(info);
	  nb = PQntuples(res);
	  hold = -1;
	  for (i=0; i<nb; i++)
	    {
	      text[0] = PQgetvalue(res, i, 0);
	      text[1] = PQgetvalue(res, i, 1);
	      row = gtk_clist_append(GTK_CLIST(info->u.users_ac.clist), text);
	      if (htext && (hold<0) && (!strcmp(text[0], htext)))
		hold = i;
	    }
	  gtk_clist_thaw(GTK_CLIST(info->u.users_ac.clist));
	  if (hold>=0)
	    gtk_clist_select_row(GTK_CLIST(info->u.users_ac.clist), hold, 0);
	  if (htext)
	    g_free(htext);
	  PQclear(res);
	}
      PQfinish(conn);
    }
}

static void users_ac_start_reset(PostgresPropsInfo *info)
{
  users_ac_start_reset_users_and_groups(info);
  users_ac_start_reset_clist(info);
  users_ac_start_reset_rights(info);
}

void users_option_menu_cb(GtkWidget *wid, PostgresPropsInfo *info)
{
  users_ac_start_reset_clist(info);
}

void users_ac_clist_sel_cb(GtkWidget *wid, gint row, gint column, 
			     GdkEventButton *event, PostgresPropsInfo *info)
{
  /* setting the row selection */
  gtk_object_set_data(GTK_OBJECT(info->u.users_ac.clist), "selrow", 
		      GINT_TO_POINTER(row));

  /* refreshing the rights */
  users_ac_start_reset_rights(info);
}

static void rights_toggled_cb_cb(GtkWidget *wid, PostgresPropsInfo *info)
{
  gboolean grant=FALSE;
  gchar *who=NULL, *whores=NULL, *whoq=NULL, *what=NULL, *toobj;
  GtkWidget *menu, *mitem;
  PGresult *res;
  PGconn *conn;
  GString *query;
  gint row;
  gpointer prow;

  /* on which object ? */
  row = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(info->u.users_ac.clist),
					    "selrow"));
  if (row < 0) return; /* nothing to do */

  gtk_clist_get_text(GTK_CLIST(info->u.users_ac.clist), row, 0, &toobj);

  /* grant or revoke? */
  if (GTK_TOGGLE_BUTTON(wid)->active)
    {
      query = g_string_new("GRANT ");
      grant = TRUE;
    }
  else
    query = g_string_new("REVOKE ");
  
  /* what and who are ALWAYS filled */
  /* SELECT ? */
  if (wid == info->u.users_ac.pub_sel)
    {
      who = "PUBLIC";
      what = "SELECT";
    }
  if (wid == info->u.users_ac.group_sel)
    {
      menu = gtk_option_menu_get_menu
	(GTK_OPTION_MENU(info->u.users_ac.group_choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
      row = GPOINTER_TO_INT(prow);
      whoq = g_strdup_printf("SELECT groname FROM pg_group "
			    "where grosysid = %d", row);
      who = "GROUP";
      what = "SELECT";
    }
  if (wid == info->u.users_ac.user_sel)
    {
      menu = gtk_option_menu_get_menu
	(GTK_OPTION_MENU(info->u.users_ac.user_choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
      row = GPOINTER_TO_INT(prow);
      whoq = g_strdup_printf("SELECT usename FROM pg_user "
			    "where usesysid = %d", row);
      who = "";
      what = "SELECT";
    }
  /* INSERT ? */
  if (wid == info->u.users_ac.pub_ins)
    {
      who = "PUBLIC";
      what = "INSERT";
    }
  if (wid == info->u.users_ac.group_ins)
    {
      menu = gtk_option_menu_get_menu
	(GTK_OPTION_MENU(info->u.users_ac.group_choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
      row = GPOINTER_TO_INT(prow);
      whoq = g_strdup_printf("SELECT groname FROM pg_group "
			    "where grosysid = %d", row);
      who = "GROUP";
      what = "INSERT";
    }
  if (wid == info->u.users_ac.user_ins)
    {
      menu = gtk_option_menu_get_menu
	(GTK_OPTION_MENU(info->u.users_ac.user_choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
      row = GPOINTER_TO_INT(prow);
      whoq = g_strdup_printf("SELECT usename FROM pg_user "
			    "where usesysid = %d", row);
      who = "";
      what = "INSERT";
    }
  /* UPDATE ? */
  if (wid == info->u.users_ac.pub_up)
    {
      who = "PUBLIC";
      what = "UPDATE";
    }
  if (wid == info->u.users_ac.group_up)
    {
      menu = gtk_option_menu_get_menu
	(GTK_OPTION_MENU(info->u.users_ac.group_choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
      row = GPOINTER_TO_INT(prow);
      whoq = g_strdup_printf("SELECT groname FROM pg_group "
			    "where grosysid = %d", row);
      who = "GROUP";
      what = "UPDATE";
    }
  if (wid == info->u.users_ac.user_up)
    {
      menu = gtk_option_menu_get_menu
	(GTK_OPTION_MENU(info->u.users_ac.user_choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
      row = GPOINTER_TO_INT(prow);
      whoq = g_strdup_printf("SELECT usename FROM pg_user "
			    "where usesysid = %d", row);
      who = "";
      what = "UPDATE";
    }
  /* RULE ? */
  if (wid == info->u.users_ac.pub_rule)
    {
      who = "PUBLIC";
      what = "RULE";
    }
  if (wid == info->u.users_ac.group_rule)
    {
      menu = gtk_option_menu_get_menu
	(GTK_OPTION_MENU(info->u.users_ac.group_choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
      row = GPOINTER_TO_INT(prow);
      whoq = g_strdup_printf("SELECT groname FROM pg_group "
			    "where grosysid = %d", row);
      who = "GROUP";
      what = "RULE";
    }
  if (wid == info->u.users_ac.user_rule)
    {
      menu = gtk_option_menu_get_menu
	(GTK_OPTION_MENU(info->u.users_ac.user_choice));
      mitem = gtk_menu_get_active(GTK_MENU(menu));
      prow = gtk_object_get_data(GTK_OBJECT(mitem), "sysid");
      row = GPOINTER_TO_INT(prow);
      whoq = g_strdup_printf("SELECT usename FROM pg_user "
			    "where usesysid = %d", row);
      who = "";
      what = "RULE";
    }
  
  /* make the query */
  conn = get_setup_conn(info);
  if (conn)
    {
      if (whoq)
	{
	  res = PQexec(conn, whoq);
	  if (is_pgresult_valid(res))
	    {
	      if (PQntuples(res)>0)
		whores = g_strdup(PQgetvalue(res, 0, 0));
	      PQclear(res);
	    }
	  else /* error */
	    {
	      PQfinish(conn);
	      g_free(whoq);
	      users_ac_start_reset_rights(info);
	      return;
	    }
	}

      g_string_sprintfa(query, "%s ON %s", what, toobj);
      if (grant)
	g_string_append(query, " TO "); 
      else
	g_string_append(query, " FROM ");
      
      if (whores)
	g_string_sprintfa(query, "%s %s", who, whores);
      else
	g_string_sprintfa(query, "%s", who);
      g_print("rights -> %s\n", query->str);
      PQexec(conn, query->str);
      g_string_free(query, TRUE);
      PQfinish(conn);
    }
  if (whoq)
    g_free(whoq);

  /* better to call it since some other users can have modified the same 
     parameters (at least it will refresh) */
  users_ac_start_reset_rights(info);  

  /* We require that other components be updated */
  require_components_refresh(info);
}
