/* GNOME DB Front End
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-fe.h"
#include <stdio.h>

static void clear_cmdlist_cb (GtkWidget *w, gpointer data);
static void execute_cmd_cb   (GtkWidget *w, gpointer data);
static void open_file_cb     (GtkWidget *w, gpointer data);

static GnomeUIInfo batch_toolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Execute"), N_("Execute batch file"),
    execute_cmd_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Open file"),
    open_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save file"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear command list"),
    clear_cmdlist_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
load_file_into_clist (connection_data_t *cnc_data,
		      const gchar *file,
		      const gchar *seps,
		      gboolean append)
{
  FILE* fp;

  g_return_if_fail(cnc_data != NULL);
  g_return_if_fail(GTK_IS_CLIST(cnc_data->batch_cmdlist));
  g_return_if_fail(file != NULL);
  g_return_if_fail(seps != NULL);

  /* open the file */
  fp = fopen(file, "r");
  if (fp)
    {
      GString* buffer = NULL;
      gchar    line[2048];
      gchar*   tmp;

      /* read file contents */
      while (fgets(line, sizeof(line) - 1, fp))
	{
	  if (!buffer) buffer = g_string_new(line);
	  else buffer = g_string_append(buffer, line);
	}

      gtk_clist_freeze(GTK_CLIST(cnc_data->batch_cmdlist));
      if (!append) clear_cmdlist_cb(NULL, (gpointer) cnc_data);

      /* load commands into list */
      tmp = strtok(buffer->str, seps);
      while (tmp)
	{
	  gchar* row[3];

	  row[0] = "";
	  row[1] = "";
	  row[2] = tmp;
	  gtk_clist_append(GTK_CLIST(cnc_data->batch_cmdlist), row);
	  tmp = strtok(NULL, seps);
	}

      gtk_clist_thaw(GTK_CLIST(cnc_data->batch_cmdlist));
      g_string_free(buffer, TRUE);
      fclose(fp);
    }
  else gda_ui_show_error(_("Could not open file\n'%s'"), file);
}

static void
manage_cmd (connection_data_t *cnc_data, gboolean edit_mode)
{
  GtkWidget* dialog;
  GtkWidget* frame;
  GtkWidget* table;
  GtkWidget* label;
  GtkWidget* scroll;
  GtkWidget* cmd_area;
  GtkWidget* insert_button;
  gchar*     text = NULL;
  guint      row;
  GList*     sel;

  g_return_if_fail(cnc_data != NULL);
  g_return_if_fail(GTK_IS_CLIST(cnc_data->batch_cmdlist));

  /* retrieve selection */
  sel = GTK_CLIST(cnc_data->batch_cmdlist)->selection;
  if (sel) row = GPOINTER_TO_UINT(sel->data);
  else row = 0;

  if (edit_mode)
    {
      if (sel)
	{
	  if (!gtk_clist_get_text(GTK_CLIST(cnc_data->batch_cmdlist), row, 2, &text))
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					    _("No command selected"));
	      return;
	    }
	}
      else return;
    }

  /* create dialog */
  dialog = gnome_dialog_new(_("Command"),
			    GNOME_STOCK_BUTTON_OK,
			    GNOME_STOCK_BUTTON_CANCEL,
			    NULL);
  gtk_widget_set_usize(dialog, 300, 300);
  gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
  frame = gda_ui_new_frame_widget(NULL);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
  table = gda_ui_new_table_widget(2, 3, FALSE);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gda_ui_new_label_widget(_("Enter command text"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  scroll = gda_ui_new_scrolled_window_widget();
  gtk_table_attach(GTK_TABLE(table), scroll, 0, 2, 1, 2,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		   3, 3);
  cmd_area = gda_ui_new_text_widget();
  if (edit_mode)
    gtk_text_insert(GTK_TEXT(cmd_area), NULL, NULL, NULL, text, strlen(text));
  gtk_container_add(GTK_CONTAINER(scroll), cmd_area);
  
  if (!edit_mode)
    {
      insert_button = gda_ui_new_check_button_widget(_("Insert at current position"),
						     FALSE);
      gtk_table_attach(GTK_TABLE(table), insert_button, 0, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
    }

  if (!gnome_dialog_run(GNOME_DIALOG(dialog)))
    {
      gchar* cols[3];
      text = gtk_editable_get_chars(GTK_EDITABLE(cmd_area), 0,
				    gtk_text_get_length(GTK_TEXT(cmd_area)));
      if (text)
	{
	  if (edit_mode)
	    {
	      gtk_clist_set_text(GTK_CLIST(cnc_data->batch_cmdlist), row, 2, text);
	    }
	  else
	    {
	      cols[0] = "";
	      cols[1] = "";
	      cols[2] = text;
	      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(insert_button)))
		{
		  gtk_clist_insert(GTK_CLIST(cnc_data->batch_cmdlist), row, cols);
		}
	      else gtk_clist_append(GTK_CLIST(cnc_data->batch_cmdlist), cols);
	    }
	}
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

/*
 * Callbacks
 */
static void
add_cmd_cb (GtkButton *button, connection_data_t *cnc_data)
{
  g_return_if_fail(cnc_data != NULL);
  manage_cmd(cnc_data, FALSE);
}

static void
clear_cmdlist_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = (connection_data_t *) data;

  g_return_if_fail(cnc_data != NULL);
  g_return_if_fail(gda_connection_is_open(cnc_data->cnc));

  gda_ui_clear_clist(GTK_CLIST(cnc_data->batch_cmdlist));
}

static void
edit_cmd_cb (GtkButton *button, connection_data_t *cnc_data)
{
  g_return_if_fail(cnc_data != NULL);
  manage_cmd(cnc_data, TRUE);
}

static void
execute_cmd_cb (GtkWidget *w, gpointer data)
{
  Gda_Batch*         batch;
  gint               cnt;
  connection_data_t* cnc_data = (connection_data_t *) data;

  g_return_if_fail(cnc_data != NULL);
  g_return_if_fail(gda_connection_is_open(cnc_data->cnc));

  /* create Gda_Batch object */
  batch = gda_batch_new();
  gda_batch_set_connection(batch, cnc_data->cnc);
  gda_batch_set_transaction_mode(batch,
				 gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cnc_data->batch_transmode)));

  /* add all commands */
  for (cnt = 0; cnt < GTK_CLIST(cnc_data->batch_cmdlist)->rows; cnt++)
    {
      gchar* cmd;

      if (gtk_clist_get_text(GTK_CLIST(cnc_data->batch_cmdlist), cnt, 2, &cmd))
	{
	  gda_batch_add_command(batch, (const gchar *) cmd);
	}
    }

  gda_batch_start(batch);

  gda_batch_free(batch);
}

static void
move_down_cmd_cb (GtkButton *button, connection_data_t *cnc_data)
{
  GList* sel;

  g_return_if_fail(cnc_data != NULL);
  g_return_if_fail(GTK_IS_CLIST(cnc_data->batch_cmdlist));

  sel = GTK_CLIST(cnc_data->batch_cmdlist)->selection;
  if (sel)
    {
      guint row = GPOINTER_TO_UINT(sel->data);
      gtk_clist_swap_rows(GTK_CLIST(cnc_data->batch_cmdlist), row, row + 1);
    }
}

static void
move_up_cmd_cb (GtkButton *button, connection_data_t *cnc_data)
{
    GList* sel;

  g_return_if_fail(cnc_data != NULL);
  g_return_if_fail(GTK_IS_CLIST(cnc_data->batch_cmdlist));

  sel = GTK_CLIST(cnc_data->batch_cmdlist)->selection;
  if (sel)
    {
      guint row = GPOINTER_TO_UINT(sel->data);
      gtk_clist_swap_rows(GTK_CLIST(cnc_data->batch_cmdlist), row, row - 1);
    }
}

static void
open_file_cb (GtkWidget *w, gpointer data)
{
  GtkWidget*         dialog;
  GtkWidget*         frame;
  GtkWidget*         table;
  GtkWidget*         label;
  GtkWidget*         sep_entry;
  GtkWidget*         file_entry;
  GtkWidget*         append_mode;
  connection_data_t* cnc_data = (connection_data_t *) data;

  g_return_if_fail(cnc_data != NULL);
  g_return_if_fail(gda_connection_is_open(cnc_data->cnc));

  /* create dialog */
  dialog = gnome_dialog_new(_("Open File"),
			    GNOME_STOCK_BUTTON_OK,
			    GNOME_STOCK_BUTTON_CANCEL,
			    NULL);
  gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);

  frame = gda_ui_new_frame_widget(NULL);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
  table = gda_ui_new_table_widget(2, 3, FALSE);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gda_ui_new_label_widget(_("Command separators"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  sep_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_entry_set_text(GTK_ENTRY(sep_entry), ";");
  gtk_table_attach(GTK_TABLE(table), sep_entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  label = gda_ui_new_label_widget(_("File"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
  file_entry = gda_ui_new_file_entry_widget("FE_Batch_history");
  gtk_table_attach(GTK_TABLE(table), file_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

  append_mode = gda_ui_new_check_button_widget(_("Append to current file"), FALSE);
  gtk_table_attach(GTK_TABLE(table), append_mode, 0, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

  /* run dialog */
  if (!gnome_dialog_run(GNOME_DIALOG(dialog)))
    {
      gchar* seps;
      gchar* file;

      seps = gtk_entry_get_text(GTK_ENTRY(sep_entry));
      if (seps)
	{
	  file = gnome_file_entry_get_full_path(GNOME_FILE_ENTRY(file_entry), FALSE);
	  if (file)
	    {
	      load_file_into_clist(cnc_data, file, seps,
				   gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(append_mode)));
	      g_free((gpointer) file);
	    }
	  else gda_ui_show_error(_("You must specify a file to be loaded"));
	}
      else gda_ui_show_error(_("You must specifiy the characters used as command separators"));
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
remove_cmd_cb (GtkButton *button, connection_data_t *cnc_data)
{
  GList* sel;

  g_return_if_fail(cnc_data != NULL);
  g_return_if_fail(GTK_IS_CLIST(cnc_data->batch_cmdlist));

  sel = GTK_CLIST(cnc_data->batch_cmdlist)->selection;
  if (sel)
    {
      guint row = GPOINTER_TO_UINT(sel->data);
      gtk_clist_remove(GTK_CLIST(cnc_data->batch_cmdlist), row);
    }
}

/*
 * Public functions
 */
void
fe_connection_show_batch_window (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      /* create the tab if not created yet */
      if (!cnc_data->batch_tab)
	{
	  GtkWidget* scroll;
	  GtkWidget* toolbar;
	  GtkWidget* frame;
	  GtkWidget* table;
	  GtkWidget* box;
	  GtkWidget* button;

	  cnc_data->batch_tab = gda_ui_new_table_widget(4, 4, FALSE);

	  toolbar = gda_ui_new_toolbar_widget(GTK_ORIENTATION_VERTICAL,
					      GTK_TOOLBAR_ICONS,
					      batch_toolbar,
					      (gpointer) cnc_data);
	  gtk_table_attach(GTK_TABLE(cnc_data->batch_tab), toolbar, 0, 1, 0, 1,
			   GTK_SHRINK, GTK_FILL, 3, 3);

	  scroll = gda_ui_new_scrolled_window_widget();
	  gtk_table_attach(GTK_TABLE(cnc_data->batch_tab), scroll, 0, 4, 1, 4,
			   GTK_FILL,
			   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			   3, 3);
	  cnc_data->batch_cmdlist = gda_ui_new_clist_widget(NULL, 3);
	  gtk_container_add(GTK_CONTAINER(scroll), cnc_data->batch_cmdlist);

	  /* add right toolbar */
	  box = gtk_vbox_new(FALSE, 0);
	  gtk_widget_show(box);
	  gtk_table_attach(GTK_TABLE(cnc_data->batch_tab), box, 2, 3, 0, 1,
			   GTK_FILL, GTK_FILL, 3, 3);

	  button = gda_ui_new_button_widget_with_pixmap(_("Add"), GNOME_STOCK_MENU_NEW);
	  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
	  gtk_signal_connect(GTK_OBJECT(button),
			     "clicked",
			     GTK_SIGNAL_FUNC(add_cmd_cb),
			     (gpointer) cnc_data);

	  button = gda_ui_new_button_widget_with_pixmap(_("Edit"), GNOME_STOCK_MENU_PROP);
	  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
	  gtk_signal_connect(GTK_OBJECT(button),
			     "clicked",
			     GTK_SIGNAL_FUNC(edit_cmd_cb),
			     (gpointer) cnc_data);

	  button = gda_ui_new_button_widget_with_pixmap(_("Remove"), GNOME_STOCK_MENU_TRASH);
	  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
	  gtk_signal_connect(GTK_OBJECT(button),
			     "clicked",
			     GTK_SIGNAL_FUNC(remove_cmd_cb),
			     (gpointer) cnc_data);

	  box = gtk_vbox_new(FALSE, 0);
	  gtk_widget_show(box);
	  gtk_table_attach(GTK_TABLE(cnc_data->batch_tab), box, 3, 4, 0, 1,
			   GTK_FILL, GTK_FILL, 3, 3);
	  
	  button = gda_ui_new_button_widget_with_pixmap(_("Up"), GNOME_STOCK_MENU_UP);
	  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
	  gtk_signal_connect(GTK_OBJECT(button),
			     "clicked",
			     GTK_SIGNAL_FUNC(move_up_cmd_cb),
			     (gpointer) cnc_data);

	  button = gda_ui_new_button_widget_with_pixmap(_("Down"), GNOME_STOCK_MENU_DOWN);
	  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
	  gtk_signal_connect(GTK_OBJECT(button),
			     "clicked",
			     GTK_SIGNAL_FUNC(move_down_cmd_cb),
			     (gpointer) cnc_data);

	  /* create options area */
	  frame = gda_ui_new_frame_widget(_("Options"));
	  gtk_table_attach(GTK_TABLE(cnc_data->batch_tab), frame, 1, 2, 0, 1,
			   GTK_FILL | GTK_EXPAND,
			   GTK_FILL,
			   3, 3);
	  table = gda_ui_new_table_widget(4, 8, FALSE);
	  gtk_container_add(GTK_CONTAINER(frame), table);

	  cnc_data->batch_transmode =
	    gda_ui_new_check_button_widget(_("Transaction Mode"),
		       gda_connection_supports(cnc_data->cnc,
					       GDA_Connection_FEATURE_TRANSACTIONS));
	  gtk_table_attach(GTK_TABLE(table), cnc_data->batch_transmode, 0, 1, 0, 1,
			   GTK_FILL, GTK_FILL, 1, 1);
	  cnc_data->batch_sendxml = gda_ui_new_check_button_widget(_("Send as XML"), FALSE);
	  gtk_table_attach(GTK_TABLE(table), cnc_data->batch_sendxml, 0, 1, 1, 2,
			   GTK_FILL, GTK_FILL, 1, 1);

	  gtk_notebook_append_page(GTK_NOTEBOOK(cnc_data->widget),
				   cnc_data->batch_tab,
				   gtk_label_new(_("Batch")));
	}
      gtk_notebook_set_page(GTK_NOTEBOOK(cnc_data->widget),
			    gtk_notebook_page_num(GTK_NOTEBOOK(cnc_data->widget),
						  cnc_data->batch_tab));
    }
}



