/* GNOME DB components libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome-db-container.h>

typedef struct
{
  BonoboViewFrame*  view_frame;
  GnomeDbContainer* container;
  GtkWidget*        widget;
} COMPONENTS_ContainerView;

static void gnome_db_container_init       (GnomeDbContainer *container);
static void gnome_db_container_class_init (GnomeDbContainerClass *klass);

static void gnome_db_container_destroy    (GnomeDbContainer *container);

enum
{
  GNOME_DB_CONTAINER_CLOSE,
  GNOME_DB_CONTAINER_INSERT_OBJECT,
  GNOME_DB_CONTAINER_LAST_SIGNAL
};

static gint gnome_db_container_signals[GNOME_DB_CONTAINER_LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
static gint
close_cb (GnomeApp *app, GdkEvent *event, GnomeDbContainer *container)
{
  g_return_val_if_fail(GNOME_IS_APP(app), -1);
  g_return_val_if_fail(GNOME_DB_IS_CONTAINER(container), -1);

  gtk_signal_emit(GTK_OBJECT(container),
		  gnome_db_container_signals[GNOME_DB_CONTAINER_CLOSE]);
  //gtk_object_destroy(GTK_OBJECT(container));
}

static Bonobo_Unknown
get_object_cb (BonoboObject* bonobo_container,
	       CORBA_char *item_name,
	       CORBA_boolean only_if_exists,
	       CORBA_Environment *ev,
	       GnomeDbContainer *container)
{
  GtkWidget* widget;

  g_return_val_if_fail(BONOBO_IS_CONTAINER(bonobo_container), CORBA_OBJECT_NIL);
  g_return_val_if_fail(item_name != NULL, CORBA_OBJECT_NIL);
  g_return_val_if_fail(GNOME_DB_IS_CONTAINER(container), CORBA_OBJECT_NIL);

  if (container->get_object_func)
    {
      widget = container->get_object_func(container, (gchar *) item_name);
      if (GTK_IS_WIDGET(widget))
	{
	  /* FIXME: must create GnomeEmbeddable from 'widget' */
	}
    }
  return CORBA_OBJECT_NIL;
}

static gint
persist_file_load_cb (BonoboPersistFile *ps, CORBA_char *filename, void *closure)
{
}

static gint
persist_file_save_cb (BonoboPersistFile *ps, const CORBA_char *filename, void *closure)
{
}

static gint
user_activation_request_cb (BonoboViewFrame *view_frame, COMPONENTS_ContainerView *view)
{
  g_return_val_if_fail(view != NULL, FALSE);

  if (view->view_frame)
    {
      bonobo_view_frame_view_deactivate(view->view_frame);
      if (view->view_frame)
	bonobo_view_frame_set_covered(view->view_frame, TRUE);
      view->view_frame = NULL;
    }

  bonobo_view_frame_view_activate(view_frame);

  return FALSE;
}

static gint
view_activated_cb (BonoboViewFrame *view_frame, gboolean activated, COMPONENTS_ContainerView *view)
{
  g_return_val_if_fail(view != NULL, FALSE);

  if (activated)
    {
      if (view->view_frame)
	{
	  g_warning(_("View requested to be activated but there is already an active view"));
	  return FALSE;
	}

      bonobo_view_frame_set_covered(view_frame, FALSE);
      view->view_frame = view_frame;
    }
  else
    {
      /* deactivate */
      bonobo_view_frame_set_covered(view_frame, TRUE);
      if (view_frame == view->view_frame)
	view->view_frame = NULL;
    }
  return FALSE;
}

/*
 * GnomeDbContainer object implementation
 */
static void
gnome_db_container_class_init (GnomeDbContainerClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *) klass;

  gnome_db_container_signals[GNOME_DB_CONTAINER_CLOSE] =
    gtk_signal_new("close",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GnomeDbContainerClass, close),
		   gtk_signal_default_marshaller,
		   GTK_TYPE_NONE, 0);
  gnome_db_container_signals[GNOME_DB_CONTAINER_INSERT_OBJECT] =
    gtk_signal_new("insert_object",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GnomeDbContainerClass, insert_object),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  gtk_object_class_add_signals(object_class, gnome_db_container_signals,
			       GNOME_DB_CONTAINER_LAST_SIGNAL);

  object_class->destroy = gnome_db_container_destroy;

  klass->close = NULL;
  klass->insert_object = NULL;
}

static void
gnome_db_container_init (GnomeDbContainer *container)
{
  container->bonobo_container = BONOBO_CONTAINER(bonobo_container_new());
  container->persist_file = bonobo_persist_file_new(persist_file_load_cb,
						    persist_file_save_cb,
						    (gpointer) container);

  bonobo_object_add_interface(BONOBO_OBJECT(container),
			      BONOBO_OBJECT(container->bonobo_container));
  bonobo_object_add_interface(BONOBO_OBJECT(container),
			      BONOBO_OBJECT(container->persist_file));

  gtk_signal_connect(GTK_OBJECT(container->bonobo_container),
		     "get_object",
		     GTK_SIGNAL_FUNC(get_object_cb),
		     (gpointer) container);

  container->toplevel = NULL;
  container->inserted_objects = NULL;
  container->get_object_func = NULL;
}

GtkType
gnome_db_container_get_type (void)
{
  static GtkType db_container_type = 0;

  if (!db_container_type)
    {
      GtkTypeInfo db_container_info =
      {
	"GnomeDbContainer",
	sizeof (GnomeDbContainer),
	sizeof (GnomeDbContainerClass),
	(GtkClassInitFunc) gnome_db_container_class_init,
	(GtkObjectInitFunc) gnome_db_container_init,
	(GtkArgSetFunc) NULL,
	(GtkArgSetFunc) NULL
      };
      db_container_type = gtk_type_unique(bonobo_object_get_type(), &db_container_info);
    }
  return db_container_type;
}

/**
 * gnome_db_container_new
 */
GnomeDbContainer *
gnome_db_container_new (const gchar *name,
			GnomeUIInfo *menus[],
			GnomeUIInfo *toolbar[],
			GnomeDbContainerGetObjectFunc func)
{
  GnomeDbContainer*           container;
  BonoboUIHandlerMenuItem*    menu_list;
  BonoboUIHandlerToolbarItem* toolbar_list;

  container = gtk_type_new(gnome_db_container_get_type());
  container->toplevel = gnome_app_new(name, name);
  gtk_signal_connect(GTK_OBJECT(container->toplevel),
		     "delete_event",
		     GTK_SIGNAL_FUNC(close_cb),
		     (gpointer) container);

  /* set menus */
  container->uih = bonobo_ui_handler_new();
  bonobo_ui_handler_set_app(container->uih, GNOME_APP(container->toplevel));
  if (menus)
    {
      bonobo_ui_handler_create_menubar(container->uih);
      menu_list = bonobo_ui_handler_menu_parse_uiinfo_list(menus);
      bonobo_ui_handler_menu_add_list(container->uih, "/", menu_list);
      bonobo_ui_handler_menu_free_list(menu_list);
    }

  /* create toolbar */
  if (toolbar)
    {
      bonobo_ui_handler_create_toolbar(container->uih, "Toolbar");
      toolbar_list = bonobo_ui_handler_toolbar_parse_uiinfo_list(toolbar);
      bonobo_ui_handler_toolbar_add_list(container->uih, "/Toolbar", toolbar_list);
      bonobo_ui_handler_toolbar_free_list(toolbar_list);
    }

  /* set status bar */
  container->app_bar = gnome_appbar_new(TRUE, TRUE, GNOME_PREFERENCES_USER);
  gnome_appbar_set_default(GNOME_APPBAR(container->app_bar), _("Ready"));
  gnome_app_set_statusbar(GNOME_APP(container->toplevel), container->app_bar);
  if (menus)
    {
      gnome_app_install_menu_hints(GNOME_APP(container->toplevel), menus);
    }

  return container;
}

/**
 * gnome_db_container_new_glade
 */
GnomeDbContainer *
gnome_db_container_new_glade ( GnomeApp *app, 
                               GtkMenuBar *menubar, 
                               GtkToolbar *toolbar, 
			       GnomeAppBar *app_bar,
                               GnomeDbContainerGetObjectFunc func)
{
  GnomeDbContainer* container;
 
  g_return_if_fail(GNOME_IS_APP(app));
  g_return_if_fail(GTK_IS_MENU_BAR(menubar));
  g_return_if_fail(GTK_IS_TOOLBAR(toolbar));
  g_return_if_fail(GNOME_IS_APPBAR(app_bar));
  

  container = gtk_type_new(gnome_db_container_get_type());
  container->toplevel = app;

  gtk_signal_connect(GTK_OBJECT(container->toplevel),
                     "delete_event",
                     GTK_SIGNAL_FUNC(close_cb),
                     (gpointer) container);

  /* Set uihandler to control teh menubar and toolbar */
  container->uih = bonobo_ui_handler_new();
  bonobo_ui_handler_set_app(container->uih, GNOME_APP(container->toplevel));
  /* set menus */
  bonobo_ui_handler_set_menubar(container->uih, menubar);
  /* set toolbar */
  bonobo_ui_handler_set_toolbar(container->uih, "Fix me", toolbar);

  /* set gnome application bar */
  container->app_bar = app_bar; 
  gnome_appbar_set_default(GNOME_APPBAR(container->app_bar), _("Ready"));
  gnome_app_set_statusbar(GNOME_APP(container->toplevel), container->app_bar);
  return container;
}

static void
gnome_db_container_destroy (GnomeDbContainer *container)
{
  g_return_if_fail(GNOME_DB_IS_CONTAINER(container));
}

/**
 * gnome_db_container_get_toplevel
 */
GtkWidget *
gnome_db_container_get_toplevel (GnomeDbContainer *container)
{
  g_return_val_if_fail(GNOME_DB_IS_CONTAINER(container), NULL);
  return container->toplevel;
}

/**
 * gnome_db_container_get_contents
 */
GtkWidget *
gnome_db_container_get_contents (GnomeDbContainer *container)
{
  g_return_val_if_fail(GNOME_DB_IS_CONTAINER(container), NULL);
  g_return_val_if_fail(GNOME_IS_APP(container->toplevel), NULL);
}

/**
 * gnome_db_container_set_contents
 */
void
gnome_db_container_set_contents (GnomeDbContainer *container, GtkWidget *w)
{
  g_return_if_fail(GNOME_DB_IS_CONTAINER(container));
  g_return_if_fail(GNOME_IS_APP(container->toplevel));
  g_return_if_fail(GTK_IS_WIDGET(w));

  gnome_app_set_contents(GNOME_APP(container->toplevel), w);
}

/**
 * gnome_db_container_set_status_message
 */
void
gnome_db_container_set_status (GnomeDbContainer *container, const gchar *msg, ...)
{
  va_list args;
  gchar   sz[256];

  g_return_if_fail(GNOME_DB_IS_CONTAINER(container));
  g_return_if_fail(GNOME_IS_APPBAR(container->app_bar));
  g_return_if_fail(msg != NULL);

  va_start(args, msg);
  vsprintf(sz, msg, args);
  va_end(args);

  gnome_appbar_set_status(GNOME_APPBAR(container->app_bar), (const gchar *) sz);
}

/**
 * gnome_db_container_get_progress
 */
gfloat
gnome_db_container_get_progress (GnomeDbContainer *container)
{
  g_return_val_if_fail(GNOME_DB_IS_CONTAINER(container), 0.0);
  return gtk_progress_get_value(gnome_appbar_get_progress(GNOME_APPBAR(container->app_bar)));
}

/**
 * gnome_db_container_set_progress
 */
void
gnome_db_container_set_progress (GnomeDbContainer *container, gfloat percentage)
{
  g_return_if_fail(GNOME_DB_IS_CONTAINER(container));
  g_return_if_fail(GNOME_IS_APPBAR(container->app_bar));
  gnome_appbar_set_progress(GNOME_APPBAR(container->app_bar), percentage);
}

/**
 * gnome_db_container_show
 */
void
gnome_db_container_show (GnomeDbContainer *container)
{
  g_return_if_fail(GNOME_DB_IS_CONTAINER(container));
  g_return_if_fail(GNOME_IS_APP(container->toplevel));

  gtk_widget_show_all(container->toplevel);
}

/**
 * gnome_db_container_insert_object
 */
void
gnome_db_container_insert_object (GnomeDbContainer *container, const gchar *id)
{
  BonoboObjectClient*       object_server;
  BonoboClientSite*         client_site;
  BonoboViewFrame*          view_frame;
  COMPONENTS_ContainerView* view;

  g_return_if_fail(GNOME_DB_IS_CONTAINER(container));
  g_return_if_fail(id != NULL);

  if (!strncmp(id, "moniker_url:", 12))
    object_server = bonobo_object_activate(id, 0);
  else
    object_server = bonobo_object_activate_with_goad_id(NULL, id, 0, NULL);

  if (object_server)
    {
      client_site = bonobo_client_site_new(container->bonobo_container);
      bonobo_container_add(container->bonobo_container, BONOBO_OBJECT(client_site));
      if (!bonobo_client_site_bind_embeddable(client_site, object_server))
	{
	  g_warning(_("Could not bind object %s"), id);
	  gtk_object_unref(GTK_OBJECT(object_server));
	  gtk_object_unref(GTK_OBJECT(client_site));
	  return;
	}

      /* add view to list of active views */
      view = g_new0(COMPONENTS_ContainerView, 1);
      view->view_frame = NULL;
      view_frame = bonobo_client_site_new_view(client_site, container->uih->top_level_uih);
      view->container = container;
      gtk_signal_connect(GTK_OBJECT(view_frame),
			 "user_activate",
			 GTK_SIGNAL_FUNC(user_activation_request_cb),
			 (gpointer) view);
      gtk_signal_connect(GTK_OBJECT(view_frame),
			 "activated",
			 GTK_SIGNAL_FUNC(view_activated_cb),
			 (gpointer) view);

      view->widget = bonobo_view_frame_get_wrapper(view_frame);
      gtk_signal_emit(GTK_OBJECT(container),
		      gnome_db_container_signals[GNOME_DB_CONTAINER_INSERT_OBJECT],
		      view->widget);

      container->inserted_objects = g_list_append(container->inserted_objects,
						  (gpointer) view);
    }
  else g_warning(_("Unable to activate object %s"), id);
}




