/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(HAVE_CONFIG_H)
#  include <config.h>
#endif
#include "gda-ui.h"

static void gnome_db_list_class_init (GnomeDbListClass *klass);
static void gnome_db_list_init       (GnomeDbList *dblist);

void gnome_db_list_destroy (GnomeDbList *dblist);

/*
 * GnomeDbList widget signals
 */
enum
{
  GNOME_DB_LIST_LOADED,
  GNOME_DB_LIST_SELECT_ROW,
  GNOME_DB_LIST_ADD_ROW,
  GNOME_DB_LIST_LAST_SIGNAL
};

static gint gnome_db_list_signals[GNOME_DB_LIST_LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
static void
row_selected_cb (GtkCList *clist,
                 gint row,
                 gint col,
                 GdkEvent *event,
                 GnomeDbList *dblist)
{
  gchar *str = 0;

  g_return_if_fail(GNOME_DB_IS_LIST(dblist));
  g_return_if_fail(GTK_IS_CLIST(dblist->list));

  gtk_clist_get_text(clist, row, col, &str);
  if (str != 0)
    {
      gtk_signal_emit_by_name(GTK_OBJECT(dblist), "select_row");
    }
}

static gint
timeout_callback (GnomeDbList *dblist)
{
  gint position, cnt;
  
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
  g_return_val_if_fail(GTK_IS_CLIST(dblist->list), 0);
  g_return_val_if_fail(IS_GDA_RECORDSET(dblist->recset), 0);
  
  position = gda_recordset_move(dblist->recset, 1, 0);
  if (position != GDA_RECORDSET_INVALID_POSITION && !gda_recordset_eof(dblist->recset))
    {
      gchar*     row[2] = { "", 0 };
      Gda_Field* field;
      
      field = gda_recordset_field_idx(dblist->recset, dblist->col);
      if (field)
	{
	  row[1] = gda_stringify_value(0, 0, field);
	  gtk_clist_freeze(GTK_CLIST(dblist->list));
	  gtk_clist_append(GTK_CLIST(dblist->list), (gchar **) row);
	  gtk_signal_emit(GTK_OBJECT(dblist),
	                  gnome_db_list_signals[GNOME_DB_LIST_ADD_ROW],
	                  dblist->total_rows);
	  gtk_clist_thaw(GTK_CLIST(dblist->list));
      
	  dblist->total_rows++;
	}
    }
  else
    {
      gtk_signal_emit_by_name(GTK_OBJECT(dblist), "loaded");
      dblist->timeout_handle = -1;
      return (0);
    }
  return (1);
}

/*
 * GnomeDbList widget interface
 */
static void
gnome_db_list_class_init (GnomeDbListClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  gnome_db_list_signals[GNOME_DB_LIST_LOADED] =
           gtk_signal_new("loaded", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbListClass, loaded),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);
  gnome_db_list_signals[GNOME_DB_LIST_SELECT_ROW] =
           gtk_signal_new("select_row", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbListClass, select_row),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);
  gnome_db_list_signals[GNOME_DB_LIST_ADD_ROW] =
           gtk_signal_new("add_row", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbListClass, add_row),
                          gtk_marshal_NONE__INT,
                          GTK_TYPE_NONE, 1, GTK_TYPE_INT);
  gtk_object_class_add_signals(object_class, gnome_db_list_signals,
                               GNOME_DB_LIST_LAST_SIGNAL);

  klass->loaded = 0;
  klass->select_row = 0;
  klass->add_row = 0;
  object_class->destroy = gnome_db_list_destroy;
}

static void
gnome_db_list_init (GnomeDbList *dblist)
{
  GtkWidget *scroll;

  dblist->recset = 0;
  dblist->total_rows = 0;
  dblist->col = -1;
  dblist->timeout_handle = -1;

  scroll = gda_ui_new_scrolled_window_widget();
  dblist->list = gda_ui_new_clist_widget(0, 2);
  gtk_signal_connect(GTK_OBJECT(dblist->list), "select_row",
                     GTK_SIGNAL_FUNC(row_selected_cb), (gpointer) dblist);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), 
                                        dblist->list);
  gtk_box_pack_start(GTK_BOX(dblist), scroll, 1, 1, 0);
}

guint
gnome_db_list_get_type (void)
{
  static guint db_list_type = 0;

  if (!db_list_type)
    {
      GtkTypeInfo db_list_info =
      {
        "GnomeDbList",
        sizeof (GnomeDbList),
        sizeof (GnomeDbListClass),
        (GtkClassInitFunc) gnome_db_list_class_init,
        (GtkObjectInitFunc) gnome_db_list_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_list_type = gtk_type_unique(gtk_vbox_get_type(), &db_list_info);
    }
  return (db_list_type);
}

/**
 * gnome_db_list_new
 * @recset: recordset to be shown
 * @pos: position of the field being shown
 *
 * Creates a new GnomeDbList widget
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_list_new (Gda_Recordset *recset, gint pos)
{
  GtkWidget *dblist = gtk_type_new(gnome_db_list_get_type());

  if (recset != 0)
    gnome_db_list_set_recordset(GNOME_DB_LIST(dblist), recset, pos);
  return (dblist);
}

void
gnome_db_list_destroy (GnomeDbList *dblist)
{
  g_return_if_fail(GNOME_DB_IS_LIST(dblist));

  gnome_db_list_clear(dblist);
}

/**
 * gnome_db_list_clear
 * @dblist: GnomeDbList widget
 *
 * Clears the DbList widget. It clears all entries in the list and frees
 * the recordset associated
 */
void
gnome_db_list_clear (GnomeDbList *dblist)
{
  g_return_if_fail(GNOME_DB_IS_LIST(dblist));

  /* stop timeout if active */
  if (dblist->timeout_handle != -1)
    {
      gtk_timeout_remove(dblist->timeout_handle);
      dblist->timeout_handle = -1;
    }

  if (dblist->recset != 0)
    {
      gda_recordset_free(dblist->recset);
      dblist->recset = 0;
    }
  dblist->total_rows = 0;
  gda_ui_clear_clist(GTK_CLIST(dblist->list));
}

/**
 * gnome_db_list_get_row_count
 * @dblist: GnomeDbList widget
 *
 * Returns number of rows actually loaded in the recordset. As the list widget loads
 * data asynchronously, you may call this function before all rows have been fetched
 * from the database, in which case, it won't reflect the "real" number of rows
 * returned in the recordset
 *
 * Returns: number of rows in the recordset , or -1 on error
 */
gint
gnome_db_list_get_row_count (GnomeDbList *dblist)
{
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), -1);
  return (dblist->total_rows);
}

/**
 * gnome_db_list_get_string
 * @dblist: GnomeDbList widget
 *
 * Returns the currently selected item's text. That is, the literal of the first
 * selected item
 *
 * Returns: a string containing the literal selected, or NULL if none is
 * selected or if there is an error
 */
gchar *
gnome_db_list_get_string (GnomeDbList *dblist)
{
  GList *selected;
  
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
  
  selected = GTK_CLIST(dblist->list)->selection;
  if (selected)
    {
      gchar *name = 0;
      guint row = GPOINTER_TO_UINT(selected->data);
      gtk_clist_get_text(GTK_CLIST(dblist->list), row, 1, &name);
      g_warning("name = %s", name);
      return (name);
    }
  return (0);
}

/**
 * gnome_db_list_get_recordset
 * @dblist: GnomeDbList widget
 *
 * Returns the Gda_Recordset object associated with the given GnomeDbList
 * widget
 *
 * Returns: a pointer to the Gda_Recordset object being used, or NULL if none
 * or if there is an error
 */
Gda_Recordset *
gnome_db_list_get_recordset (GnomeDbList *dblist)
{
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
  return (dblist->recset);
}

/**
 * gnome_db_list_set_recordset
 * @dblist: GnomeDbList widget
 * @recset: recordset to be shown
 * @pos: position of the field being shown
 *
 * Associates a Gda_Recordset object with the given GnomeDbList
 * widget and shows its contents
 */
void
gnome_db_list_set_recordset (GnomeDbList *dblist, Gda_Recordset *recset, gint pos)
{
  gulong position;

  g_return_if_fail(GNOME_DB_IS_LIST(dblist));
  g_return_if_fail(IS_GDA_RECORDSET(recset));
  g_return_if_fail(pos >= 0);

  gnome_db_list_clear(dblist);

  dblist->recset = recset;
  dblist->total_rows = 0;
  dblist->col = pos;

  dblist->timeout_handle = gtk_timeout_add(15,
                                           (GtkFunction) timeout_callback,
                                           (gpointer) dblist);
}

/**
 * gnome_db_list_get_pixmap
 * @dblist: GnomeDbList widget
 * @row: row number
 *
 * Returns the pixmap being used for the given row
 */
GdkPixmap *
gnome_db_list_get_pixmap (GnomeDbList *dblist, gint row)
{
  GdkPixmap* pixmap;
  GdkBitmap* mask;
  
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
  
  gtk_clist_get_pixmap(GTK_CLIST(dblist->list), row, 0, &pixmap, &mask);
  return (pixmap);
}

/**
 * gnome_db_list_set_pixmap
 * @dblist: GnomeDbList widget
 * @row: row number
 * @pixmap: GdkPixmap object
 *
 * Sets the pixmap to be shown for the given row
 */
void
gnome_db_list_set_pixmap (GnomeDbList *dblist,
                          gint row,
                          GdkPixmap *pixmap,
                          GdkBitmap *mask)
{
  g_return_if_fail(GNOME_DB_IS_LIST(dblist));
  
  gtk_clist_set_pixmap(GTK_CLIST(dblist->list),
                       row,
                       0,
                       pixmap,
                       mask);
}

/**
 * gnome_db_list_get_pixmap_mask
 * @dblist: GnomeDbList widget
 * @row: row number
 */
GdkBitmap *
gnome_db_list_get_pixmap_mask (GnomeDbList *dblist, gint row)
{
  GdkPixmap* pixmap;
  GdkBitmap* mask;
  
  g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
  
  gtk_clist_get_pixmap(GTK_CLIST(dblist->list), row, 0, &pixmap, &mask);
  return (mask);
}
