/* GNOME DB Server Library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if !defined(__gda_server_impl_h__)
#  define __gda_server_impl_h__

#include <glib.h>
#include <gtk/gtk.h>
#include <orb/orbit.h>
#include <libgnorba/gnorba.h>
#include <gda.h>

BEGIN_GNOME_DECLS

typedef struct _Gda_ServerImpl      Gda_ServerImpl;
typedef struct _Gda_ServerImplClass Gda_ServerImplClass;

#define GDA_TYPE_SERVER_IMPL            (gda_server_impl_get_type())
#define GDA_SERVER_IMPL(obj)            GTK_CHECK_CAST(obj, GDA_TYPE_SERVER_IMPL, Gda_ServerImpl)
#define GDA_SERVER_IMPL_CLASS(klass)    GTK_CHECK_CLASS_CAST(klass, GDA_TYPE_SERVER_IMPL, Gda_ServerImplClass)
#define IS_GDA_SERVER_IMPL(obj)         GTK_CHECK_TYPE(obj, GDA_TYPE_SERVER_IMPL)
#define IS_GDA_SERVER_IMPL_CLASS(klass) (GTK_CHECK_CLASS_TYPE((klass), GDA_TYPE_SERVER_IMPL))

typedef struct _Gda_ServerImplFunctions Gda_ServerImplFunctions;

typedef struct
{
  Gda_ServerImpl* server_impl;
  gchar*          dsn;
  gchar*          username;
  gchar*          password;
  GList*          commands;
  GList*          errors;
  gint            users;

  gpointer        user_data;
} Gda_ServerImplConnection;

typedef struct
{
  Gda_ServerImplConnection* cnc;
  gchar*                    text;
  gulong                    type;
  gint                      users;

  gpointer                  user_data;
} Gda_ServerImplCommand;

typedef struct
{
  Gda_ServerImplConnection* cnc;
  GList*                    fields;
  gulong                    position;
  gboolean                  at_begin;
  gboolean                  at_end;
  gint                      users;

  gpointer                  user_data;
} Gda_ServerImplRecordset;

typedef struct
{
  gchar*     name;
  gulong     sql_type;
  gshort     c_type;
  gshort     nullable;
  GDA_Value* value;
  guchar     precision;
  gshort     num_scale;
  glong      defined_length;
  glong      actual_length;
  gint       malloced;
} Gda_ServerImplField;

typedef struct
{
  gchar* description;
  glong  number;
  gchar* source;
  gchar* helpfile;
  gchar* helpctxt;
  gchar* sqlstate;
  gchar* native;
} Gda_ServerImplError;

Gda_ServerImplConnection* gda_server_impl_connection_new  (Gda_ServerImpl *server_impl);
gchar*                    gda_server_impl_connection_get_dsn (Gda_ServerImplConnection *cnc);
void                      gda_server_impl_connection_set_dsn (Gda_ServerImplConnection *cnc,
							      const gchar *dsn);
gchar*                    gda_server_impl_connection_get_username (Gda_ServerImplConnection *cnc);
void                      gda_server_impl_connection_set_username (Gda_ServerImplConnection *cnc,
								   const gchar *username);
gchar*                    gda_server_impl_connection_get_password (Gda_ServerImplConnection *cnc);
void                      gda_server_impl_connection_set_password (Gda_ServerImplConnection *cnc,
								   const gchar *password);
void                      gda_server_impl_connection_add_error (Gda_ServerImplConnection *cnc,
								Gda_ServerImplError *error);
gpointer                  gda_server_impl_connection_get_user_data (Gda_ServerImplConnection *cnc);
void                      gda_server_impl_connection_set_user_data (Gda_ServerImplConnection *cnc, gpointer user_data);
void                      gda_server_impl_connection_free (Gda_ServerImplConnection *cnc);

Gda_ServerImplCommand*    gda_server_impl_command_new  (Gda_ServerImplConnection *cnc);
Gda_ServerImplConnection* gda_server_impl_command_get_connection (Gda_ServerImplCommand *cmd);
gchar*                    gda_server_impl_command_get_text (Gda_ServerImplCommand *cmd);
void                      gda_server_impl_command_set_text (Gda_ServerImplCommand *cmd,
							    const gchar *text);
gulong                    gda_server_impl_command_get_type (Gda_ServerImplCommand *cmd);
void                      gda_server_impl_command_set_type (Gda_ServerImplCommand *cmd,
							    gulong type);
gpointer                  gda_server_impl_command_get_user_data (Gda_ServerImplCommand *cmd);
void                      gda_server_impl_command_set_user_data (Gda_ServerImplCommand *cmd,
								 gpointer user_data);
void                      gda_server_impl_command_free (Gda_ServerImplCommand *cmd);

Gda_ServerImplRecordset*  gda_server_impl_recordset_new  (Gda_ServerImplConnection *cnc);
Gda_ServerImplConnection* gda_server_impl_recordset_get_connection (Gda_ServerImplRecordset *recset);
void                      gda_server_impl_recordset_add_field (Gda_ServerImplRecordset *recset,
							       Gda_ServerImplField *field);
GList*                    gda_server_impl_recordset_get_fields (Gda_ServerImplRecordset *recset);
gboolean                  gda_server_impl_recordset_is_at_begin (Gda_ServerImplRecordset *recset);
void                      gda_server_impl_recordset_set_at_begin (Gda_ServerImplRecordset *recset,
								  gboolean at_begin);
gboolean                  gda_server_impl_recordset_is_at_end (Gda_ServerImplRecordset *recset);
void                      gda_server_impl_recordset_set_at_end (Gda_ServerImplRecordset *recset,
								  gboolean at_end);
gpointer                  gda_server_impl_recordset_get_user_data (Gda_ServerImplRecordset *recset);
void                      gda_server_impl_recordset_set_user_data (Gda_ServerImplRecordset *recset,
								   gpointer user_data);
void                      gda_server_impl_recordset_free (Gda_ServerImplRecordset *recset);

Gda_ServerImplField* gda_server_impl_field_new  (void);
void                 gda_server_impl_field_set_name (Gda_ServerImplField *field, const gchar *name);
void                 gda_server_impl_field_set_sql_type (Gda_ServerImplField *field, gulong sql_type);
void                 gda_server_impl_field_set_defined_length (Gda_ServerImplField *field, glong length);
void                 gda_server_impl_field_set_actual_length (Gda_ServerImplField *field, glong length);
void                 gda_server_impl_set_scale (Gda_ServerImplField *field, gshort scale);
void                 gda_server_impl_field_free (Gda_ServerImplField *field);

Gda_ServerImplError* gda_server_impl_error_new              (void);
gchar*               gda_server_impl_error_get_description  (Gda_ServerImplError *error);
void                 gda_server_impl_error_set_description  (Gda_ServerImplError *error,
							     const gchar *description);
glong                gda_server_impl_error_get_number       (Gda_ServerImplError *error);
void                 gda_server_impl_error_set_number       (Gda_ServerImplError *error,
							     glong number);
void                 gda_server_impl_error_set_source       (Gda_ServerImplError *error,
							     const gchar *source);
void                 gda_server_impl_error_set_help_file    (Gda_ServerImplError *error,
							     const gchar *helpfile);
void                 gda_server_impl_error_set_help_context (Gda_ServerImplError *error,
							     const gchar *helpctxt);
void                 gda_server_impl_error_set_sqlstate     (Gda_ServerImplError *error,
							     const gchar *sqlstate);
void                 gda_server_impl_error_set_native       (Gda_ServerImplError *error,
							     const gchar *native);
void                 gda_server_impl_error_free             (Gda_ServerImplError *error);

struct _Gda_ServerImplFunctions
{
  /* Connection interface */
  gboolean                 (*connection_new)(Gda_ServerImplConnection *cnc);
  gint                     (*connection_open)(Gda_ServerImplConnection *cnc,
					      const gchar *dsn,
					      const gchar *user,
					      const gchar *password);
  void                     (*connection_close)(Gda_ServerImplConnection *cnc);
  gint                     (*connection_begin_transaction)(Gda_ServerImplConnection *cnc);
  gint                     (*connection_commit_transaction)(Gda_ServerImplConnection *cnc);
  gint                     (*connection_rollback_transaction)(Gda_ServerImplConnection *cnc);
  Gda_ServerImplRecordset* (*connection_open_schema)(Gda_ServerImplConnection *cnc,
						     Gda_ServerImplError *error,
						     GDA_Connection_QType t,
						     GDA_Connection_Constraint *constraints,
						     gint length);
  gint                     (*connection_start_logging)(Gda_ServerImplConnection *cnc,
						       const gchar *filename);
  gint                     (*connection_stop_logging)(Gda_ServerImplConnection *cnc);
  gchar*                   (*connection_create_table)(Gda_ServerImplConnection *cnc,
						      GDA_RowAttributes *columns);
  gboolean                 (*connection_supports)(Gda_ServerImplConnection *cnc,
						  GDA_Connection_Feature feature);
  GDA_ValueType            (*connection_get_gda_type)(Gda_ServerImplConnection *cnc, gulong sql_type);
  gshort                   (*connection_get_c_type)(Gda_ServerImplConnection *cnc,
						    GDA_ValueType type);
  void                     (*connection_free)(Gda_ServerImplConnection *cnc);

  /* Command interface */
  gboolean                 (*command_new)(Gda_ServerImplCommand *cmd);
  Gda_ServerImplRecordset* (*command_execute)(Gda_ServerImplCommand *cmd,
					      Gda_ServerImplError *error,
					      const GDA_CmdParameterSeq *params,
					      gulong *affected,
					      gulong options);
  void                     (*command_free)(Gda_ServerImplCommand *cmd);

  /* Recordset interface */
  gboolean (*recordset_new)      (Gda_ServerImplRecordset *recset);
  gint     (*recordset_move_next)(Gda_ServerImplRecordset *recset);
  gint     (*recordset_move_prev)(Gda_ServerImplRecordset *recset);
  gint     (*recordset_close)    (Gda_ServerImplRecordset *recset);
  void     (*recordset_free)     (Gda_ServerImplRecordset *recset);

  /* Error interface */
  void (*error_make)(Gda_ServerImplError *error,
		     Gda_ServerImplRecordset *recset,
		     Gda_ServerImplConnection *cnc,
		     gchar *where);
};

/* convenience functions for calling Gda_ServerImplFunctions' functions */
gboolean                 gda_server_connection_new (Gda_ServerImplConnection *cnc);
gint                     gda_server_connection_open (Gda_ServerImplConnection *cnc,
						     const gchar *dsn,
						     const gchar *user,
						     const gchar *password);
void                     gda_server_connection_close (Gda_ServerImplConnection *cnc);
gint                     gda_server_connection_begin_transaction (Gda_ServerImplConnection *cnc);
gint                     gda_server_connection_commit_transaction (Gda_ServerImplConnection *cnc);
gint                     gda_server_connection_rollback_transaction (Gda_ServerImplConnection *cnc);
Gda_ServerImplRecordset* gda_server_connection_open_schema (Gda_ServerImplConnection *cnc,
							    Gda_ServerImplError *error,
							    GDA_Connection_QType t,
							    GDA_Connection_Constraint *constraints,
							    gint length);
gint                     gda_server_connection_start_logging (Gda_ServerImplConnection *cnc,
							      const gchar *filename);
gint                     gda_server_connection_stop_logging (Gda_ServerImplConnection *cnc);
gchar*                   gda_server_connection_create_table (Gda_ServerImplConnection *cnc,
							     GDA_RowAttributes *columns);
gboolean                 gda_server_connection_supports (Gda_ServerImplConnection *cnc,
							 GDA_Connection_Feature feature);
GDA_ValueType            gda_server_connection_get_gda_type (Gda_ServerImplConnection *cnc,
							     gulong sql_type);
gshort                   gda_server_connection_get_c_type (Gda_ServerImplConnection *cnc,
							   GDA_ValueType type);
void                     gda_server_connection_free (Gda_ServerImplConnection *cnc);

gboolean                 gda_server_command_new (Gda_ServerImplCommand *cmd);
Gda_ServerImplRecordset* gda_server_command_execute (Gda_ServerImplCommand *cmd,
						     Gda_ServerImplError *error,
						     const GDA_CmdParameterSeq *params,
						     gulong *affected,
						     gulong options);
void                     gda_server_command_free (Gda_ServerImplCommand *cmd);

gboolean gda_server_recordset_new (Gda_ServerImplRecordset *recset);
gint     gda_server_recordset_move_next (Gda_ServerImplRecordset *recset);
gint     gda_server_recordset_move_prev (Gda_ServerImplRecordset *recset);
gint     gda_server_recordset_close (Gda_ServerImplRecordset *recset);
void     gda_server_recordset_free (Gda_ServerImplRecordset *recset);

void gda_server_error_make (Gda_ServerImplError *error,
			    Gda_ServerImplRecordset *recset,
			    Gda_ServerImplConnection *cnc,
			    gchar *where);

/* Gda_ServerImpl object - interface for applications */
struct _Gda_ServerImpl
{
  GtkObject               object;
  CORBA_Object            connection_factory_obj;
  PortableServer_POA      root_poa;
  CORBA_Environment*      ev;
  gchar*                  name;
  Gda_ServerImplFunctions functions;
  GList*                  connections;
  gboolean                is_running;
};

struct _Gda_ServerImplClass
{
  GtkObjectClass parent_class;
};

GtkType         gda_server_impl_get_type (void);
Gda_ServerImpl* gda_server_impl_new      (const gchar *name, Gda_ServerImplFunctions *functions);
Gda_ServerImpl* gda_server_impl_find     (const gchar *id);
void            gda_server_impl_start    (Gda_ServerImpl *server_impl);
void            gda_server_impl_stop     (Gda_ServerImpl *server_impl);

/* for private use */
gint       gda_server_impl_exception         (CORBA_Environment *ev);
GDA_Error* gda_server_impl_make_error_buffer (Gda_ServerImplConnection *cnc);

END_GNOME_DECLS

#include <gda-server-impl-factory.h>
#include <gda-server-impl-connection.h>
#include <gda-server-impl-command.h>
#include <gda-server-impl-recordset.h>

#endif
