/* GNOME-DB
 * Copyright (c) 1998 by Rodrigo Moya
 * Copyright (c) 2000 by Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-postgres-types.h"
#include "gda-postgres-command.h"
#include "gda-postgres-connection.h"
#include <stdlib.h>
#include <time.h>

/* private functions */
/* Converting MM-DD-YYYY to struct tm */
static struct tm *
str_to_tmstruct_date(gchar *date)
{
  int day, month, year;
  char *ptr;
  char mdate[11];
  struct tm *stm;
  
  stm = (struct tm *) g_malloc(sizeof(struct tm));
  if ((date==NULL) || (*date=='\0')) 
    {
      g_free(stm);
      return NULL;
    }

  strncpy(mdate, date, 10);
  mdate[10] = '\0';
  ptr=(char *) strtok(mdate, "-/.");
  month = atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/."))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  day=atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/."))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  year=atoi(ptr);
  
  stm->tm_mday = day;
  stm->tm_mon = month -1;
  stm->tm_year = year - 1900;
  
  return stm;
}

/* private functions */
/* Converting YYYY-MM-DD to struct tm */
static struct tm *
str_to_tmstruct_date2(gchar *date)
{
  int day, month, year;
  char *ptr;
  char mdate[11];
  struct tm *stm;
  
  stm = (struct tm *) g_malloc(sizeof(struct tm));
  if ((date==NULL) || (*date=='\0')) 
    {
      g_free(stm);
      return NULL;
    }

  strncpy(mdate, date, 10);
  mdate[10] = '\0';
  ptr=(char *) strtok(mdate, "-/.");
  year = atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/."))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  month=atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/."))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  day=atoi(ptr);
  
  stm->tm_mday = day;
  stm->tm_mon = month -1;
  stm->tm_year = year - 1900;
  
  return stm;
}

/* Converting HH:MM:SS to struct tm */
static struct tm *
str_to_tmstruct_time(gchar *time)
{
  int hh, mm, ss;
  char *ptr;
  char mtime[9];
  struct tm *stm;
  
  stm = (struct tm *) g_malloc(sizeof(struct tm));
  if ((time==NULL) || (*time=='\0')) 
    {
      g_free(stm);
      return NULL;
    }

  strncpy(mtime, time, 8);
  mtime[8] = '\0';
  ptr=(char *) strtok(mtime, "-/.:");
  hh = atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/.:"))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  mm=atoi(ptr);
  if (!(ptr=(char *) strtok(NULL, "-/.:"))) 
    {
      g_free(stm);
      return NULL; /* Error */
    }
  ss=atoi(ptr);

  if ((hh>23) || (mm>60) || (ss>60))
    {
      g_free(stm);
      return NULL; /* Error */
    }

  stm->tm_hour = hh;
  stm->tm_min = mm;
  stm->tm_sec = ss;
  
  return stm;
}

/* converts a abstime like '01/26/2000 15:42:32.00 CET' to a struct tm */
static struct tm *
str_to_tmstruct_abstime(gchar *time) 
{
  int hh, mm, ss, frac;
  int day, month, year;
  char *ptr, *ptr1, *ptr2;
  char mtime[27];
  struct tm *stm;
  
  if ((time==NULL) || (*time=='\0')) 
      return NULL;

  strncpy(mtime, time, 26);
  mtime[26] = '\0';
  /* warning: do all the fractionning here because str_to_tmstruct_*
     functions use strtok as well! */
  ptr1=(char *) strtok(mtime, " "); /* the complete date */
  if (!(ptr2=(char *) strtok(NULL, ".")))  /* the complete time */
      return NULL; 
  if (!(ptr=(char *) strtok(NULL, " ")))  /* the fraction */
    return NULL; 
  frac = atoi(ptr);


  stm = str_to_tmstruct_date(ptr1);
  if (!stm)
      return NULL;
  year = stm->tm_year;
  month = stm->tm_mon;
  day = stm->tm_mday;
  g_free(stm);

  stm = str_to_tmstruct_time(ptr2);
  if (!stm)
      return NULL;
  hh = stm->tm_hour;
  mm = stm->tm_min;
  ss = stm->tm_sec;
  g_free(stm);

  stm = (struct tm *) g_malloc(sizeof(struct tm));
  stm->tm_year = year;
  stm->tm_mon = month;
  stm->tm_mday = day;
  stm->tm_hour = hh;
  stm->tm_min = mm;
  stm->tm_sec = ss;
  stm->tm_isdst = frac; /* use this field for the fraction! */

  return stm;
}

/* converts a abstime like '2000-05-12 16:43:29+02' to a struct tm */
static struct tm *
str_to_tmstruct_timestamp(gchar *time) 
{
  int hh, mm, ss;
  int day, month, year;
  char *ptr, *ptr1, *ptr2;
  char mtime[23];
  struct tm *stm;
  
  if ((time==NULL) || (*time=='\0')) 
      return NULL;

  strncpy(mtime, time, 22);
  mtime[22] = '\0';
  /* warning: do all the fractionning here because str_to_tmstruct_*
     functions use strtok as well! */
  ptr1=(char *) strtok(mtime, " "); /* the complete date */
  if (!(ptr2=(char *) strtok(NULL, ".+-")))  /* the complete time */
      return NULL; 

  stm = str_to_tmstruct_date2(ptr1);
  if (!stm)
      return NULL;
  year = stm->tm_year;
  month = stm->tm_mon;
  day = stm->tm_mday;
  g_free(stm);

  stm = str_to_tmstruct_time(ptr2);
  if (!stm)
      return NULL;
  hh = stm->tm_hour;
  mm = stm->tm_min;
  ss = stm->tm_sec;
  g_free(stm);

  stm = (struct tm *) g_malloc(sizeof(struct tm));
  stm->tm_year = year;
  stm->tm_mon = month;
  stm->tm_mday = day;
  stm->tm_hour = hh;
  stm->tm_min = mm;
  stm->tm_sec = ss;

  return stm;
}

/* 
 * takes the OID of the Postgres Type and returns the corresponding GdaType.
 * The returned string is allocated and needs to be freed.
 */
gchar * replace_PROV_TYPES_with_gdatype(Gda_POSTGRES_Recordset *recset, 
					gchar * oid)
{
  gchar *str;
  gint sqltype;

  sqltype = atoi(oid);
  str = g_strdup_printf("%d",
			gda_postgres_connection_get_gda_type(recset->cnc, 
							     sqltype));
  return str;
}


/* 
 * takes the table name and returns the SQL statement used to create the
 * table.
 * The returned string is allocated and needs to be freed.
 */
gchar * replace_TABLE_NAME_with_SQL(Gda_POSTGRES_Recordset *recset, 
				    gchar *table)
{
  GString *str;
  gchar *retval, *strptr;
  gboolean begin;
  PGresult *res, *res2;
  guint i, nrows;
  
  str = g_string_new("CREATE TABLE ");
  g_string_sprintfa(str, "%s (", table);
  begin = TRUE;
  retval = g_strdup_printf("SELECT a.attname, a.attnum, t.typname, a.attlen, "
			   "a.atttypmod, a.attnotnull, a.atthasdef "
			   "FROM pg_class c, pg_attribute a , "
			   "pg_type t WHERE c.relname = '%s' and "
			   "a.attnum > 0 AND a.attrelid = c.oid and "
			   "a.atttypid = t.oid ORDER BY attnum", table);
  res = PQexec(recset->cnc->pq_conn, retval);
  g_free(retval);
  if (!res || (PQresultStatus(res) != PGRES_TUPLES_OK))
    {
      if (res)
	PQclear(res);
      return NULL;
    }
  nrows = PQntuples(res);
  
  /* going through all the fields */
  for(i=0; i<nrows; i++)
    {
      if (begin)
	begin = FALSE;
      else
	g_string_append(str, ", ");
      strptr = PQgetvalue(res, i, 2);
      g_string_sprintfa(str, "%s %s", PQgetvalue(res, i, 0), strptr);
      if (!strcmp(strptr, "varchar") || !strcmp(strptr, "bpchar"))
	g_string_sprintfa(str, "(%d)", /*4=sizeof(int32), VARHDRSZ in pg */
			  atoi(PQgetvalue(res, i, 4))-4); 

      /* default value */
      if (*PQgetvalue(res, i, 6) == 't')
	{
	  retval = g_strdup_printf("SELECT d.adsrc FROM pg_attrdef d, "
				   "pg_class c, pg_attribute a WHERE "
				   "c.relname ='%s' and c.oid = d.adrelid "
				   "and d.adnum = a.attnum and a.attname='%s' "
				   "and a.attrelid = c.oid",
				   table, PQgetvalue(res, i, 0));
	  res2 = PQexec(recset->cnc->pq_conn, retval);
	  g_free(retval);
	  if (!res2 || (PQresultStatus(res2) != PGRES_TUPLES_OK))
	    {
	      if (res2)
		PQclear(res2);
	      PQclear(res);
	      return NULL;
	    }
	  if (PQntuples(res2)>0)
	    g_string_sprintfa(str, " DEFAULT %s", PQgetvalue(res2, 0, 0));
	  else 
	    {
	      PQclear(res2);
	      PQclear(res);
	      return NULL;
	    }
	  PQclear(res2);
	}
	  
      /* Nullable */
      if (*PQgetvalue(res, i, 5) == 't')
	g_string_append(str, " NOT NULL");
    }
  PQclear(res);
  g_string_append(str, ")");

  /* final */
  retval = str->str;
  g_string_free(str, FALSE);
  return retval;
}

/* 
 * takes the sequence name and returns the SQL statement used to create the
 * sequence.
 * The returned string is allocated and needs to be freed.
 */
gchar * replace_SEQUENCE_NAME_with_SQL(Gda_POSTGRES_Recordset *recset, 
				       gchar *seq)
{
  GString *str;
  gchar *retval, *strptr;
  gboolean begin;
  PGresult *res;
  
  str = g_string_new("CREATE SEQUENCE ");
  g_string_sprintfa(str, "%s ", seq);
  begin = TRUE;
  retval = g_strdup_printf("SELECT increment_by, min_value, "
			   "max_value, last_value, cache_value, is_cycled "
			   "FROM %s", seq);
  res = PQexec(recset->cnc->pq_conn, retval);
  g_free(retval);
  if (!res || (PQresultStatus(res) != PGRES_TUPLES_OK) ||
      (PQntuples(res) < 1))
    {
      if (res)
	PQclear(res);
      return NULL;
    }
  
  /* going through all the fields */
  g_string_sprintfa(str, "increment %s ", PQgetvalue(res, 0, 0));
  g_string_sprintfa(str, "minvalue %s ", PQgetvalue(res, 0, 1));
  g_string_sprintfa(str, "maxvalue %s ", PQgetvalue(res, 0, 2));
  g_string_sprintfa(str, "start %s ", PQgetvalue(res, 0, 3));
  g_string_sprintfa(str, "cache %s ", PQgetvalue(res, 0, 4));
  if (*PQgetvalue(res, 0, 5) == 't')
    g_string_append(str, "cycle");
  PQclear(res);

  /* final */
  retval = str->str;
  g_string_free(str, FALSE);
  return retval;
}

gchar * replace_FUNCTION_OID_with_SQL(Gda_POSTGRES_Recordset *recset, 
				      gchar *fnoid)
{
  /* FIXME: for the given OID,
   *  do a "select p.prosrc, l.lanname from pg_proc p, pg_language l "
   *       "where p.oid=%s and l.oid=p.prolang;", fnoid
   * if the second col is "internal", return NULL
   * otherwise return the first col
   */

  return NULL;
}

gchar * replace_AGGREGATE_OID_with_SQL(Gda_POSTGRES_Recordset *recset, 
				       gchar *oid)
{
  /* FIXME: find something like for the functions... */

  return NULL;
}

gchar * replace_TABLE_FIELD_with_length(Gda_POSTGRES_Recordset *recset, 
					gchar *tf)
{
  /* the format is x/y */
  gchar *retval, *buf, *ptr;
  gint i=-1, j=-1;

  buf = g_strdup(tf);
  ptr = strtok(buf, "/");
  if (ptr)
    i = atoi(ptr);
  ptr = strtok(NULL, " ");
  if (ptr)
    j = atoi(ptr);
  g_free(buf);
  if (j>i)
    i = j-4; /* 4 is the sizeof(int32) under postgres */
  retval = g_strdup_printf("%d", i);

  return retval;
}

gchar * replace_TABLE_FIELD_with_defaultval(Gda_POSTGRES_Recordset *recset, 
					    gchar *tf)
{
  gchar *retval, *buf;
  gchar *table, *field;
  PGresult *res;

  buf = g_strdup(tf);
  table = strtok(buf, " ");
  field = strtok(NULL, " ");
  if (table && field)
    {
      retval = g_strdup_printf("SELECT d.adsrc FROM pg_attrdef d, pg_class c, "
			       "pg_attribute a WHERE c.relname = '%s' and "
			       "c.oid = d.adrelid and d.adnum = a.attnum and "
			       "a.attname='%s' and a.attrelid = c.oid",
			       table, field);
      g_free(buf);
      res = PQexec(recset->cnc->pq_conn, retval);
      g_free(retval);
      if (res && (PQresultStatus(res)== PGRES_TUPLES_OK ))
	{
	  if (PQntuples(res)>0)
	    retval = g_strdup(PQgetvalue(res, 0, 0));
	  else
	    retval = NULL;
	  PQclear(res);
	  return retval;
	}
      else
	{
	  if (res)
	    PQclear(res);
	  return NULL;
	}
    }

  g_free(buf);
  return NULL;
}

/* creates a GSList with each node pointing to a string which appears
   in the tabular (space separated values).
   the created list will have to be freed, as well as its strings contents.*/
GSList *convert_tabular_to_list(gchar *tab)
{
  GSList *list;
  gchar *wtab, *ptr, *ptr2;

  list = NULL;
  wtab = g_strdup(tab);
  ptr = strtok(wtab, " ");
  while (ptr) {
    if (*ptr != '0')
      {
	ptr2 = g_strdup(ptr);
	list = g_slist_append(list, ptr2);
      }
    ptr = strtok(NULL, " ");
  }
  g_free(wtab);
  
  return list;
}


gchar * replace_TABLE_FIELD_with_iskey(Gda_POSTGRES_Recordset *recset, 
				       gchar *tf)
{
  gchar *retval, *buf;
  gchar *table, *field;
  gint pof; /* position of field */
  PGresult *res;

  buf = g_strdup(tf);
  table = strtok(buf, " ");
  field = strtok(NULL, " ");
  if (table && field)
    {
      pof = atoi(field);
      retval = g_strdup_printf("SELECT indkey, indisunique FROM pg_index "
			       "WHERE indrelid=pg_class.oid AND "
			       "pg_class.relname='%s'", 
			       table);

      res = PQexec(recset->cnc->pq_conn, retval);
      g_free(retval);
      if (res && (PQresultStatus(res)== PGRES_TUPLES_OK ))
	{
	  gint i, j=0;
	  GSList *list;

	  i = PQntuples(res);
	  retval = NULL;
	  while ((j<i) && !retval)
	    {
	      list = convert_tabular_to_list(PQgetvalue(res, j, 0));
	      while (list)
		{
		  if (pof == atoi((gchar *)(list->data)))
		    retval = g_strdup("t");
		  if (list->data)
		    g_free(list->data);
		  list = g_slist_remove_link(list, list);
		}
	      j++;
	    }
	  if (!retval)
	    retval = g_strdup("f");
	  PQclear(res);
	}
      else
	{
	  if (res)
	    PQclear(res);
	  retval = g_strdup("f");
	}
    }
  else
    retval = g_strdup("f");

  g_free(buf);
  return retval;
}


static gboolean
fill_field_values (Gda_POSTGRES_Recordset *recset, glong row)
{
  register gint cnt = 0;
  GList *node;
  GSList *repl_list;
  Gda_POSTGRES_Recordset_Replacement *repl;
  /* check parameters */
  g_return_val_if_fail(recset != NULL, FALSE);
  g_return_val_if_fail((recset->pq_data != NULL) || (recset->btin_res != NULL), FALSE);
  /* traverse all recordset values */

  node = g_list_first(recset->fields);

  fprintf(stderr,"getting data from row %ld\n", row);
  while (node != NULL)
    {
      Gda_POSTGRES_Field *field = (Gda_POSTGRES_Field *) node->data;
      struct tm *stm;

      if (field != 0)
        {
	  /* 
	   * field value (GDA_Value) 
	   */
          gchar *native_value=NULL, *tmpstr;
	  /* is there a replacement for that col? */
	  repl_list = recset->replacements;
	  repl = NULL;
	  while (repl_list && !repl)
	    {
	      if (((Gda_POSTGRES_Recordset_Replacement *)(repl_list->data))
		  ->colnum == cnt)
		repl = (Gda_POSTGRES_Recordset_Replacement *)
		  (repl_list->data);
	      else
		repl_list = g_slist_next(repl_list);
	    }

	  if (recset->cnc)
	    {
	      if (repl)
		field->value->_d = repl->newtype;
	      else
		field->value->_d = 
		  gda_postgres_connection_get_gda_type(recset->cnc, 
						       field->sql_type);
	    }
	  else
	    field->value->_d = GDA_TypeVarchar;

          /* get field's value from server */
	  if(recset->pq_data)
	    {
	      if (PQgetisnull(recset->pq_data, row, cnt))
		tmpstr = NULL;
	      else
		tmpstr = PQgetvalue(recset->pq_data, row, cnt);
	    }
	  else
	    tmpstr = Gda_Builtin_Result_get_value(recset->btin_res, row, cnt);
	  if (tmpstr)
	    {
	      if (repl)
		native_value = (*repl->trans_func)
		  (recset, tmpstr);
	      else
		native_value = g_strdup(tmpstr);
	    }
	  else
	    native_value = NULL;

	  switch (field->value->_d) 
	    {
	    case GDA_TypeBoolean:
	      if (native_value)
		{
		  if (*native_value=='t')
		    field->value->_u.b = TRUE;
		  else
		    field->value->_u.b = FALSE;
		  field->actual_length = sizeof(CORBA_boolean);
		}
	      else
		{
		    field->value->_u.b = FALSE;
		    field->actual_length = 0;
		}
	      break;

	    case GDA_TypeDbDate:
	      /* the date is DD-MM-YYYY, as set at connection opening */
	      stm = NULL;
	      if (native_value)
		{
		  stm = str_to_tmstruct_date(native_value);
		  if (stm)
		    {
		      field->value->_u.dbd.year = stm->tm_year + 1900;
		      field->value->_u.dbd.month = stm->tm_mon + 1;
		      field->value->_u.dbd.day = stm->tm_mday;
		    }
		  field->actual_length = sizeof(GDA_DbDate);
		}
	      if (!native_value || !stm)
		{
		  field->value->_u.dbd.year = 0;
		  field->value->_u.dbd.month = 0;
		  field->value->_u.dbd.day = 0;
		  field->actual_length = 0;
		}
	      if (stm)
		g_free(stm);
	      break;

	    case GDA_TypeDbTime:
	      stm = NULL;
	      if (native_value)
		{
		  stm = str_to_tmstruct_time(native_value);
		  if (stm)
		    {
		      field->value->_u.dbt.hour = stm->tm_hour;
		      field->value->_u.dbt.minute = stm->tm_min;
		      field->value->_u.dbt.second = stm->tm_sec;
		    }
		  field->actual_length = sizeof(GDA_DbTime);
		}
	      if (!native_value || !stm)
		{
		  field->value->_u.dbt.hour = 0;
		  field->value->_u.dbt.minute = 0;
		  field->value->_u.dbt.second = 0;
		  field->actual_length = 0;
		}

	      if (stm)
		g_free(stm);
	      break;

	    case GDA_TypeDbTimestamp: 
	      /* the output format for abstime and datetime will be
		 for ex: 05/27/2000 15:26:36.00 CET
		 and the one of timestamp for ex:
		 2000-05-12 16:43:29+02
		 because the DATESTYLE is set in gda-postgres-connection.c
		 to SQL with US (NonEuropean) conventions. */
	      field->value->_u.dbtstamp.year = 0;
	      field->value->_u.dbtstamp.month = 0;
	      field->value->_u.dbtstamp.day = 0;
	      field->value->_u.dbtstamp.hour = 0;
	      field->value->_u.dbtstamp.minute = 0;
	      field->value->_u.dbtstamp.second = 0;
	      field->value->_u.dbtstamp.fraction = 0;
	      if (! native_value)
		{
		  field->actual_length = 0;
		  break;
		}
	      field->actual_length = sizeof(GDA_DbTimestamp);

	      if ((field->sql_type == gda_postgres_connection_get_sql_type
		  (recset->cnc, "abstime")) ||
		  (field->sql_type == gda_postgres_connection_get_sql_type
		  (recset->cnc, "datetime")))
		{
		  stm = str_to_tmstruct_abstime(native_value);

		  if (!stm)
		    break;
		  field->value->_u.dbtstamp.year = stm->tm_year+1900;
		  field->value->_u.dbtstamp.month = stm->tm_mon+1;
		  field->value->_u.dbtstamp.day = stm->tm_mday;
		  field->value->_u.dbtstamp.hour = stm->tm_hour;
		  field->value->_u.dbtstamp.minute = stm->tm_min;
		  field->value->_u.dbtstamp.second = stm->tm_sec;
		  field->value->_u.dbtstamp.fraction = stm->tm_isdst; /*!*/
		  g_free(stm);
		  break;
		}
	      if (field->sql_type == gda_postgres_connection_get_sql_type
		  (recset->cnc, "timestamp"))
		{
		  stm = str_to_tmstruct_timestamp(native_value);

		  if (!stm)
		    break;
		  field->value->_u.dbtstamp.year = stm->tm_year+1900;
		  field->value->_u.dbtstamp.month = stm->tm_mon+1;
		  field->value->_u.dbtstamp.day = stm->tm_mday;
		  field->value->_u.dbtstamp.hour = stm->tm_hour;
		  field->value->_u.dbtstamp.minute = stm->tm_min;
		  field->value->_u.dbtstamp.second = stm->tm_sec;
		  g_free(stm);
		  break;
		}
	      break;

	    case GDA_TypeSmallint:
	      if (native_value)
		{
		  field->actual_length = sizeof(CORBA_short);
		  field->value->_u.si = atoi(native_value);
		}
	      else
		{
		  field->actual_length = 0;
		  field->value->_u.si = 0;
		}
	      break;

	    case GDA_TypeInteger:
	      if (native_value)
		{
		  field->actual_length = sizeof(CORBA_short);
		  field->value->_u.i = atol(native_value);
		}
	      else
		{
		  field->actual_length = 0;
		  field->value->_u.i = 0;
		}
	      break;

	    case GDA_TypeLongvarchar:
	    case GDA_TypeChar:
	    case GDA_TypeVarchar:
	      if (native_value)
		{
		  if (recset->pq_data)
		    field->actual_length = PQgetlength(recset->pq_data, 
						       row, cnt);
		  else
		    field->actual_length = 
		      Gda_Builtin_Result_get_length(recset->btin_res, 
						    row, cnt);
		  /* FIXME: where is it de-allocated? */
		  field->value->_u.lvc = CORBA_string_dup(native_value);
		  field->malloced = TRUE;
		}
	      else
		{
		  field->actual_length = 0;
		  field->value->_u.lvc = 0;
		}
	      break;

	    case GDA_TypeSingle:
	      if (native_value)
		{
		  field->actual_length = sizeof(CORBA_float);
		  field->value->_u.f = atof(native_value);
		}
	      else
		{
		  field->actual_length = 0;
		  field->value->_u.f = 0;
		}
	      break;

	    case GDA_TypeDouble:
	      if (native_value)
		{
		  field->actual_length = sizeof(CORBA_double);
		  field->value->_u.dp = atof(native_value);
		}
	      else
		{
		  field->actual_length = 0;
		  field->value->_u.dp = 0;
		}
	      break;

	    case GDA_TypeVarbin: 
	      /* FIXME:
		 the IDL is typedef sequence<octet> VarBinString;
		 and the struct:
		 typedef struct {
		   CORBA_unsigned_long _maximum,
		                       _length;
		   CORBA_octet *_buffer;
		   CORBA_boolean _release;
		 } CORBA_sequence_CORBA_octet;
	      */
	      field->value->_u.lvb._buffer = NULL;
	      field->value->_u.lvb._maximum = 0;
	      field->value->_u.lvb._length = 0;
	      field->actual_length = sizeof(GDA_VarBinString);
	      break;

	    default:
	      /* should not happen since the default value for field->value->_d
		 is GDA_TypeVarchar. */
	      break;
	    }
	  if (native_value)
	    g_free(native_value);
	  /*
	   * c_type 
	   */
	  field->c_type = 
	    gda_postgres_connection_get_c_type_from_gda(recset->cnc,
							field->value->_d);
	  /* FIXME:
	     What about the nullable, precision, num_scale ? */
	}
      node = g_list_next(node);
      cnt++;
    }

  return (TRUE);
}

/* create a new result set object */
Gda_POSTGRES_Recordset *
gda_postgres_recset_new (void)
{
  Gda_POSTGRES_Recordset *rc = g_new0(Gda_POSTGRES_Recordset, 1);
  memset(rc, 0, sizeof(Gda_POSTGRES_Recordset));
  rc->replacements = NULL;
  return (rc);
}

/* free Recordset object */
void
gda_postgres_recset_free (Gda_POSTGRES_Recordset *recset)
{
  g_return_if_fail(recset != NULL);
  gda_postgres_recset_close(recset);
  g_free((gpointer) recset);
}

/* cursor movement */
gint
gda_postgres_recset_move_next (Gda_POSTGRES_Recordset *recset)
{
  gint ntuples;

  g_return_val_if_fail(recset != NULL, -1);

  if (recset->pq_data)
    ntuples = PQntuples(recset->pq_data);
  else
    ntuples = Gda_Builtin_Result_get_nbtuples(recset->btin_res);

  /* check if we can fetch forward */
  if (recset->pos < ntuples)
    {
      if (fill_field_values(recset, recset->pos))
        {
          recset->at_end = recset->at_begin = 0;
          recset->pos++;
          return (0);
        }
    }
  else
    {
      recset->at_end = 1;
      return (1);
    }
  return (-1);
}

gint
gda_postgres_recset_move_prev (Gda_POSTGRES_Recordset *recset)
{
  g_return_val_if_fail(recset != NULL, -1);

  if (recset->pos > 0)
    {
      if (fill_field_values(recset, recset->pos - 1))
        {
          recset->at_begin = recset->at_end = 0;
          recset->pos--;
          return (0);
        }
    }
  else
    {
      recset->at_begin = 1;
      return (1);
    }
  return (-1);
}

/* close given recordset */
gint
gda_postgres_recset_close (Gda_POSTGRES_Recordset *recset)
{
  GList *ptr;

  g_return_val_if_fail(recset != NULL, -1);
  /* free associated command object */
  if (recset->pq_data != NULL)
    {
      PQclear(recset->pq_data);
      recset->pq_data = 0;
    }

  /* if there is a builtin result, free it */
  if (recset->btin_res)
    Gda_Builtin_Result_free(recset->btin_res);

  /* free fields' list */
  ptr = recset->fields;
  while (ptr != NULL)
    {
      gda_postgres_field_free((gpointer) ptr->data);
      ptr = g_list_next(ptr);
    }
  g_list_free(recset->fields);
  recset->fields = NULL;
  /* free the list of replacements */
  while (recset->replacements)
    {
      g_free(recset->replacements->data);
      recset->replacements = g_slist_remove_link(recset->replacements,
						 recset->replacements);
    }
  return (0);
}

/* debugging */
void
gda_postgres_recset_dump (Gda_POSTGRES_Recordset *recset)
{
}
