/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include <mysql.h>
#include <glib.h>

#include "gda-mysql-error.h"
#include "gda-mysql-connection.h"
#include "gda-mysql-recset.h"

typedef Gda_MYSQL_Recordset* (*schema_ops_fn) (Gda_MYSQL_Recordset*,
                                               Gda_MYSQL_Connection*,
                                               GDA_Connection_Constraint* constraints,
                                               gint length);

schema_ops_fn schema_ops[GDA_Connection_GDCN_SCHEMA_LAST] = { 0 };

static Gda_MYSQL_Recordset* schema_tables(Gda_MYSQL_Recordset* recset,
                                          Gda_MYSQL_Connection*,
                                          GDA_Connection_Constraint* constraints,
                                          gint length);

static Gda_MYSQL_Recordset* schema_views(Gda_MYSQL_Recordset* recset,
                                         Gda_MYSQL_Connection*,
                                         GDA_Connection_Constraint* constraints,
                                         gint length);
static Gda_MYSQL_Recordset* schema_cols(Gda_MYSQL_Recordset* recset,
                                        Gda_MYSQL_Connection*,
                                        GDA_Connection_Constraint* constraints,
                                        gint length);

void           initialize_schema_ops(void);

void
initialize_schema_ops(void)
{
  memset((gpointer) schema_ops, 0, sizeof(schema_ops));

  schema_ops[GDA_Connection_GDCN_SCHEMA_TABLES] = schema_tables;
  schema_ops[GDA_Connection_GDCN_SCHEMA_VIEWS]  = schema_views;
  schema_ops[GDA_Connection_GDCN_SCHEMA_COLS]   = schema_cols;
}


gint SQLTABLES_order[]={2};

static MYSQL_FIELD SQLTABLES_fields[] = {
  {"Table_qualifer","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,0},
  {"Table_owner","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,0},
  {"Table_name","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,NAME_LEN,0},
  {"Table_type","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,5,0},
  {"Remarks","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,11}};

char *SQLTABLES_values[]={"","",NULL,"TABLE","MySQL table"};

static MYSQL_ROW
fix_fields_copy(Gda_MYSQL_Recordset* rs, MYSQL_ROW row)
{
  /* Dunno what this does yet - cdw */
  gint i;
  for (i = 0 ; i < rs->order_count; i++)
    rs->array[rs->order[i]]=row[i];
  return rs->array;
}

static void
link_fields(MYSQL_RES* result, MYSQL_FIELD* fields, gint field_count)
{
  /* Dunno what this does yet - cdw */
  result->fields = fields;
  result->field_count = field_count;
  result->current_field = 0;
}

static Gda_MYSQL_Recordset*
schema_tables(Gda_MYSQL_Recordset* recset, Gda_MYSQL_Connection* cnc,
	      GDA_Connection_Constraint* constraints, gint length)
{
  /* Dunno what this does yet - cdw */
  recset->cnc = cnc;
  
  recset->mysql_res = mysql_list_tables(recset->cnc->mysql, 0);
  link_fields(recset->mysql_res, SQLTABLES_fields, 5);
  recset->order = SQLTABLES_order;
  recset->order_count = 1;
  recset->array = g_new0(gchar*, 5);
  memcpy(recset->array, SQLTABLES_values, sizeof(SQLTABLES_values));
  recset->fix_fields = fix_fields_copy;
  recset->mysql_res->lengths = g_new0(gint, 5); 
  return recset;
}

static Gda_MYSQL_Recordset*
schema_views(Gda_MYSQL_Recordset* recset, Gda_MYSQL_Connection* cnc,
	      GDA_Connection_Constraint* constraints, gint length)
{
  /* Dunno what this does yet - cdw */
  return recset;
}

static Gda_MYSQL_Recordset*
schema_cols(Gda_MYSQL_Recordset* recset, Gda_MYSQL_Connection* cnc,
	      GDA_Connection_Constraint* constraints, gint length)
{
  /* Dunno what this does yet - cdw */
  return recset;
}


Gda_MYSQL_Connection*
gda_mysql_connection_new()
{
  /* this creates and initializes the connection var - cdw */
  static gint initialized = 0;
  
  Gda_MYSQL_Connection* c = g_new0(Gda_MYSQL_Connection, 1);

  if (!initialized)
    {
      initialize_schema_ops();
      initialized = 1;
    }
  return c;
}


void
gda_mysql_connection_free(Gda_MYSQL_Connection* c)
{
  /* this frees the Gda_MYSQL_Connection safely - cdw */
  if (c->host)
    g_free(c->host);
  if (c->db)
    g_free(c->db);
  if (c->port)
    g_free(c->port);
  if (c->unix_socket)
    g_free(c->unix_socket);
  if (c->flags)
    g_free(c->flags);
  if (c->mysql)
    {
      gda_mysql_connection_close(c);
      g_free(c->mysql);
    }
  if (c->user)
    g_free(c->user);
  if (c->password)
    g_free(c->password);
  g_free(c);
}


static gchar*
get_value(gchar* ptr)
{
  /* Okay, this function takes "a = c" and returns c.
   * Note that it trims beginning space before the c, but not after. 
   * cdw */
  while (*ptr && *ptr != '=')
    ptr++;
  
  if (!*ptr)
    return 0;
  ptr++;
  if (!*ptr)
    return 0;
  
  while (*ptr && isspace(*ptr))
    ptr++;

  return g_strdup(ptr);
}


gint
gda_mysql_connection_open(Gda_MYSQL_Connection* cnc,
			  const gchar*          cnc_string,
			  const gchar*          user,
			  const gchar*          passwd)
{
  /* Opens the connection THAT IS ALREADY CREATED - cdw */
  gchar* ptr_s;
  gchar* ptr_e;
  MYSQL* rc;
#if MYSQL_VERSION_ID < 32200
  int    err;
#endif
  
  ptr_s = (gchar*)cnc_string;
  while (ptr_s && *ptr_s)
    {
      ptr_e = strchr(ptr_s, ';');

      if (ptr_e)
	*ptr_e = '\0';
      if (strncasecmp(ptr_s, "HOST", strlen("HOST")) == 0)
        cnc->host = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "DATABASE", strlen("DATABASE")) == 0)
        cnc->db = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "USERNAME", strlen("USERNAME")) == 0)
        cnc->user = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PASSWORD", strlen("PASSWORD")) == 0)
        cnc->password = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PORT", strlen("PORT")) == 0)
        cnc->port = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "UNIX_SOCKET", strlen("UNIX_SOCKET")) == 0)
        cnc->unix_socket = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "FLAGS", strlen("FLAGS")) == 0)
        cnc->flags = get_value(ptr_s);
      ptr_s = ptr_e;
      if (ptr_s)
	ptr_s++;
    }

  /* Okay, now let's copy the overriding user and/or password if applicable */
  if (user) {
    if (cnc->user) g_free(cnc->user); /* deallocate old one if not null */
    cnc->user = g_strdup(user);
  }
  if (passwd) {
    if (cnc->password) g_free(cnc->password); /* deallocate old one if not null */
    cnc->password = g_strdup(passwd);
  }

  /* We can't have both a host/port pair AND a unix_socket, if both are provided, error out now... */
  if ((cnc->host || cnc->port) && cnc->unix_socket) {
      gda_mysql_error_ours_make(cnc, "gda_mysql_connection_open","You cannot provide a UNIX_SOCKET if you also provide either a HOST or a PORT. Please remove the UNIX_SOCKET, or remove both the HOST and PORT options.");
      return -1;
  }

  /* Okay, now let's set the default of localhost:3306 if neither is provided... */
  if (!cnc->unix_socket) {
    if (!cnc->port) cnc->port = g_strdup("3306");
    if (!cnc->host) cnc->host = g_strdup("localhost");
  }

  fprintf(stderr,"cnc->user = %s\n", cnc->user);
  fprintf(stderr,"cnc->password = %s\n", cnc->password);
  fprintf(stderr,"cnc->host = %s\n", cnc->host);
  fprintf(stderr,"cnc->port = %s\n", cnc->port);
  fprintf(stderr,"cnc->unix_socket = %s\n", cnc->unix_socket);

  cnc->mysql = g_new0(MYSQL, 1);
  fprintf(stderr,"cnc->mysql = %p\n", cnc->mysql);
  fprintf(stderr,"** Calling mysql_real_connect\n");

#if MYSQL_VERSION_ID >= 32200
  rc = mysql_real_connect(cnc->mysql, cnc->host,
			  cnc->user,
			  cnc->password,
			  cnc->db,
			  cnc->port ? atoi(cnc->port) : 0,
			  cnc->unix_socket,
			  cnc->flags ? atoi(cnc->flags) : 0);
#else
  rc = mysql_real_connect(cnc->mysql, cnc->host,
			  cnc->user,
			  cnc->password,
			  cnc->port ? atoi(cnc->port) : 0,
			  cnc->unix_socket,
			  cnc->flags ? atoi(cnc->flags) : 0);
#endif

  fprintf(stderr,"cnc->mysql after connect = %p\n", cnc->mysql);
  
  if (!rc)
    {
      gda_mysql_error_make(cnc, "gda_mysql_connection_open");
      return -1;
    }

#if MYSQL_VERSION_ID < 32200
  fprintf(stderr,"before mysql_select_db: cnc->db = '%s'\n", cnc->db);
  fprintf(stderr,"cnc->mysql = %p\n", cnc->mysql);
  fprintf(stderr,"*** Calling mysql_select_db\n");
  err = mysql_select_db(cnc->mysql, cnc->db);
  fprintf(stderr,"mysql_select_db returns %d\n", err);
  if (err != 0)
    {
      fprintf(stderr,"mysql errno  = %d\n",mysql_errno(cnc->mysql));
      fprintf(stderr,"mysql errmsg = '%s'\n",
	      mysql_error(cnc->mysql));
      return -1;
    }
#endif

  fprintf(stderr,"gda_mysql_connection_open: returning\n");
  return 0;
}

Gda_MYSQL_Recordset*
gda_mysql_connection_open_schema (Gda_MYSQL_Connection* cnc,
				  Gda_MYSQL_Error* e,
				  GDA_Connection_QType t,
				  GDA_Connection_Constraint* constraints,
				  gint length)
{
  /* Dunno what this does yet - cdw */
  Gda_MYSQL_Recordset* rc = gda_mysql_recset_new();
  schema_ops_fn        fn = schema_ops[t];

  if (fn)
    {
      return fn(rc, cnc, constraints, length);
    }
  else
    {
      g_warning("connection_open_schema: Unhandled SCHEMA_QTYPE %d\n", t);
    }
  return rc;
}


gint
gda_mysql_connection_close(Gda_MYSQL_Connection* c)
{
  /* closes the connection, but doesn't free the variable - cdw */
  fprintf(stderr, "Calling mysql_close\n");
  mysql_close(c->mysql);
  return 0;
}

gboolean
gda_mysql_connection_supports (Gda_MYSQL_Connection *cnc, GDA_Connection_Feature feature)
{
  /* tells us if we support various abilities - cdw */
  g_return_val_if_fail(cnc != 0, FALSE);

  switch (feature)
    {
    case GDA_Connection_FEATURE_TRANSACTIONS : return FALSE;
    case GDA_Connection_FEATURE_SEQUENCES : return TRUE; /* FIXME: ??? */
    case GDA_Connection_FEATURE_INHERITANCE : return FALSE;
    case GDA_Connection_FEATURE_PROCS : return FALSE;
    default :
      fprintf(stderr, "Invalid GDA_Connection_Feature %d", feature);
    }
  return FALSE;
}
