/* GNOME DB components libary
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome-db-control.h>
#include <gnome-db-factory.h>

typedef struct
{
  guint    id;
  gchar*   name;
  gchar*   desc;
  guint    type;
  gpointer value;
  gboolean allocated;
} instance_prop_data_t;

static void gnome_db_control_class_init (GnomeDbControlClass *klass);
static void gnome_db_control_init       (GnomeDbControl *control);

static void gnome_db_control_destroy    (GnomeDbControl *control);

static instance_prop_data_t* property_find_by_id   (GnomeDbControlInstance *control_instance,
						    guint id);
static instance_prop_data_t* property_find_by_name (GnomeDbControlInstance *control_instance,
						    const gchar *name);

static void set_frame_cb (BonoboControl *bonobo_control, GnomeDbControlInstance *control_instance);

/*
 * GnomeDbControl object signals
 */
enum
{
  NEW_INSTANCE,
  GET_PROPERTY,
  SET_PROPERTY,
  LAST_SIGNAL
};

static gint gnome_db_control_signals[LAST_SIGNAL] = { 0, 0, 0};

/*
 * Private functions
 */
static void
get_control_prop (BonoboPropertyBag *bag,
		  BonoboArg *arg,
		  guint arg_id,
		  GnomeDbControlInstance *control_instance)
{
  instance_prop_data_t* prop_data;

  g_return_if_fail(BONOBO_IS_PROPERTY_BAG(bag));
  g_return_if_fail(control_instance != NULL);

  prop_data = property_find_by_id(control_instance, arg_id);
  if (prop_data)
    {
      gtk_signal_emit(GTK_OBJECT(control_instance->control),
		      gnome_db_control_signals[GET_PROPERTY],
		      control_instance,
		      prop_data->name);

      switch (prop_data->type)
	{
	case GNOME_DB_CONTROL_ARG_STRING :
	  if (prop_data->value)
	    BONOBO_ARG_SET_STRING(arg, prop_data->value);
	  else BONOBO_ARG_SET_STRING(arg, "");
	  break;
	case GNOME_DB_CONTROL_ARG_BOOLEAN :
	  {
	    gboolean value;
	    if (prop_data->value)
	      {
		memcpy((gpointer) &value, prop_data->value, sizeof(gboolean));
		BONOBO_ARG_SET_BOOLEAN(arg, value);
	      }
	    else BONOBO_ARG_SET_BOOLEAN(arg, FALSE);
	    break;
	  }
	case GNOME_DB_CONTROL_ARG_INT :
	  {
	    gint value;
	    if (prop_data->value)
	      {
		memcpy((gpointer) &value, prop_data->value, sizeof(gint));
		BONOBO_ARG_SET_INT(arg, value);
	      }
	    else BONOBO_ARG_SET_INT(arg, 0);
	    break;
	  }
	case GNOME_DB_CONTROL_ARG_LONG :
	  {
	    glong value;
	    if (prop_data->value)
	      {
		memcpy((gpointer) &value, prop_data->value, sizeof(glong));
		BONOBO_ARG_SET_LONG(arg, value);
	      }	    
	    else BONOBO_ARG_SET_LONG(arg, 0);
	    break;
	  }
	case GNOME_DB_CONTROL_ARG_FLOAT :
	  {
	    gfloat value;
	    if (prop_data->value)
	      {
		memcpy((gpointer) &value, prop_data->value, sizeof(gfloat));
		BONOBO_ARG_SET_FLOAT(arg, value);
	      }
	    else BONOBO_ARG_SET_FLOAT(arg, 0.0);
	    break;
	  }
	case GNOME_DB_CONTROL_ARG_DOUBLE :
	  {
	    gdouble value;
	    if (prop_data->value)
	      {
		memcpy((gpointer) &value, prop_data->value, sizeof(gdouble));
		BONOBO_ARG_SET_DOUBLE(arg, value);
	      }
	    else BONOBO_ARG_SET_DOUBLE(arg, 0.0);
	    break;
	  }
	}
    }
}

static void
set_control_prop (BonoboPropertyBag *bag,
		  const BonoboArg *arg,
		  guint arg_id,
		  GnomeDbControlInstance *control_instance)
{
  instance_prop_data_t* prop_data;

  g_return_if_fail(BONOBO_IS_PROPERTY_BAG(bag));
  g_return_if_fail(control_instance != NULL);

  prop_data = property_find_by_id(control_instance, arg_id);
  if (prop_data)
    {
      switch (prop_data->type)
	{
	case GNOME_DB_CONTROL_ARG_STRING :
	  if (prop_data->allocated) g_free((gpointer) prop_data->value);
	  prop_data->value = g_strdup(BONOBO_ARG_GET_STRING(arg));
	  prop_data->allocated = TRUE;
	  break;
	case GNOME_DB_CONTROL_ARG_BOOLEAN :
	  {
	    gboolean value = BONOBO_ARG_GET_BOOLEAN(arg);
	    if (prop_data->allocated) g_free((gpointer) prop_data->value);
	    prop_data->value = g_malloc(sizeof(gboolean));
	    memcpy(prop_data->value, &value, sizeof(gboolean)); 
	    prop_data->allocated = TRUE;
	    break;
	  }
	case GNOME_DB_CONTROL_ARG_INT :
	  {
	    gint value = BONOBO_ARG_GET_INT(arg);
	    if (prop_data->allocated) g_free((gpointer) prop_data->value);
	    prop_data->value = g_malloc(sizeof(gint));
	    memcpy(prop_data->value, &value, sizeof(gint));
	    prop_data->allocated = TRUE;
	    break;
	  }
	case GNOME_DB_CONTROL_ARG_LONG :
	  {
	    glong value = BONOBO_ARG_GET_LONG(arg);
	    if (prop_data->allocated) g_free((gpointer) prop_data->value);
	    prop_data->value = g_malloc(sizeof(glong));
	    memcpy(prop_data->value, &value, sizeof(glong));
	    prop_data->allocated = TRUE;
	    break;
	  }
	case GNOME_DB_CONTROL_ARG_FLOAT :
	  {
	    gfloat value = BONOBO_ARG_GET_FLOAT(arg);
	    if (prop_data->allocated) g_free((gpointer) prop_data->value);
	    prop_data->value = g_malloc(sizeof(gfloat));
	    memcpy(prop_data->value, &value, sizeof(gfloat));
	    prop_data->allocated = TRUE;
	    break;
	  }
	case GNOME_DB_CONTROL_ARG_DOUBLE :
	  {
	    gdouble value = BONOBO_ARG_GET_DOUBLE(arg);
	    if (prop_data->allocated) g_free((gpointer) prop_data->value);
	    prop_data->value = g_malloc(sizeof(gdouble));
	    memcpy(prop_data->value, &value, sizeof(gdouble));
	    prop_data->allocated = TRUE;
	    break;
	  }
	}
      gtk_signal_emit(GTK_OBJECT(control_instance->control),
		      gnome_db_control_signals[SET_PROPERTY],
		      control_instance,
		      prop_data->name, prop_data->value);
    }
}

static BonoboObject *
control_factory (BonoboGenericFactory *factory, gpointer data)
{
  GnomeDbControl*         control;
  GnomeDbControlInstance* control_instance;
  GtkWidget*              widget = NULL;
  BonoboPropertyBag*      pb;

  g_return_val_if_fail(BONOBO_IS_GENERIC_FACTORY(factory), CORBA_OBJECT_NIL);
  g_return_val_if_fail(GNOME_DB_IS_CONTROL(data), CORBA_OBJECT_NIL);

  control = GNOME_DB_CONTROL(data);

  if (control->create_widget_cb) widget = control->create_widget_cb();

  if (!GTK_IS_WIDGET(widget))
    {
      g_warning(_("%s:%d: Could not create control widget"), __FILE__, __LINE__);
      return CORBA_OBJECT_NIL;
    }

  /* add created widget to list */
  control_instance = g_new0(GnomeDbControlInstance, 1);
  control_instance->control = control;
  control_instance->widget = widget;
  control->widgets = g_list_append(control->widgets, (gpointer) control_instance);

  /* create bonobo control */
  control_instance->bonobo_control = bonobo_control_new(control_instance->widget);

  pb = bonobo_property_bag_new((BonoboPropertyGetFn) get_control_prop,
			       (BonoboPropertySetFn) set_control_prop,
			       (gpointer) control_instance);
  bonobo_control_set_property_bag(control_instance->bonobo_control, pb);

  /* FIXME: should I add_interface, or just return bonobo_control? */
  //bonobo_object_add_interface(BONOBO_OBJECT(control),
  //			      BONOBO_OBJECT(bonobo_control));
  gtk_signal_connect(GTK_OBJECT(control_instance->bonobo_control),
		     "set_frame",
		     GTK_SIGNAL_FUNC(set_frame_cb),
		     (gpointer) control_instance);

  gtk_signal_emit(GTK_OBJECT(control),
		  gnome_db_control_signals[NEW_INSTANCE],
		  control_instance);

  // return BONOBO_OBJECT(control);
  return BONOBO_OBJECT(control_instance->bonobo_control);
}

static instance_prop_data_t *
property_find_by_id (GnomeDbControlInstance *control_instance, guint id)
{
  GList* names;

  g_return_val_if_fail(control_instance != NULL, NULL);

  for (names = g_list_first(control_instance->properties); names; names = g_list_next(names))
    {
      instance_prop_data_t* prop_data = (instance_prop_data_t *) names->data;
      if (prop_data && prop_data->id == id) return prop_data;
    }
  return NULL;
}

static instance_prop_data_t *
property_find_by_name (GnomeDbControlInstance *control_instance, const gchar *name)
{
  GList* names;

  g_return_val_if_fail(control_instance != NULL, NULL);
  g_return_val_if_fail(name != NULL, NULL);

  for (names = g_list_first(control_instance->properties); names; names = g_list_next(names))
    {
      instance_prop_data_t* prop_data = (instance_prop_data_t *) names->data;
      if (prop_data && !g_strcasecmp(prop_data->name, name)) return prop_data;
    }
  return NULL;
}

/*
 * Callbacks
 */
static void
set_frame_cb (BonoboControl *bonobo_control, GnomeDbControlInstance *control_instance)
{
  Bonobo_UIHandler remote_uih;
  BonoboUIHandler* uih;

  g_return_if_fail(BONOBO_IS_CONTROL(bonobo_control));
  g_return_if_fail(control_instance != NULL);

  remote_uih = bonobo_control_get_remote_ui_handler(bonobo_control);
  uih = bonobo_control_get_ui_handler(bonobo_control);
  bonobo_ui_handler_set_container(uih, remote_uih);

  gtk_widget_show_all(control_instance->widget);
}

/*
 * GnomeDbControl object interface
 */
static void
gnome_db_control_class_init (GnomeDbControlClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *) klass;

  gnome_db_control_signals[NEW_INSTANCE] =
    gtk_signal_new("new_instance",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GnomeDbControlClass, new_instance),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  gnome_db_control_signals[GET_PROPERTY] =
    gtk_signal_new("get_property",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GnomeDbControlClass, get_property),
		   gtk_marshal_NONE__POINTER_POINTER,
		   GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_STRING);
  gnome_db_control_signals[SET_PROPERTY] =
    gtk_signal_new("set_property",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GnomeDbControlClass, set_property),
		   gtk_marshal_NONE__POINTER_POINTER_POINTER,
		   GTK_TYPE_NONE, 3, GTK_TYPE_POINTER, GTK_TYPE_STRING,
		   GTK_TYPE_POINTER);


  gtk_object_class_add_signals(object_class, gnome_db_control_signals, LAST_SIGNAL);
  object_class->destroy = gnome_db_control_destroy;

  klass->new_instance = NULL;
  klass->get_property = NULL;
  klass->set_property = NULL;
}

static void
gnome_db_control_init (GnomeDbControl *control)
{
  g_return_if_fail(GNOME_DB_IS_CONTROL(control));

  control->factory = NULL;
  control->create_widget_cb = NULL;
  control->widgets = NULL;
}

GtkType
gnome_db_control_get_type (void)
{
  static GtkType db_control_type = 0;

  if (!db_control_type)
    {
      GtkTypeInfo db_control_info =
      {
	"GnomeDbControl",
	sizeof (GnomeDbControl),
	sizeof (GnomeDbControlClass),
	(GtkClassInitFunc) gnome_db_control_class_init,
	(GtkObjectInitFunc) gnome_db_control_init,
	(GtkArgSetFunc) NULL,
	(GtkArgSetFunc) NULL
      };
      db_control_type = gtk_type_unique(bonobo_object_get_type(), &db_control_info);
    }
  return db_control_type;
}

/**
 * gnome_db_control_new
 */
GtkObject *
gnome_db_control_new (const gchar *id, GnomeDbControlFunc func)
{
  GnomeDbControl* control;
  gchar*          factory_id;

  g_return_val_if_fail(id != NULL, NULL);
  g_return_val_if_fail(func != NULL, NULL);

  control = GNOME_DB_CONTROL(gtk_type_new(gnome_db_control_get_type()));
  control->create_widget_cb = func;

  /* create factory */
  factory_id = g_strdup_printf("control-factory:%s", id);
  control->factory = gnome_db_factory_new(factory_id, control_factory, (gpointer) control);
  g_free((gpointer) factory_id);
  if (!control->factory)
    {
      g_warning(_("Could not register control factory"));
      gnome_db_control_destroy(GNOME_DB_CONTROL(control));
      return NULL;
    }

  return GTK_OBJECT(control);
}

static void
gnome_db_control_destroy (GnomeDbControl *control)
{
  GList* node;

  g_return_if_fail(GNOME_DB_IS_CONTROL(control));

  /* close all open widgets */
  while ((node = g_list_first(control->widgets)))
    {
      GnomeDbControlInstance* control_instance = (GnomeDbControlInstance *) node->data;

      /* free memory */
      control->widgets = g_list_remove(control->widgets, (gpointer) control_instance);
      bonobo_object_destroy(BONOBO_OBJECT(control_instance->bonobo_control));
      gtk_object_destroy(GTK_OBJECT(control_instance->widget));
      g_free((gpointer) control_instance);
    }

  /* FIXME: should call object_class->destroy */
}

/**
 * gnome_db_control_get_instance_data
 * @control_instance: control instance
 *
 * Returns the user data associated with the given control instance
 */
gpointer
gnome_db_control_get_instance_data (GnomeDbControlInstance *control_instance)
{
  g_return_val_if_fail(control_instance != NULL, NULL);
  return control_instance->user_data;
}

/**
 * gnome_db_control_set_instance_data
 * @control_instance: control instance
 * @user_data: instance data
 */
void
gnome_db_control_set_instance_data (GnomeDbControlInstance *control_instance, gpointer user_data)
{
  g_return_if_fail(control_instance != NULL);
  control_instance->user_data = user_data;
}

/**
 * gnome_db_control_add_property
 * @control_instance: control instance
 * @name: property name
 * @desc: description
 * @type: property type
 * @value: default value
 */
void
gnome_db_control_add_property (GnomeDbControlInstance *control_instance,
			       const gchar *name,
			       const gchar *desc,
			       guint type,
			       gconstpointer value)
{
  g_return_if_fail(control_instance != NULL);
  g_return_if_fail(name != NULL);
  g_return_if_fail(desc != NULL);

  /* only add property if it does not exist */
  if (!property_find_by_name(control_instance, name))
    {
      BonoboPropertyBag*    pb;
      instance_prop_data_t* prop_data;

      prop_data = g_new0(instance_prop_data_t, 1);
      prop_data->id = control_instance->max_prop + 1;
      prop_data->name = g_strdup(name);
      prop_data->desc = g_strdup(desc);
      prop_data->type = type;
      control_instance->properties = g_list_append(control_instance->properties,
						   (gpointer) prop_data);
      control_instance->max_prop++;

      /* add the real Bonobo property */
      pb = bonobo_control_get_property_bag(control_instance->bonobo_control);
      if (BONOBO_IS_PROPERTY_BAG(pb))
	{
	  BonoboArgType bonobo_type;
	  switch (type)
	    {
	    case GNOME_DB_CONTROL_ARG_STRING :
	      bonobo_type = BONOBO_ARG_STRING;
	      break;
	    case GNOME_DB_CONTROL_ARG_BOOLEAN :
	      bonobo_type = BONOBO_ARG_BOOLEAN;
	      break;
	    case GNOME_DB_CONTROL_ARG_INT :
	      bonobo_type = BONOBO_ARG_INT;
	      break;
	    case GNOME_DB_CONTROL_ARG_LONG :
	      bonobo_type = BONOBO_ARG_LONG;
	      break;
	    case GNOME_DB_CONTROL_ARG_FLOAT :
	      bonobo_type = BONOBO_ARG_FLOAT;
	      break;
	    case GNOME_DB_CONTROL_ARG_DOUBLE :
	      bonobo_type = BONOBO_ARG_DOUBLE;
	      break;
	    default : bonobo_type = BONOBO_ARG_NULL;
	    }
	  bonobo_property_bag_add(pb, prop_data->name, prop_data->id, bonobo_type, NULL, desc, 0);
	}
      else g_warning(_("control has no property bag"));
    }
  if (value)
    gnome_db_control_set_property(control_instance, name, value);
}

/**
 * gnome_db_control_set_property
 */
void
gnome_db_control_set_property (GnomeDbControlInstance *control_instance,
			       const gchar *name,
			       gconstpointer value)
{
  instance_prop_data_t* prop_data;
  size_t                size;

  g_return_if_fail(control_instance != NULL);
  g_return_if_fail(name != NULL);
  g_return_if_fail(value != NULL);

  prop_data = property_find_by_name(control_instance, name);
  if (prop_data)
    {
      if (value)
	{
	  switch (prop_data->type)
	    {
	    case GNOME_DB_CONTROL_ARG_STRING :
	      if (prop_data->allocated) g_free((gpointer) prop_data->value);
	      prop_data->value = g_strdup((gchar *) value);
	      prop_data->allocated = TRUE;
	      break;
	    case GNOME_DB_CONTROL_ARG_BOOLEAN :
	      if (prop_data->allocated) g_free((gpointer) prop_data->value);
	      prop_data->value = g_malloc(sizeof(gboolean));
	      memcpy(prop_data->value, value, sizeof(gboolean)); 
	      prop_data->allocated = TRUE;
	      break;
	    case GNOME_DB_CONTROL_ARG_INT :
	      if (prop_data->allocated) g_free((gpointer) prop_data->value);
	      prop_data->value = g_malloc(sizeof(gint));
	      memcpy(prop_data->value, value, sizeof(gint));
	      prop_data->allocated = TRUE;
	      break;
	    case GNOME_DB_CONTROL_ARG_LONG :
	      if (prop_data->allocated) g_free((gpointer) prop_data->value);
	      prop_data->value = g_malloc(sizeof(glong));
	      memcpy(prop_data->value, value, sizeof(glong));
	      prop_data->allocated = TRUE;
	      break;
	    case GNOME_DB_CONTROL_ARG_FLOAT :
	      if (prop_data->allocated) g_free((gpointer) prop_data->value);
	      prop_data->value = g_malloc(sizeof(gfloat));
	      memcpy(prop_data->value, value, sizeof(gfloat));
	      prop_data->allocated = TRUE;
	      break;
	    case GNOME_DB_CONTROL_ARG_DOUBLE :
	      if (prop_data->allocated) g_free((gpointer) prop_data->value);
	      prop_data->value = g_malloc(sizeof(gdouble));
	      memcpy(prop_data->value, &value, sizeof(gdouble));
	      prop_data->allocated = TRUE;
	      break;
	    }
	}
    }
  else g_warning(_("property %s not found"), name);
}

/**
 * gnome_db_control_get_boolean
 * @control_instance: control instance
 * @name: property name
 */
gboolean
gnome_db_control_get_boolean (GnomeDbControlInstance *control_instance, const gchar *name)
{
  BonoboPropertyBag* pb;

  g_return_val_if_fail(control_instance != NULL, FALSE);
  g_return_val_if_fail(name != NULL, FALSE);

  pb = bonobo_control_get_property_bag(BONOBO_CONTROL(control_instance->bonobo_control));
  if (BONOBO_IS_PROPERTY_BAG(pb))
    {
      BonoboArg *arg = bonobo_property_bag_get_value(pb, name);
      if (arg) return BONOBO_ARG_GET_BOOLEAN(arg);
    }
  return FALSE;
}

/**
 * gnome_db_control_get_string
 * @control_instance: control instance
 * @name: property name
 */
gchar *
gnome_db_control_get_string (GnomeDbControlInstance *control_instance, const gchar *name)
{
  BonoboPropertyBag* pb;

  g_return_val_if_fail(control_instance != NULL, FALSE);
  g_return_val_if_fail(name != NULL, FALSE);

  pb = bonobo_control_get_property_bag(BONOBO_CONTROL(control_instance->bonobo_control));
  if (BONOBO_IS_PROPERTY_BAG(pb))
    {
      BonoboArg *arg = bonobo_property_bag_get_value(pb, name);
      if (arg) return BONOBO_ARG_GET_STRING(arg);
    }
  return FALSE;
}

/**
 * gnome_db_control_get_factory
 */
BonoboGenericFactory *
gnome_db_control_get_factory (GnomeDbControl *control)
{
  g_return_val_if_fail(GNOME_DB_IS_CONTROL(control), CORBA_OBJECT_NIL);
  return control->factory;
}





