/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

typedef struct
{
  FE_ConnectionInfo* cnc_info;
  GtkWidget*         from_xml_button;
  GtkWidget*         from_sql_button;
  GtkWidget*         filename_entry;
  GtkWidget*         file_text;
  GtkWidget*         transaction_mode;
} FE_ImportInfo;

static void begin_trans_cb      (Gda_Batch *job, gpointer data);
static void close_import_cb     (GtkWidget *w, gpointer data);
static void commit_trans_cb     (Gda_Batch *job, gpointer data);
static void exec_command_cb     (Gda_Batch *job, const gchar *cmd, gpointer data);
static void filename_changed_cb (GtkEditable *editable, FE_ImportInfo *imp_info);
static void open_cb             (GtkWidget *w, gpointer data);
static void rollback_trans_cb   (Gda_Batch *job, gpointer data);
static void save_cb             (GtkWidget *w, gpointer data);
static void start_import_cb     (GtkWidget *w, gpointer data);

static GnomeUIInfo importtoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Load file to import"),
    open_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save buffer"),
    save_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_SAVE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Start"), N_("Start importing data"),
    start_import_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear all entries"),
    0, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close this window"),
    close_import_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
import_from_sql (FE_ImportInfo *imp_info, gchar *buffer)
{
  Gda_Batch* job;
  gchar*     str;

  g_return_if_fail(imp_info != 0);
  g_return_if_fail(buffer != 0);

  /* create the batch job object */
  job = gda_batch_new();
  gda_batch_set_transaction_mode(job,
                                 gtk_toggle_button_get_active(
                                     GTK_TOGGLE_BUTTON(imp_info->transaction_mode)));
  gda_batch_set_connection(job, imp_info->cnc_info->cnc);

  gtk_signal_connect(GTK_OBJECT(job), "begin_transaction",
                     GTK_SIGNAL_FUNC(begin_trans_cb), 0);
  gtk_signal_connect(GTK_OBJECT(job), "commit_transaction",
                     GTK_SIGNAL_FUNC(commit_trans_cb), 0);
  gtk_signal_connect(GTK_OBJECT(job), "rollback_transaction",
                     GTK_SIGNAL_FUNC(rollback_trans_cb), 0);
  gtk_signal_connect(GTK_OBJECT(job), "execute_command",
                     GTK_SIGNAL_FUNC(exec_command_cb), 0);

  str = strtok(buffer, ";");
  while (str)
    {
      gda_batch_add_command(job, (const gchar *) str);
      str = strtok(NULL, ";");
    }

  /* run the 'transaction' */
  if (!gda_batch_start(job))
    {
      gda_ui_show_error(_("There were errors during the SQL execution"));
    }

  /* free memory */
  gda_batch_free(job);
}

static void
real_close (GtkWidget *w, gpointer data)
{
  FE_ImportInfo* imp_info;

  g_return_if_fail(GTK_IS_WIDGET(data));

  imp_info = (FE_ImportInfo *) fe_get_view_data(GTK_WIDGET(data), "FE_ImportInfo");
  if (imp_info)
    {
      g_free((gpointer) imp_info);
    }
}

/*
 * Callbacks
 */
static void
begin_trans_cb (Gda_Batch *job, gpointer data)
{
  g_return_if_fail(IS_GDA_BATCH(job));
  fe_status_bar_message(_("Transaction started"));
}

static void
close_connection_cb (Gda_Connection *cnc, GtkWidget *view)
{
  g_return_if_fail(IS_GDA_CONNECTION(cnc));
  g_return_if_fail(GTK_IS_WIDGET(view));

  if (view && fe_get_view_data(view, "FE_ImportInfo"))
    fe_destroy_view(view);
}

static void
commit_trans_cb (Gda_Batch *job, gpointer data)
{
  g_return_if_fail(IS_GDA_BATCH(job));
  fe_status_bar_message(_("Transaction finished successfully"));
}

static void
close_import_cb (GtkWidget *w, gpointer data)
{
  GtkWidget *view = fe_get_current_view();
  if (view && fe_get_view_data(view, "FE_ImportInfo"))
    {
      fe_destroy_view(view);
    }
}

static void
exec_command_cb (Gda_Batch *job, const gchar *cmd, gpointer data)
{
  g_return_if_fail(IS_GDA_BATCH(job));
  g_return_if_fail(cmd != 0);
}

static void
filename_changed_cb (GtkEditable *editable, FE_ImportInfo *imp_info)
{
  gchar* filename;

  g_return_if_fail(GTK_IS_ENTRY(editable));
  g_return_if_fail(imp_info != 0);

  filename = gtk_entry_get_text(GTK_ENTRY(editable));
  if (filename)
    {
      fe_load_file_in_text(GTK_TEXT(imp_info->file_text), filename);
    }
}

static void
open_cb (GtkWidget *w, gpointer data)
{
}

static void
rollback_trans_cb (Gda_Batch *job, gpointer data)
{
  g_return_if_fail(IS_GDA_BATCH(job));
  fe_status_bar_message(_("Transaction rolled back"));
}

static void
save_cb (GtkWidget *w, gpointer data)
{
  GtkWidget*     view;
  FE_ImportInfo* imp_info;

  view = fe_get_current_view();
  if (view && (imp_info = (FE_ImportInfo *) fe_get_view_data(view, "FE_ImportInfo")))
    {
      gchar* filename = fe_get_file_name(_("Select File"));
      if (filename)
        {
          fe_save_text_to_file(GTK_TEXT(imp_info->file_text), filename);
          g_free((gpointer) filename);
        }
    }
}

static void
start_import_cb (GtkWidget *w, gpointer data)
{
  FE_ImportInfo* imp_info;
  GtkWidget *view = fe_get_current_view();
  if (view && (imp_info = fe_get_view_data(view, "FE_ImportInfo")))
    {
      gchar* buffer;

      /* get text widget's contents */
      buffer = gtk_editable_get_chars(GTK_EDITABLE(imp_info->file_text), 0,
                                      gtk_text_get_length(imp_info->file_text));
      if (!buffer)
        {
          gda_ui_show_error(_("You must select a file to be imported"));
          return;
        }

      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(imp_info->from_xml_button)))
        {
        }
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(imp_info->from_sql_button)))
        {
          import_from_sql(imp_info, buffer);
        }
    }
}

/*
 * Public functions
 */
void
fe_open_import (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo* cnc_info = fe_get_current_connection();
  
  if (cnc_info != 0)
    {
      FE_ImportInfo* imp_info;
      gchar*         str;
      GtkWidget*     view;
      GtkWidget*     label;
      GtkWidget*     scroll;
      GtkWidget*     button;
      GtkWidget*     box;
      GtkWidget*     frame;

      /* create the internal strcture */
      imp_info = g_new0(FE_ImportInfo, 1);
      imp_info->cnc_info = cnc_info;

      /* create the view window */
      str = g_strdup_printf(_("Import: %s@%s"), gda_connection_get_user(cnc_info->cnc),
                            cnc_info->dsn);
      view = fe_new_view(str, 5, 10, importtoolbar);
      fe_set_view_data(view, "FE_ConnectionInfo", (gpointer) cnc_info);
      g_free((gpointer) str);

      /* create children */
      box = gtk_vbox_new(FALSE, 0);
      fe_add_widget_to_view(view, 0, 1, 1, 2, box);

      imp_info->from_xml_button = gda_ui_new_radio_button_widget(_("From XML"), 0);
      gtk_box_pack_start(GTK_BOX(box), imp_info->from_xml_button, 0, 0, 0);
      imp_info->from_sql_button = gda_ui_new_radio_button_widget(_("From SQL"),
                                         GTK_RADIO_BUTTON(imp_info->from_xml_button));
      gtk_box_pack_start(GTK_BOX(box), imp_info->from_sql_button, 0, 0, 0);

      frame = gda_ui_new_frame_widget(_("Options"));
      gtk_container_set_border_width(GTK_CONTAINER(frame), 3);
      fe_add_widget_to_view(view, 2, 1, 5, 3, frame);
      box = gtk_vbox_new(FALSE, 0);
      gtk_widget_show(box);
      gtk_container_add(GTK_CONTAINER(frame), box);
      imp_info->transaction_mode = gda_ui_new_check_button_widget(_("Transaction Mode"),
                                                                  TRUE);
      gtk_box_pack_start(GTK_BOX(box), imp_info->transaction_mode, 0, 0, 0);

      label = gda_ui_new_label_widget(_("File name"));
      fe_add_widget_to_view(view, 0, 2, 1, 3, label);
      imp_info->filename_entry = gda_ui_new_file_entry_widget("FE_Import_history");
      gtk_signal_connect(GTK_OBJECT(
                             gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(imp_info->filename_entry))),
                         "changed",
                         GTK_SIGNAL_FUNC(filename_changed_cb),
                         (gpointer) imp_info);
      fe_add_widget_to_view(view, 1, 2, 2, 3, imp_info->filename_entry);

      scroll = gda_ui_new_scrolled_window_widget();
      fe_add_widget_to_view(view, 0, 3, 5, 10, scroll);
      imp_info->file_text = fe_new_text_widget();
      gtk_container_add(GTK_CONTAINER(scroll), imp_info->file_text);

      /* connect to Gda_Connection's close signal */
      gtk_signal_connect(GTK_OBJECT(cnc_info->cnc), "close",
                         GTK_SIGNAL_FUNC(close_connection_cb), (gpointer) view);
      fe_set_view_destroy_func(view, (FE_DestroyViewFunc) real_close, (gpointer) view);
      fe_set_view_data(view, "FE_ImportInfo", (gpointer) imp_info);
      fe_display_view(view);
    }
}