/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

static GtkWidget* l_LoginDialog = NULL;

/*
 * Callbacks
 */
static void
error_cb (Gda_Connection* cnc, gpointer data)
{
  static GtkWidget *error_dialog = 0;

  if (!error_dialog)
    {
      error_dialog = gnome_db_errordlg_new(cnc, _("Error Viewer"));
      gnome_dialog_set_parent(GNOME_DIALOG(error_dialog), GTK_WIDGET(glb_App));
    }
  gnome_db_errordlg_show_errors(GNOME_DB_ERRORDLG(error_dialog)); 
}

static void
login_cb (Gda_Connection *cnc)
{
  g_return_if_fail(cnc != NULL);
  if (cnc->is_open)
    {
      gchar *dsn;
      GnomeDbLogin *login = GNOME_DB_LOGIN(GNOME_DB_LOGINDLG(l_LoginDialog)->login);
      if (login != NULL)
        dsn = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(login->gda_dbname)->entry));
      else dsn = NULL;
      fe_open_sql_window(cnc, dsn);
      fe_status_bar_message(_("Connected to %s"), dsn);

      gnome_dialog_close(GNOME_DIALOG(l_LoginDialog));
      l_LoginDialog = NULL;
    }
}

static void
cancel_cb (GnomeDbLogin* login, gpointer data)
{
  gnome_dialog_close(GNOME_DIALOG(l_LoginDialog));
  fe_status_bar_message("");
  l_LoginDialog = NULL;
}

/*
 * Private functions
 */
static void
fill_feature_list (Gda_Connection *cnc, GtkCList *clist)
{
  register gint     cnt;
  gchar*            row[2];
  const gchar*      empty_string = "";
  struct
  {
    gchar*                 label;
    GDA_Connection_Feature feature;
  } features[] =
  {
    { N_("Object Identifiers"), GDA_Connection_FEATURE_OBJECT_ID },
    { N_("Object Inheritance"), GDA_Connection_FEATURE_INHERITANCE },
    { N_("Sequences"), GDA_Connection_FEATURE_SEQUENCES },
    { N_("Stored Procedures"), GDA_Connection_FEATURE_PROCS },
    { N_("Transactions"), GDA_Connection_FEATURE_TRANSACTIONS },
    { N_("XML Queries"), GDA_Connection_FEATURE_XML_QUERIES }
  };
  static GtkWidget* on_pixmap = 0;
  static GtkWidget* off_pixmap = 0;

  g_return_if_fail(IS_GDA_CONNECTION(cnc));
  g_return_if_fail(GTK_IS_CLIST(clist));

  /* create pixmaps */
  if (!on_pixmap)
    {
      on_pixmap = gnome_stock_pixmap_widget(GTK_WIDGET(clist),
                                            GNOME_STOCK_MENU_BOOK_OPEN);
      if (!on_pixmap) return;
      gtk_widget_show(on_pixmap);
    }
  if (!off_pixmap)
    {
      off_pixmap = gnome_stock_pixmap_widget(GTK_WIDGET(clist),
                                             GNOME_STOCK_MENU_BOOK_GREEN);
      if (!off_pixmap) return;
      gtk_widget_show(off_pixmap);
    }

  row[0] = (gchar *) g_malloc0(32);
  row[1] = empty_string;

  gtk_clist_freeze(clist);
  for (cnt = 0; cnt < sizeof(features) / sizeof(features[0]); cnt ++)
    {
      strcpy(row[0], features[cnt].label);
      row[1] = empty_string;
      gtk_clist_append(clist, row);
      if (gda_connection_supports(cnc, features[cnt].feature))
        {
          gtk_clist_set_pixmap(clist, cnt, 1,
                               GNOME_PIXMAP(on_pixmap)->pixmap,
                               GNOME_PIXMAP(on_pixmap)->mask);
        }
      else
        {
          gtk_clist_set_pixmap(clist, cnt, 1,
                               GNOME_PIXMAP(off_pixmap)->pixmap,
                               GNOME_PIXMAP(off_pixmap)->mask);
        }
    }

  gtk_clist_thaw(clist);
  g_free((gpointer) row[0]);
}

/*
 * Public interface
 */
FE_ConnectionInfo *
fe_get_current_connection (void)
{
  GtkWidget *current_view;
  current_view = gtk_notebook_get_nth_page(GTK_NOTEBOOK(glb_WorkArea),
                           gtk_notebook_get_current_page(GTK_NOTEBOOK(glb_WorkArea)));
  if (current_view != NULL)
    {
      gpointer info;
      if ((info = fe_get_view_data(current_view, "FE_ConnectionInfo")))
        return ((FE_ConnectionInfo *) info);
    }
  return (NULL);
}

void
fe_login (GtkWidget *w, gpointer data)
{
  Gda_Connection *cnc = gda_connection_new(glb_Orb);
  /* check if dialog box is already open */
  if (l_LoginDialog == NULL)
    {
      l_LoginDialog = gnome_db_logindlg_new(GNOME_DB_LOGIN(gnome_db_login_new(cnc,
                                                                              NULL,
                                                                              NULL)),
                                            _("Open Connection"));
      gnome_dialog_set_parent(GNOME_DIALOG(l_LoginDialog), GTK_WIDGET(glb_App));
      gtk_signal_connect(GTK_OBJECT(cnc), "error", GTK_SIGNAL_FUNC(error_cb), NULL);
      gtk_signal_connect_after(GTK_OBJECT(cnc),
                               "open",
                               GTK_SIGNAL_FUNC(login_cb),
                               NULL);
      gtk_signal_connect(GTK_OBJECT(l_LoginDialog),
                         "cancel",
                         GTK_SIGNAL_FUNC(cancel_cb),
                         NULL);
      gnome_dialog_close_hides(GNOME_DIALOG(l_LoginDialog), TRUE);
    }
  fe_status_bar_message(_("Connecting..."));
  while (gtk_events_pending())
    gtk_main_iteration();
  gnome_db_logindlg_popup(GNOME_DB_LOGINDLG(l_LoginDialog));
}

void
fe_begin_transaction (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      if (gda_connection_supports(cnc_info->cnc, GDA_Connection_FEATURE_TRANSACTIONS))
        {
          if (gda_connection_begin_transaction(cnc_info->cnc) == 0)
            fe_status_bar_message(_("Transaction started"));
        }
      else fe_status_bar_message(_("Transactions not supported"));
    }
}

void
fe_commit_transaction (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      if (gda_connection_supports(cnc_info->cnc, GDA_Connection_FEATURE_TRANSACTIONS))
        {
          if (gda_connection_commit_transaction(cnc_info->cnc) == 0)
            fe_status_bar_message(_("Commit executed"));
        }
      else fe_status_bar_message(_("Transactions not supported"));
    }
}

void
fe_rollback_transaction (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      if (gda_connection_supports(cnc_info->cnc, GDA_Connection_FEATURE_TRANSACTIONS))
        {
          if (gda_connection_rollback_transaction(cnc_info->cnc) == 0)
            fe_status_bar_message(_("Rollback executed"));
        }
      else fe_status_bar_message(_("Transactions not supported"));
    }
}

void
fe_show_connection_info (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info)
    {
      GtkWidget *dialog, *frame, *table, *label, *entry, *button;
      GtkWidget *clist, *scroll, *frame2;
      gchar      txt[256];
      
      /* create dialog */
      dialog = gnome_dialog_new(_("Connection Properties"),
                                GNOME_STOCK_BUTTON_CLOSE, 0);
      frame = gda_ui_new_frame_widget(_("Properties"));
      gtk_container_set_border_width(GTK_CONTAINER(frame), 3);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 0, 0, 0);
      table = gtk_table_new(6, 3, 0);
      gtk_widget_show(table);
      gtk_container_add(GTK_CONTAINER(frame), table);
      
      label = gda_ui_new_label_widget(_("GDA Name"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), cnc_info->dsn);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       GTK_FILL, 3, 3);
      
      label = gda_ui_new_label_widget(_("Provider"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_provider(cnc_info->cnc));
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       GTK_FILL, 3, 3);
                       
      label = gda_ui_new_label_widget(_("DSN"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_dsn(cnc_info->cnc));
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 2, 3,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Command Timeout"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      //sprintf(txt, "%ld", gda_connection_get_cmd_timeout(cnc_info->cnc));
      //gtk_entry_set_text(GTK_ENTRY(entry), txt);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 3, 4,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       GTK_FILL, 3, 3);
                       
      label = gda_ui_new_label_widget(_("Connect Timeout"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      //sprintf(txt, "%ld", gda_connection_get_connect_timeout(cnc_info->cnc));
      //gtk_entry_set_text(GTK_ENTRY(entry), txt);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 4, 5,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Trace File"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      if (cnc_info->trace_file)
        gtk_entry_set_text(GTK_ENTRY(entry), cnc_info->trace_file);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 5, 6,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       GTK_FILL, 3, 3);
                       
      button = gda_ui_new_button_widget_with_pixmap(_("View"), GNOME_STOCK_MENU_PROP);
      gtk_signal_connect(GTK_OBJECT(button),
                         "clicked",
                         GTK_SIGNAL_FUNC(fe_view_server_trace), NULL);
      gtk_table_attach(GTK_TABLE(table), button, 2, 3, 5, 6, GTK_SHRINK, GTK_SHRINK, 3, 3);

      frame2 = gda_ui_new_frame_widget(_("Provider Features"));
      gtk_container_set_border_width(GTK_CONTAINER(frame2), 3);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame2, 1, 1, 0);
      scroll = gda_ui_new_scrolled_window_widget();
      gtk_container_add(GTK_CONTAINER(frame2), scroll);
      clist = gda_ui_new_clist_widget(0, 2);
      gtk_container_add(GTK_CONTAINER(scroll), clist);
      fill_feature_list(cnc_info->cnc, GTK_CLIST(clist));

      /* run the dialog */
      gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
    }
}