/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <memory.h>
#include <glib.h>

#include "odbc.h"
#include "sqldriver.h"

#include "gda-types.h"
#include "gda-recset.h"
#include "gda-error.h"
#include "gda-command.h"
#include "gda-field.h"
#include "gda-odbc-misc.h"


Gda_ODBC_Recordset*
gda_odbc_recset_new()
{
  Gda_ODBC_Recordset* rc = g_new0(Gda_ODBC_Recordset, 1);
  
  return rc;
}

void
gda_odbc_recset_free(Gda_ODBC_Recordset* recset)
{

  gda_odbc_recset_close(recset);
  
  g_free(recset);
}




gint
gda_odbc_recset_allocmem(Gda_ODBC_Recordset* rec)
{
  Gda_ODBC_Field* server_field;
  GList*          ptr = rec->fields;
  gint            rc;
  gint            i;

  i = 0;
  while (ptr)
    {
      server_field = ptr->data;
      if (!server_field || !server_field->value)
	{
	  g_warning("allocate_recordset_memory: server_field = %p, it's value = %p\n",
		    server_field, server_field ? server_field->value : 
		    0);
	  return -1;
	}
      server_field->value->_d = gda_odbc_sql2gdatype(server_field->sql_type);      
      server_field->c_type = gda_odbc_sql2ctype(server_field->sql_type);
      switch (server_field->c_type)
	{
	  /* GDA_TypeBigint, GDA_TypeChar, GDA_TypeCurrency,
	   * GDA_TypeDecimal, GDA_TypeError (not used here),
	   * GDA_TypeLongvarchar, GDA_TypeLongvarwchar,
	   * GDA_TypeNumeric, GDA_TypeUBigInt, GDA_TypeVarchar,
	   * GDA_TypeVarwchar, GDA_TypeFixchar, GDA_TypeFixbin,
	   * GDA_TypeFixwchar, GDA_TypeChar
	   */
	case SQL_C_CHAR:	/* unsigned gchar* (zero terminated)*/
	  {
	    gchar* ptr;
	    gint   alloced_length;
	    
	    /* Various SQL (GDA) types are mapped to SQL_C_CHAR for retrieval.
	       Be smart about allocating storage.
	       SQL_LONGVARCHAR and SQL_LONGVARBINARY get allocated 255 bytes of data.
	       If that's enough, fine. Otherwise the client must use the get_chunk
	       method to retrieve the data used for (BLOBS).
	    */
	    switch (server_field->sql_type)
	      {
	      case SQL_LONGVARBINARY:
	      case SQL_LONGVARCHAR:
		server_field->value->_u.lvc = CORBA_string_alloc(256);
		ptr = server_field->value->_u.lvc;
		alloced_length = 255;
		break;
	      default:
		server_field->value->_u.lvc = CORBA_string_alloc(server_field->defined_length + 1);
		ptr = server_field->value->_u.lvc;
		alloced_length = server_field->defined_length + 1;
	      }
	    server_field->malloced      = 1;
	    rc = SQLBindCol(rec->hstmt,
			    i + 1,
			    server_field->c_type,
			    ptr,
			    alloced_length,
			    &server_field->actual_length);
	    if (rc != 0)
	      goto bad_binding;
	  }
	  break;
	  /*
	   * GDA_TypeSmallint
	   */
	case SQL_C_SSHORT:	/* gshort */
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.si,
			  sizeof(server_field->value->_u.si),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	  /*
	   * GDA_TypeUSmallint
	   */
	case SQL_C_USHORT:	/* gushort */
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.us,
			  sizeof(server_field->value->_u.us),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	case SQL_C_SLONG:	/* glong (the same as intger, CORBA
				 * doesn't distinguish between them*/
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.i,
			  sizeof(server_field->value->_u.i),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	case SQL_C_ULONG:	/* gulong */
	  g_warning("allocate_recordset_memory: SQL_C_ULONG: NYI\n");
	  goto bad_binding;
	  /*
	   * GDA_TypeSingle
	   */
	case SQL_C_FLOAT:	/* gfloat */
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.f,
			  sizeof(server_field->value->_u.f),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	  /*
	   * GDA_TypeDouble
	   */
	case SQL_C_DOUBLE:	/* gdouble */
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.dp,
			  sizeof(server_field->value->_u.dp),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	  /*
	   * @mla@ missing
	   */
	case SQL_C_STINYINT:	/* gchar */
	  g_warning("allocate_recordset_memory: SQL_C_TINYINT: missing");
	  goto bad_binding;
	  /*
	   * @mla@ missing
	   */
	case SQL_C_UTINYINT:	/* unsigned gchar */
	  g_warning("allocate_recordset_memory: SQL_C_UTINYINT: missing");
	  goto bad_binding;
	  /*
	   * GDA_TypeVarbin, GDA_TypeBinary, GDA_TypeBstr,
	   * GDA_TypeLongvarbin
	   */
	case SQL_C_BINARY:	/* unsigned gchar* (with length) */
	  server_field->value->_u.lvb._buffer = CORBA_string_alloc(server_field->defined_length);
	  server_field->value->_u.lvb._maximum = server_field->defined_length;
	  server_field->malloced      = 1;
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  server_field->value->_u.lvb._buffer,
			  server_field->value->_u.lvb._maximum,
			  (glong*)&server_field->actual_length);
	  if (rc != 0)
	      goto bad_binding;
	  break;
	  
	case SQL_C_DATE:	/* GDA_DbDate */
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.dbd,
			  sizeof(server_field->value->_u.dbd),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	case SQL_C_TIME:	/* GDA_DbTime */
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.dbt,
			  sizeof(server_field->value->_u.dbt),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	case SQL_C_TIMESTAMP:	/* GDA_DbTimestamp */
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.dbtstamp,
			  sizeof(server_field->value->_u.dbtstamp),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	  /*
	   * @mla@ missing
	   */
	case SQL_C_TINYINT:	/* gchar */
	  g_warning("allocate_recordset_memory: SQL_C_TINYINT: NYI");
	  goto bad_binding;
	  /*
	   * @mla@ missing
	   */
	case SQL_C_SHORT:	/* gshort */
	  g_warning("allocate_recordset_memory: SQL_C_SHORT: NYI");
	  goto bad_binding;
	case SQL_C_LONG:	/* glong */
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.d,
			  sizeof(server_field->value->_u.d),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	  
	case SQL_C_BIT:
	  rc = SQLBindCol(rec->hstmt,
			  i + 1,
			  server_field->c_type,
			  &server_field->value->_u.c,
			  sizeof(server_field->value->_u.c),
			  &server_field->actual_length);
	  if (rc != 0)
	    goto bad_binding;
	  break;
	default:
	  g_print("Unknown SQL C Type in SqlBindCol received\n"); 
	  break;
	}
      ptr = g_list_next(ptr);
      i++;
    }
  return 0;
 bad_binding:
  {
    gint idx;
    ptr = rec->fields;
    for (idx = 0; idx < i; idx++)
      {
	server_field = ptr->data;
	if (server_field->c_type == SQL_C_CHAR)
	  CORBA_free(server_field->value->_u.lvc);
	if (server_field->c_type == SQL_C_BINARY)
	  CORBA_free(server_field->value->_u.lvb._buffer);
	server_field->value->_d = 1;
	ptr = g_list_next(ptr);
      }
  }
  gda_odbc_error_make( rec, 0, "SQLBindCol");
  return -1;
}



gint
gda_odbc_recset_move_next(Gda_ODBC_Recordset* recset)
{
  gint rc;

  if (recset->at_end)
    {
      fprintf(stderr,"EOF already reached\n");
      return 1;
    }
  if (gda_odbc_recset_allocmem(recset) < 0)
    {
      fprintf(stderr,"gda_odbc_recset_move_next: gda_odbc_recset_allocmem returns -1\n");
      return -1;
    }
  do {
    rc = SQLFetch(recset->hstmt);
    if (rc == SQL_NO_DATA_FOUND)
      {
	recset->at_end = 1;
	return 1;
      }
    if (rc == SQL_ERROR)
      {
	gda_odbc_error_make(recset, 0, "SQLFetch");
	return -1;
      }
    if (rc == SQL_INVALID_HANDLE)
      {
	gda_odbc_error_make(recset, 0, "SQLFetch");
	return -1;
      }
  }while (recset->internal_filter_function && !recset->internal_filter_function(recset, 0));
  recset->pos++;
  return 0;
}

gint
gda_odbc_recset_move_prev(Gda_ODBC_Recordset* recset)
{
  gint rc;

  GList* l = recset->fields;
  while (l)
    {
      Gda_ODBC_Field* f = l->data;
      
      if (f->sql_type == SQL_VARCHAR && f->value->_u.lvc)
	{
	  memset(f->value->_u.lvc, 0, f->defined_length);
	}
      else if (f->sql_type == SQL_CHAR && f->defined_length > 1 && f->value->_u.lvc)
	{
	  memset(f->value->_u.lvc,' ', f->defined_length);
	}
      l = g_list_next(l);
    }
  rc = SQLFetchPrev(recset->hstmt);
  if (rc == SQL_NO_DATA_FOUND)
    {
      recset->at_end = 1;
      return 1;
    }
  if (rc == SQL_ERROR)
    {
      return -1;
    }
  if (rc == SQL_INVALID_HANDLE)
    {
      return -1;
    }
  recset->pos--;
  return 0;
}

gint
gda_odbc_recset_close(Gda_ODBC_Recordset* rec)
{
  GList* ptr;

  fprintf(stderr,"gda_odbc_recset_close: enter\n");
  
  if (rec->hstmt) {
    SQLFreeStmt(rec->hstmt, SQL_DROP);
  }
  rec->hstmt = 0;
  
  ptr = rec->fields;
  while(ptr)
    {
      gda_odbc_field_free(ptr->data);
      ptr = g_list_next(ptr);
    }
  g_list_free(rec->fields);

  rec->fields = 0;
  fprintf(stderr,"gda_odbc_recset_close: left\n");
  return 0;
}


