/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <memory.h>

#include "gda-mysql-field.h"
#include "gda-mysql-command.h"
#include "gda-mysql-recset.h"
#include "gda-mysql-error.h"
#include "gda-mysql-connection.h"
#include "gda-mysql-misc.h"

static guint
string_hash (gconstpointer v)
{
  register gchar *c;
  register guint h;
  
  h = 0;
  for (c = (gchar*)v; *c; c++)
    {
      register guint g;
      
      h = (h << 4) + *c;
      g = h & 0xf0000000;
      if (g)
	{
	  h = h ^ (g >> 24);
	  h = h ^ g;
	}
    }
  
  return h;
}

static gint
string_compare(gconstpointer s1, gconstpointer s2)
{
  return !strcmp(s1, s2);
}

Gda_MYSQL_Command*
gda_mysql_cmd_new()
{
  Gda_MYSQL_Command* rc = g_new0(Gda_MYSQL_Command, 1);
  return rc;
}


void
gda_mysql_cmd_free(Gda_MYSQL_Command* cmd)
{
  g_free(cmd);
	
}

Gda_MYSQL_Connection*
gda_mysql_cmd_set_connection(Gda_MYSQL_Command* cmd,
			     Gda_MYSQL_Connection* cnc)
{
  Gda_MYSQL_Connection* old_cnc;

  old_cnc = cmd->cnc;
  cmd->cnc = cnc;
  return old_cnc;
  
}

static GString*
splice_parameters(gchar* cmd, const GDA_CmdParameterSeq* params, Gda_MYSQL_Connection* cnc)
{
  GString* retval;
  gint     in_string = 0;
  gint     escape_seen = 0;
  gint     current_param_idx = 0;
  
  retval = g_string_new("");
  while (*cmd)
    {
      if (*cmd == '\\')
	escape_seen = 1;
      if (*cmd == '\'')
	{
	  if (in_string)
	    {
	      if (!escape_seen)
		in_string = 0;
	    }
	}
      if (*cmd == '?' && !in_string)
	{
	  if (params->_length <= current_param_idx)
	    {
	      g_warning("Number of parameters less then '?' in statement");
	      g_string_free(retval, 1);
	      return 0;
	    }
	  gda_mysql_gdavalue2mysql(retval, &params->_buffer[current_param_idx].value);
	  current_param_idx++;
	  cmd++;
	  continue;
	}
      g_string_append_c(retval, *cmd);
      cmd++;
    }
  return retval;
}

Gda_MYSQL_Recordset*
gda_mysql_cmd_execute        (Gda_MYSQL_Command* cmd, const GDA_CmdParameterSeq* params, gulong options)
{
  gint                 rc;
  Gda_MYSQL_Recordset* recset;
  GString*             cmd_string;

  cmd_string = splice_parameters(cmd->cmd, params, cmd->cnc);
  if (!cmd_string)
    return 0;
  fprintf(stderr,"*** Calling mysql_real_query\n");
  rc = mysql_real_query(cmd->cnc->mysql, cmd->cmd, strlen(cmd->cmd));
  if (rc != 0)
    {
      gda_mysql_error_make(cmd->cnc, "mysql_real_query");
      return 0;
    }
  recset = gda_mysql_recset_new();
  recset->cnc = cmd->cnc;
  recset->mysql_res = mysql_store_result(cmd->cnc->mysql);
  return recset;
}

