/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-ui.h"
#include "gda-error.h"
#include "gda-server.h"

enum
{
  GNOME_DB_LOGIN,
  LAST_SIGNAL
};

static gint   gnome_db_login_signals[LAST_SIGNAL] = {0,};
static GList* last_connections = 0;

static void gnome_db_login_class_init      (GnomeDbLoginClass* klass);
static void gnome_db_login_init            (GnomeDbLogin* login);

static void
remove_duplicates (GList *node, gchar *str)
{
  g_return_if_fail(str != 0);
  
  while (node != 0)
    {
      gchar *tmp = (gchar *) node->data;
      if (tmp != 0)
        {
          if (!strcmp(tmp, str))
            {
              last_connections = g_list_remove(last_connections, node->data);
              return;
            }
        }
      node = g_list_next(node);
    }
}

static void
select_last_connection_cb (GtkCList *clist,
                           gint row,
                           gint column,
                           GdkEvent *event,
                           gpointer data)
{
  GList*        sel;
  GnomeDbLogin* login = GNOME_DB_LOGIN(data);
  
  g_return_if_fail(GTK_IS_CLIST(clist));
  g_return_if_fail(GNOME_DB_IS_LOGIN(login));
  
  sel = GTK_CLIST(clist)->selection;
  if (sel != 0)
    {
      gchar* str = 0;
      gchar* empty_string = "";
      
      gtk_clist_get_text(clist, GPOINTER_TO_UINT(sel->data), 0, &str);
      if (str)
        {
          gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(login->gda_dbname)->entry), str);
          str = 0;
          gtk_clist_get_text(clist, GPOINTER_TO_UINT(sel->data), 1, &str);
          if (str)
            {
              gtk_entry_set_text(GTK_ENTRY(login->username_entry), str);
            }
          else gtk_entry_set_text(GTK_ENTRY(login->username_entry), empty_string);
        }
    }
}

static void
gnome_db_login_real_login (GnomeDbLogin* login)
{
  gchar* db_name;
  gchar* dsn;
  gchar* user_name;
  gchar* password;
  gchar* provider;
  gchar  resource[128];

  db_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(login->gda_dbname)->entry));
  g_return_if_fail(db_name != 0);
  
  user_name = gtk_entry_get_text(GTK_ENTRY(login->username_entry));
  password = gtk_entry_get_text(GTK_ENTRY(login->password_entry));
  g_snprintf(resource, sizeof(resource), "/gdalib/%s/DSN", db_name);
  dsn = gnome_config_get_string(resource);

  if (!dsn)
    {
      Gda_Error* error;
      error = gda_error_new();
      error->description = g_strdup_printf(_("No datasource '%s' defined in your GDA configuration file"), db_name);
      error->source = g_strdup("[GNOME DB Client Widgets]");
      gtk_signal_emit_stop_by_name(GTK_OBJECT(login), "login");
      gda_connection_add_single_error(login->cnc,  error);
    }
  
  g_snprintf(resource, sizeof(resource), "/gdalib/%s/Provider", db_name);
  provider = gnome_config_get_string(resource);
  gda_connection_set_provider(login->cnc, provider);
  if (gda_connection_open(login->cnc, dsn, user_name, password) < 0)
    {
      gtk_signal_emit_stop_by_name(GTK_OBJECT(login), "login");
    }
  else
    {
      GList *node;
      gchar *str;
      gint   cnt;
      
      /* add connection info to gdalib file */
      last_connections = g_list_prepend(last_connections, (gpointer) db_name);
      if (g_list_length(last_connections) > 10)
        {
          GList *node = g_list_last(last_connections);
          if (node != 0)
            {
              str = (gchar *) node->data;
              last_connections = g_list_remove(last_connections, str);
              g_free((gpointer) str);
            }
        }
      remove_duplicates(g_list_next(g_list_first(last_connections)), db_name);
      
      /* save Last Connections */
      node = g_list_first(last_connections);
      cnt = 1;
      while (node != 0)
        {
          str = g_strdup_printf("/gdalib/Last Connections/Connection%d", cnt);
          gnome_config_set_string(str, (gchar *) node->data);
          g_free((gpointer) str);
          cnt++;
          node = g_list_next(node);
        }

      /* save username in GDA configuration */
      g_snprintf(resource, sizeof(resource), "/gdalib/%s/Username", db_name);
      gnome_config_set_string(resource, user_name);

      gnome_config_sync();
    }
}

guint
gnome_db_login_get_type (void)
{
  static guint db_login_type = 0;

  if (!db_login_type)
    {
      GtkTypeInfo db_login_info =
      {
	"GnomeDbLogin",
	sizeof (GnomeDbLogin),
	sizeof(GnomeDbLoginClass),
	(GtkClassInitFunc)gnome_db_login_class_init,
	(GtkObjectInitFunc)gnome_db_login_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      db_login_type = gtk_type_unique(gtk_vbox_get_type(), &db_login_info);
    }
  return db_login_type;
}

static void
gnome_db_login_class_init (GnomeDbLoginClass* klass)
{
  GtkObjectClass* object_class;
  register gint   cnt;

  object_class = (GtkObjectClass*) klass;

  gnome_db_login_signals[GNOME_DB_LOGIN] = gtk_signal_new("login",
                                                          GTK_RUN_LAST,
                                                          object_class->type,
                                                          GTK_SIGNAL_OFFSET(GnomeDbLoginClass, login),
                                                          gtk_signal_default_marshaller,
                                                          GTK_TYPE_INT, 0);
  gtk_object_class_add_signals(object_class, gnome_db_login_signals, LAST_SIGNAL);
  
  klass->login = gnome_db_login_real_login;
  
  /* read last successful connections */
  for (cnt = 1; cnt < 11; cnt++)
    {
      gchar *str, *data;
      
      str = g_strdup_printf("/gdalib/Last Connections/Connection%d", cnt);
      data = gnome_config_get_string(str);
      g_free((gpointer) str);
      if (data != 0)
        {
          str = g_strdup(data);
          last_connections = g_list_append(last_connections, (gpointer) str);
        }
    }
}

static void
gnome_db_login_init (GnomeDbLogin* login)
{
  GtkWidget* label;
  GtkWidget* table;
  GtkWidget* frame;
  GtkWidget* scroll;
  GList*     node;
  gchar*     titles[] = { N_("GDA Name"), N_("User Name") };

  table = gtk_table_new(4, 3, 0);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);

  label = gda_ui_new_label_widget(_("GDA Datasource"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.1, 1.0);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
		   GTK_FILL, 0, 0, 0);

  login->gda_dbname = gtk_combo_new();
  gtk_combo_set_value_in_list(GTK_COMBO(login->gda_dbname), TRUE, FALSE);
  gtk_table_attach(GTK_TABLE(table), login->gda_dbname, 1, 2, 0, 1,
		    GTK_FILL, 0, 0, 0);
  gtk_widget_show(login->gda_dbname);
  
  label = gda_ui_new_label_widget(_("Username"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.1, 1.0);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
		   GTK_FILL, 0, 0, 0);
  
  login->username_entry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(table), login->username_entry, 1, 2, 1, 2,
		   GTK_FILL, 0, 0, 0);
  gtk_widget_show(login->username_entry);
  
  label = gda_ui_new_label_widget(_("Password"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.1, 1.0);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
		   GTK_FILL, 0, 0, 0);
  
  login->password_entry = gtk_entry_new();
  gtk_widget_show(login->password_entry);
  gtk_table_attach(GTK_TABLE(table), login->password_entry, 1, 2, 2, 3,
		   GTK_FILL, 0, 0, 0);
  gtk_entry_set_visibility(GTK_ENTRY(login->password_entry), FALSE);

  frame = gda_ui_new_frame_widget(_("Last Connections"));
  gtk_table_attach(GTK_TABLE(table), frame, 0, 3, 3, 4,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   0, 0);
                   
  scroll = gda_ui_new_scrolled_window_widget();
  gtk_container_add(GTK_CONTAINER(frame), scroll);
  login->connection_list = gda_ui_new_clist_widget(titles, sizeof(titles) / sizeof(titles[0]));
  gtk_signal_connect(GTK_OBJECT(login->connection_list), "select_row",
                     GTK_SIGNAL_FUNC(select_last_connection_cb), (gpointer) login);
  gtk_container_add(GTK_CONTAINER(scroll), login->connection_list);
  node = g_list_first(last_connections);
  while (node != 0)
    {
      gchar* row[2];
      gchar  resource[128];
      gchar* user_name;
      gchar* str = (gchar *) node->data;

      if (str != 0)
        {
          g_snprintf(resource, sizeof(resource), "/gdalib/%s/Username", str);
          user_name = gnome_config_get_string(resource);
          row[0] = str;
          row[1] = user_name;
          gtk_clist_append(GTK_CLIST(login->connection_list), (gchar **) row);
        }
      node = g_list_next(node);
    }
  
  gtk_box_pack_start(GTK_BOX(login), table, TRUE, TRUE, 0);
  gtk_widget_show(table);
}

/**
 * gnome_db_login_new
 * @cnc: the Gda_Connection object to be used to connect
 * @gda_provider: GDA provider
 * @gda_dbname: GDA data source name
 *
 * Create a new GnomeDbLogin widget. This widget contains all the entry fields
 * required for connecting to a GDA data source, as well as a history list,
 * which allows the user to select from previously successful connections
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget*
gnome_db_login_new(Gda_Connection* cnc, gchar* gda_provider, gchar* gda_dbname)
{
  
  GtkWidget* widget;
  GnomeDbLogin* login;
  GList*        db_names;
  
  if (gda_provider)
    {
      db_names = gda_list_datasources_for_provider(gda_provider);
    }
  else
    {
      db_names = gda_list_datasources();
    }
  if (gda_dbname && db_names)
    {
      if (!g_list_find_custom(db_names, gda_dbname, (GCompareFunc)strcmp))
	{
	  Gda_Error* error;

	  error = gda_error_new();
	  error->source = g_strdup("[GNOME DB Client Widgets]");
	  error->description = g_strdup_printf(_("No datasource '%s' defined in your GDA configuration file"),
					       gda_dbname);
	  g_list_foreach(db_names, (GFunc)g_free, 0);
	  g_list_free(db_names);
	  return 0;
	}
    }
  widget = gtk_type_new(gnome_db_login_get_type());
  login = GNOME_DB_LOGIN(widget);
  login->cnc = cnc;
  if (db_names)
    {
      gtk_combo_set_popdown_strings(GTK_COMBO(login->gda_dbname), db_names);
    }
  
  if (gda_dbname)
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(login->gda_dbname)->entry), gda_dbname);
  return widget;
}

/**
 * gnome_db_login_login
 * @login: the GnomeDbLogin widget
 *
 * Attempts to establish a connection to a GDA provider by using the
 * current information entered by the user in the GnomeDbLogin child
 * widgets
 */
void
gnome_db_login_login (GnomeDbLogin* login)
{
  gtk_signal_emit(GTK_OBJECT(login), gnome_db_login_signals[GNOME_DB_LOGIN]);
}
