/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

/*
 * File selection dialog
 */
static void
filesel_button_clicked (GtkWidget *w, glong *selected)
{
  *selected = (glong) gtk_object_get_data(GTK_OBJECT(w), "button");
}

gchar *
fe_get_file_name (gchar *title)
{
  GtkWidget *filesel;
  glong *selected;
  gchar *res = NULL;
  
  /* initialize variables */
  selected = (glong *) g_malloc(sizeof(glong));
  *selected = 0;
  
  /* create dialog */
  filesel = gtk_file_selection_new(title);
  gtk_widget_set_parent_window(filesel, GTK_WIDGET(glb_App)->window);
  gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
		      "button", (gpointer) 1);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(filesel_button_clicked),
		     (gpointer) selected);
  gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
		      "button", (gpointer) 2);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
		     "clicked", GTK_SIGNAL_FUNC(filesel_button_clicked),
		     (gpointer) selected);
  gtk_widget_show(filesel);
  gtk_file_selection_show_fileop_buttons(GTK_FILE_SELECTION(filesel));
  
  /* wait for selection from user */
  while (!*selected)
    gtk_main_iteration_do(TRUE);
  if (*selected == 1) /* Ok-button */
    res = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel)));
  else res = NULL;
  
  /* free memory before exiting */
  gtk_widget_destroy(filesel);
  g_free((gpointer) selected);
  return (res);
}

/*
 * Utility functions
 */
GtkWidget *
fe_add_list_item (GtkList *list, const gchar *text)
{
  GtkWidget *label, *list_item;
  gchar *string;
  
  g_return_val_if_fail(list != NULL, NULL);

  label = gtk_label_new((gchar *) text);
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
  list_item = gtk_list_item_new();
  gtk_container_add(GTK_CONTAINER(list_item), label);
  gtk_widget_show(label);
  gtk_container_add(GTK_CONTAINER(list), list_item);
  gtk_widget_show(list_item);
  gtk_label_get(GTK_LABEL(label), &string);
  gtk_object_set_data(GTK_OBJECT(list_item), FE_LIST_ITEM_DATA_KEY, string);
  return (list_item);
}

void
fe_fill_gtk_list_with_glist (GtkList *gtklist, GList *glist)
{
  GList *node;

  g_return_if_fail(gtklist != NULL);
  g_return_if_fail(glist != NULL);

  gtk_list_clear_items(gtklist, 0, -1);
  for (node = g_list_first(glist); node != NULL; node = g_list_next(node))
    {
      fe_add_list_item(gtklist, (const gchar *) node->data);
    } 
}

gchar *
fe_get_combo_current_string (GtkCombo *combo)
{
  g_return_val_if_fail(combo != NULL, NULL);
  g_return_val_if_fail(GTK_IS_COMBO(combo), NULL);
  return (gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry)));
}

gchar *
fe_get_list_current_string (GtkList *list)
{
  GList *node;
  g_return_val_if_fail(list != NULL, NULL);
  node = g_list_first(list->selection);
  if (node != NULL)
    {
      return ((gchar *) gtk_object_get_data(GTK_OBJECT(node->data), FE_LIST_ITEM_DATA_KEY));
    }
  return (NULL);
}

void
fe_insert_file_into_text (GtkText *text, gchar *filename)
{
  FILE *file;
  
  /* check parameters */
  g_return_if_fail(text != NULL);
  g_return_if_fail(filename != NULL);
  
  /* open file */
  if ((file = fopen(filename, "r")))
    {
      gtk_text_set_point(text, gtk_text_get_point(text));
      gtk_text_freeze(text);
      while (!feof(file))
	{
	  gchar buffer[1025];
	  memset(buffer, 0, sizeof(buffer));
	  fread(buffer, sizeof(buffer) - 1, 1, file);
	  if (ferror(file))
	    {
	      gda_ui_show_error(_("Error reading file\n %s"), filename);
	      break;
	    }
	  gtk_text_insert(text, NULL, NULL, NULL, buffer, strlen(buffer));
	}
      gtk_text_thaw(text);
      fclose(file);
      fe_status_bar_message(_("File %s inserted"), filename);
    }
  else gda_ui_show_error(_("Error opening file\n %s"), filename);
}

gboolean
fe_is_radio_button_active (GtkRadioButton *radio_button)
{
  g_return_val_if_fail(radio_button != NULL, FALSE);
  g_return_val_if_fail(GTK_IS_RADIO_BUTTON(radio_button), FALSE);
  return (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radio_button)));
}

void
fe_load_file_in_text (GtkText *text, gchar *filename)
{
  FILE *file;
  
  /* check parameters */
  g_return_if_fail(text != NULL);
  g_return_if_fail(filename != NULL);
  
  /* open file */
  if ((file = fopen(filename, "r")))
    {
      gtk_editable_delete_text(GTK_EDITABLE(text), 0,
                               gtk_text_get_length(text));
      gtk_text_freeze(text);
      while (!feof(file))
	{
	  gchar buffer[1025];
	  memset(buffer, 0, sizeof(buffer));
	  fread(buffer, sizeof(buffer) - 1, 1, file);
	  if (ferror(file))
	    {
	      gda_ui_show_error(_("Error reading file\n %s"), filename);
	      break;
	    }
	  gtk_text_insert(text, NULL, NULL, NULL, buffer, strlen(buffer));
	}
      gtk_text_thaw(text);
      fclose(file);
      fe_status_bar_message(_("File %s loaded"), filename);
    }
  else gda_ui_show_error(_("Error opening file\n %s"), filename);
}

gboolean
fe_save_text_to_file (GtkText *text, gchar *filename)
{
  gchar *buffer;
  
  /* check parameters */
  g_return_val_if_fail(text != NULL, FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);
  
  /* get text widget's contents */
  buffer = gtk_editable_get_chars(GTK_EDITABLE(text), 0, gtk_text_get_length(text));
  if (buffer != NULL)
    {
      FILE *file = fopen(filename, "w");
      if (file != NULL)
	{
	  fwrite(buffer, gtk_text_get_length(text), 1,
		 file);
	  if (ferror(file))
	    gda_ui_show_error(_("Error writing to file\n %s"), filename);
          else fe_status_bar_message(_("Wrote file %s"), filename);
	  
	  /* close file */
	  fclose(file);
	}
      else gda_ui_show_error(_("Error opening file\n %s"), filename);
    }
}

void
fe_set_notebook_page (GtkNotebook *notebook, GtkWidget *tab)
{
  gint current;
  
  /* check parameters */
  g_return_if_fail(notebook != NULL);
  g_return_if_fail(tab != NULL);
  
  /* activate the page by id */
  current = gtk_notebook_page_num(notebook, tab);
  gtk_notebook_set_page(notebook, current);
}

void
fe_show_message (const gchar *format, ...)
{
  GtkWidget *dialog;
  va_list args;
  gchar sz[256];
    
  /* build the message string */
  va_start(args, format);
  vsprintf(sz, format, args);
  va_end(args);
    
  /* create the dialog box */
  dialog = gnome_message_box_new(sz, GNOME_MESSAGE_BOX_WARNING, GNOME_STOCK_BUTTON_OK, 0);
  gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WIDGET(glb_App));
  gnome_dialog_run(GNOME_DIALOG(dialog));
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

void
fe_status_bar_message (const gchar *format, ...)
{
  va_list args;
  gchar sz[256];
  
  va_start(args, format);
  vsprintf(sz, format, args);
  va_end(args);
  gnome_appbar_set_status(GNOME_APPBAR(glb_StatusBar), (const gchar *) sz);
}

gboolean
fe_yes_no_dialog (const gchar *format, ...)
{
  va_list args;
  gchar sz[256];
  GtkWidget *dialog, *table, *label, *icon;
  gint btn;
  
  va_start(args, format);
  vsprintf(sz, format, args);
  va_end(args);
  
  /* create dialog box */
  dialog = gnome_dialog_new(_("Question"), 
                            GNOME_STOCK_BUTTON_YES,
                            GNOME_STOCK_BUTTON_NO,
                            0);
  gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WIDGET(glb_App));
  table = gtk_table_new(1, 2, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, GNOME_PAD);
  icon = gnome_stock_new_with_icon(GNOME_STOCK_PIXMAP_ABOUT);
  gtk_table_attach(GTK_TABLE(table), icon, 0, 1, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  label = gda_ui_new_label_widget(sz);
  gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
                            
  btn = gnome_dialog_run(GNOME_DIALOG(dialog));
  gnome_dialog_close(GNOME_DIALOG(dialog));
  return (btn == 0 ? TRUE : FALSE);
}
