/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include "gdafe.h"

static void clear_cb (GtkWidget *w, gpointer data);
static void close_log_cb (GtkWidget *w, gpointer data);
static void refresh_cb (GtkWidget *w, gpointer data);
static void save_cb (GtkWidget *w, gpointer data);
static void select_list_cb (GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer data);

static GtkWidget *l_view = NULL;
static GtkWidget *l_list = NULL;
static GtkWidget *l_text = NULL;

static GnomeUIInfo logtoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh log"),
    refresh_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear file contents"),
    clear_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save As"), N_("Save log file with a different name"),
    save_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_SAVE_AS, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close this window"),
    close_log_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
get_provider_list (GtkWidget *list)
{
  static GtkWidget *pixmap = 0;
  register gint cnt = 0;
  GList *providers = gda_server_list();
  gda_ui_clear_clist(GTK_CLIST(list));
  if (providers != NULL)
    {
      GList *node;
      while ((node = g_list_first(providers)))
        {
          gchar *row[2], *empty_string = "";
          Gda_Server *server = (Gda_Server *) node->data;
          
          if (server)
            {
              row[0] = empty_string;
              row[1] = server->name;
              gtk_clist_append(GTK_CLIST(l_list), row);
              
              /* create shared pixmap */
              if (!pixmap)
                {
                  pixmap = gnome_stock_pixmap_widget(l_list, GNOME_STOCK_MENU_BOOK_RED);
                  if (!pixmap) return;
                  gtk_widget_show(pixmap);
                }
              gtk_clist_set_pixmap(GTK_CLIST(l_list), cnt, 0,
                                   GNOME_PIXMAP(pixmap)->pixmap,
                                   GNOME_PIXMAP(pixmap)->mask);
            }
          
          providers = g_list_remove(providers, (gpointer) server);
          gda_server_free(server);
          cnt++;
        }
    }
}

static void
real_close (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != 0);
  l_view = 0;
}

/*
 * Callbacks
 */
static void
clear_cb (GtkWidget *w, gpointer data)
{
  if (l_view != NULL)
    {
      GList *tmp = GTK_CLIST(l_list)->selection;
      if (tmp)
        {
          gchar *src = 0;
          guint row = GPOINTER_TO_UINT(tmp->data);
          
          gtk_clist_get_text(GTK_CLIST(l_list), row, 1, &src);
          if (src)
            {
              gchar *logfile = g_strdup_printf("/tmp/%s-srv.log", src);
              /* truncate file */
              if (truncate(logfile, 0) != 0)
                gda_ui_show_error(_("Could not truncate file %s"), logfile);
              else gnome_less_clear(GNOME_LESS(l_text));
              g_free((gpointer) logfile);
            }
        }
    }
}

static void
close_log_cb (GtkWidget *w, gpointer data)
{
  if (l_view != 0)
    {
      fe_destroy_view(l_view);
      l_view = 0;
    }
}

static void
refresh_cb (GtkWidget *w, gpointer data)
{
  if (l_view != 0)
    {
      GList *tmp = GTK_CLIST(l_list)->selection;
      if (tmp)
        {
          select_list_cb(GTK_CLIST(l_list), GPOINTER_TO_UINT(tmp->data), 1, 0, data);
        }
    }
}

static void
save_cb (GtkWidget *w, gpointer data)
{
  if (l_view != NULL)
    {
      gchar *file_name = fe_get_file_name(_("Save Log File"));
      if (file_name != NULL)
        {
          fe_save_text_to_file(GTK_TEXT(l_text), file_name);
          g_free((gpointer) file_name);
        }
    }
}

static void
select_list_cb (GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer data)
{
  if (l_view != NULL)
    {
      gchar *server = 0;
      gtk_clist_get_text(GTK_CLIST(l_list), row, 1, &server);
      if (server)
        {
          gchar *tmp = g_strdup_printf("/tmp/%s-srv.log", server);
          if (!gnome_less_show_file(GNOME_LESS(l_text), tmp))
            {
              gda_ui_show_error(_("Could not load file\n %s"), tmp);
              gnome_less_clear(GNOME_LESS(l_text));
              fe_status_bar_message(_("Ready"));
            }
          else fe_status_bar_message(_("File %s loaded"), tmp);
          g_free((gpointer) tmp);
        }
    }
}

/*
 * Public functions
 */
void
fe_open_log (GtkWidget *w, gpointer data)
{
  GtkWidget *scrolled_window;
  if (l_view == 0)
    {
      l_view = fe_new_view(_("GDA logs"), 6, 2, logtoolbar);
      fe_set_view_destroy_func(l_view, (FE_DestroyViewFunc) real_close, 0);

      /* create children */
      scrolled_window = gda_ui_new_scrolled_window_widget();
      l_list = gda_ui_new_clist_widget(0, 2);
      get_provider_list(l_list);
      gtk_signal_connect(GTK_OBJECT(l_list), "select_row",
                         GTK_SIGNAL_FUNC(select_list_cb), 0);
      gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
                                            l_list);
      fe_add_widget_to_view(l_view, 0, 1, 1, 2, scrolled_window);

      l_text = gnome_less_new();
      if (glb_EditorFont != 0) gnome_less_set_font(GNOME_LESS(l_text), glb_EditorFont);
      gtk_widget_show(l_text);
      fe_add_widget_to_view(l_view, 1, 1, 6, 2, l_text);
    }
  fe_display_view(l_view);
}
