/* Sample Rolodex Application
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>

#include <libgnorba/gnorba.h>
#include <gtk/gtk.h>

#include <gda-connection.h>
#include <gda-command.h>
#include <gda-recordset.h>
#include <gda-error.h>
#include <gda-field.h>
#include <gda.h>

#include "gdaform.h"
#include "rolonav.h"

#define TABLE_QULIFIER_IDX     0
#define TABLE_OWNER_IDX        1
#define TABLE_NAME_IDX         2
#define COLUMN_NAME            3
#define DATA_TYPE              4
#define TYPE_NAME              5
#define PRECISION              6
#define LENGTH                 7
#define SCALE                  8
#define RADIX                  9
#define NULLABLE               10


GtkWidget*         app;
GtkWidget*         status_bar;

    

void Exception( CORBA_Environment *ev )
{
  switch( ev->_major )
    {
    case CORBA_SYSTEM_EXCEPTION:
      fprintf( stderr, "CORBA system exception %s.\n",
	       CORBA_exception_id(ev));
      exit ( 1 );
    case CORBA_USER_EXCEPTION:
      fprintf( stderr, "CORBA user exception: %s.\n",
	       CORBA_exception_id( ev ) );
      exit ( 1 );
    default:
      break;
    }
}


gchar* provider = 0;
gchar* database = 0;
gchar* username = 0;
gchar* password = 0;
gchar* tablename = 0;

struct poptOption options[] = {
  { "provider", 's', POPT_ARG_STRING, &provider, 0, N_("database provider"), N_("PROVIDER")},
  { "database", 'd', POPT_ARG_STRING, &database, 0, N_("database name"), N_("DATABASE")},
  { "user",     'u', POPT_ARG_STRING, &username, 0, N_("user name"), N_("USER-NAME")},
  { "password", 'p', POPT_ARG_STRING, &password, 0, N_("password"), N_("PASSWORD")},
  { "table",    't', POPT_ARG_STRING, &tablename, 0, N_("table"), N_("TABLE NAME")},
  {0, 0, 0, 0, 0, 0, 0}
};

void
quit_cmd()
{
  gtk_main_quit();
}

static void
display_errors(GtkWidget* application, GList* errors)
{
  GList* current_error = errors;
  while (current_error)
    {
      Gda_Error* error = current_error->data;
      
      fprintf(stderr,"error: %s\n", error->description);
      gda_error_free(error);
      current_error = g_list_next(current_error);
    }
  g_list_free(errors);
}

static void
next_record(GtkWidget* button, gpointer data)
{
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);
  float    value;
  
  value = adj->value;
  value++;
  gtk_adjustment_set_value(adj, value);
}

static void
prev_record(GtkWidget* button, gpointer data)
{
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);
  float    value;

  value = adj->value;
  value--;
  gtk_adjustment_set_value(adj, value);
}

static void
first_record(GtkWidget* button, gpointer data)
{
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);
  
  gtk_adjustment_set_value(adj, 1);
}

static void
last_record(GtkWidget* button, gpointer data)
{
  GtkAdjustment* adj = GTK_ADJUSTMENT(data);
  
  gtk_adjustment_set_value(adj, G_MAXFLOAT);
}



static void
update_status_bar(GtkObject* obj, gpointer statusbar)
{
  gchar    bfr[128];

  g_snprintf(bfr, sizeof(bfr), _("Record %8.0f"), GTK_ADJUSTMENT(obj)->value);

  gnome_appbar_pop(GNOME_APPBAR(statusbar));
  gnome_appbar_push(GNOME_APPBAR(statusbar), bfr);
}


int
main(int argc, char* argv[])
{
  CORBA_ORB orb;
  CORBA_Environment ev;
  Gda_Connection*    cnc;
  GtkWidget*         form;
  gchar              cmd_bfr[1024];
  GtkWidget*         hbox;
  GtkWidget* vbox;
  GtkWidget* next;
  GtkWidget* prev;
  GtkWidget* first;
  GtkWidget* last;
  GtkWidget* top_hbox;
  GtkWidget* rnav;
  GtkObject* grs;
  
  CORBA_exception_init(&ev);
  
  orb = gnome_CORBA_init_with_popt_table("gda-client-test1", /* app_id */
					 "0.1",	/* app_version */
					 &argc,
					 argv,
					 options, /* options */
					 0, /* popt_flags */
					 0, /* return_ctx */
					 0, /* gnorba_flags */
					 &ev);
  Exception(&ev);
  app = gnome_app_new("rolodex", _("Rolodex demo"));
  gtk_signal_connect(GTK_OBJECT(app), "delete_event",
		     GTK_SIGNAL_FUNC(quit_cmd), NULL);
  
  cnc = gda_connection_new(orb);
  gda_connection_set_provider(cnc, provider);
  if (gda_connection_open(cnc, database, username, password) < 0)
    {
      fprintf(stderr,"gda_connection_open failed\n");
      exit(-1);
    }

  grs = gda_recordset_new();
  gda_recordset_set_connection(GDA_RECORDSET(grs), cnc);
  gda_recordset_set_cursorloc(GDA_RECORDSET(grs), GDA_USE_CLIENT_CURSOR);
  
  sprintf(cmd_bfr, "select * from %s", tablename);

  gda_recordset_open_txt(GDA_RECORDSET(grs), cmd_bfr, GDA_OPEN_STATIC, GDA_LOCK_PESSIMISTIC, 0);
  
  hbox = gtk_hbox_new(1,1);
  vbox = gtk_vbox_new(0,1);
  form = gdaform_new();
  gtk_box_pack_start(GTK_BOX(vbox), form, 1, 1, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, 0, 1, 0);
  next = gtk_button_new_with_label("Next");
  prev = gtk_button_new_with_label("Prev");
  first = gtk_button_new_with_label("First");
  last = gtk_button_new_with_label("Last");
  
  gtk_signal_connect(GTK_OBJECT(next),
		     "clicked",
		     GTK_SIGNAL_FUNC(next_record),
		     grs);

  gtk_signal_connect(GTK_OBJECT(prev),
		     "clicked",
		     GTK_SIGNAL_FUNC(prev_record),
		     grs);

  gtk_signal_connect(GTK_OBJECT(first),
		     "clicked",
		     GTK_SIGNAL_FUNC(first_record),
		     grs);

  gtk_signal_connect(GTK_OBJECT(last),
		     "clicked",
		     GTK_SIGNAL_FUNC(last_record),
		     grs);

  gtk_box_pack_start(GTK_BOX(hbox), first, 1, 1, 0);
  gtk_box_pack_start(GTK_BOX(hbox), prev, 1, 1, 0);
  gtk_box_pack_start(GTK_BOX(hbox), next, 1, 1, 0);
  gtk_box_pack_start(GTK_BOX(hbox), last, 1, 1, 0);
  
  gdaform_set_datasource(GDAFORM(form), GDA_RECORDSET(grs));

  top_hbox = gtk_hbox_new(0, 0);
  rnav = rolonav_new(GTK_ADJUSTMENT(grs));
  
  gtk_box_pack_start(GTK_BOX(top_hbox), rnav, 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(top_hbox), vbox, 0, 0, 0);


  gnome_app_set_contents(GNOME_APP(app), top_hbox);
  status_bar = gnome_appbar_new(TRUE, TRUE, FALSE);
  gtk_signal_connect_after(GTK_OBJECT(grs),
			   "value-changed",
			   GTK_SIGNAL_FUNC(update_status_bar),
			   status_bar);
  
  gnome_app_set_statusbar(GNOME_APP(app), status_bar);
  gtk_widget_show_all(app);
  gtk_main();
  return(0);
}

  

