/* GDA Oracle server
 * Copyright (c) 1998 by Rodrigo Moya
 * Copyright (C) 1999 Stephan Heinze
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>

#include "gda-oracle-types.h"
#include "gda-oracle-command.h"
#include "gda-oracle-connection.h"
#include "gda-oracle-field.h"
#include "gda-oracle-recset.h"

/* private functions */
static gboolean
fill_field_values (Gda_ORACLE_Recordset *recset, glong row)
{
  GList *node;
  
  g_return_val_if_fail(NULL != recset, FALSE);
  
  /* traverse all recordset values */
  node = g_list_first(recset->fields);

  fprintf(stderr,"getting data from row %ld\n", row);
  while (node != NULL)
    {
      Gda_ORACLE_Field *field = (Gda_ORACLE_Field *) node->data;
      if (field != NULL)
        {
          /* field->real_value must have been allocated in init_recset_fields */
	       if (field->real_value)
            {
              field->actual_length = field->defined_length;
              switch (field->sql_type)
                {
                case SQLT_CHR : /* VARCHAR2 */
                case SQLT_STR : /* null-terminated string */
                case SQLT_VCS : /* VARCHAR */
                case SQLT_RID : /* ROWID */
                case SQLT_LNG : /* LONG */
                case SQLT_LVC : /* LONG VARCHAR */
                case SQLT_AFC : /* CHAR */
                case SQLT_AVC : /* CHARZ */
                case SQLT_LAB : /* MSLABEL */
                  field->value->_d = GDA_TypeVarchar;
                  if (-1==field->indicator)
                    field->value->_u.lvc = CORBA_string_dup("<NULL>");
                  else
                    field->value->_u.lvc = CORBA_string_dup(field->real_value);
                  field->malloced = 1;
                break;

                case SQLT_VNU : /* VARNUM */
                case SQLT_NUM : /* NUMBER */
                  /* was mapped to string while Definining output fields */
                break;

                  case SQLT_FLT : /* FLOAT */
                    field->value->_d = GDA_TypeSingle;
                    if (-1==field->indicator)
                      field->value->_u.f = 0;
                    else
                      {
                        memcpy(&field->value->_u.f,
                               field->real_value,
                               sizeof(gfloat));
                      }
                  break; 

                case SQLT_INT : /* 8-bit, 16-bit and 32-bit integers */
                case SQLT_UIN : /* UNSIGNED INT */
                  field->value->_d = GDA_TypeInteger;
                  if (-1==field->indicator)
                    field->value->_u.i = 0;
                  else
                    {
                      memcpy(&field->value->_u.i,
                             field->real_value,
                             sizeof(glong));
                    }
                break;

                case SQLT_DAT : /* DATE */
                  field->value->_d = GDA_TypeDate;
                  if (-1==field->indicator)
                    {
                      memset(&field->value->_u.dbd,
                             0,
                             sizeof(GDA_DbDate));
                    }
                  else
                    {
                      memcpy(&field->value->_u.dbd,
                             field->real_value,
                             sizeof(GDA_DbDate));
                    }
                break;

                case SQLT_VBI : /* VARRAW */
                case SQLT_BIN : /* RAW */
                case SQLT_LBI : /* LONG RAW */
                case SQLT_LVB : /* LONG VARRAW */
                  field->value->_d = GDA_TypeBinary;
                  memcpy(&field->value->_u.fb,
                         field->real_value,
                         sizeof(GDA_VarBinString));
                break;

                default :
                  g_error("Unknown ORACLE data type in fill_field_values\n");
                break;
                }
              gda_oracle_field_dump(field);
            }
          else
            {
              field->actual_length = 0;
              memset(&field->value->_u, 0, sizeof(field->value->_u));
            }
        }
      node = g_list_next(node);
    }
  return (TRUE);
}

/* create a new result set object */
Gda_ORACLE_Recordset *
gda_oracle_recset_new (void)
{
  Gda_ORACLE_Recordset *rc = g_new0(Gda_ORACLE_Recordset, 1);
  return (rc);
}

/* free Recordset object */
void
gda_oracle_recset_free (Gda_ORACLE_Recordset *recset)
{
  g_return_if_fail(NULL != recset);

  gda_oracle_recset_close(recset);
  g_free((gpointer) recset);
}

/* cursor movement */
gint
gda_oracle_recset_move_next (Gda_ORACLE_Recordset *recset)
{
  sb4 status;

  g_return_val_if_fail(NULL != recset, -1);

  status = OCIStmtFetch(recset->hstmt,
                        recset->cnc->herr,
                        1,
                        OCI_FETCH_NEXT,
                        OCI_DEFAULT);
  switch (status)
    {
    case OCI_SUCCESS :
    case OCI_SUCCESS_WITH_INFO :
      recset->at_end = 0;
      recset->at_begin = 0;
      if (fill_field_values(recset, 0))
        {
          recset->pos++;
          return (0);
        }
      break;
    case OCI_NO_DATA :
      recset->at_end = 1;
      return (1);
    }
  add_error_to_connection(recset->cnc);
  return (-1);
}

gint
gda_oracle_recset_move_prev (Gda_ORACLE_Recordset *recset)
{
  /* call fill_field_values() when fetched ok */
  return (1);
}

/* close given recordset */
gint
gda_oracle_recset_close (Gda_ORACLE_Recordset *recset)
{
  GList *ptr;

  g_return_val_if_fail(NULL != recset, -1); 

  /* free Stmt handle allocated in gda_oracle_cmd_execute() */
  if (recset->hstmt != 0)
    {
      OCIHandleFree((dvoid *) recset->hstmt, OCI_HTYPE_STMT);
    }

  /* free fields' list */
  ptr = recset->fields;
  while (ptr != NULL)
    {
      gda_oracle_field_free((gpointer) ptr->data);
      ptr = g_list_next(ptr);
    }
  g_list_free(recset->fields);
  recset->fields = 0;
  return (0);
}

/* debugging */
void
gda_oracle_recset_dump (Gda_ORACLE_Recordset *recset)
{
}
