/* GDA Oracle server
 * Copyright (C) 1999 Rodrigo Moya
 * Copyright (C) 1999 Stephan Heinze
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "gda-oracle-command.h"
#include "gda-oracle-recset.h"
#include "gda-oracle-error.h"
#include "gda-oracle-connection.h"
#include "gda-oracle-field.h"

/* max_size of oracle-table-names */
#define ORA_NAME_BUFFER_SIZE 30

/* create a new command object */
Gda_ORACLE_Command *
gda_oracle_cmd_new (void)
{
  Gda_ORACLE_Command *rc = g_new0(Gda_ORACLE_Command, 1);
  return (rc);
}

void
gda_oracle_cmd_free (Gda_ORACLE_Command *cmd)
{
  if (cmd->cmd != 0)
    {
      g_free((gpointer) cmd->cmd);
    }
  cmd->cmd = 0;
  
  g_free((gpointer) cmd);
}

Gda_ORACLE_Connection *
gda_oracle_cmd_set_connection (Gda_ORACLE_Command *cmd, Gda_ORACLE_Connection *cnc)
{
  Gda_ORACLE_Connection *rc = cmd->cnc;
  cmd->cnc = cnc;
  return (rc);
}

static Gda_ORACLE_Recordset *
init_recset_fields (Gda_ORACLE_Recordset *recset)
{
  GList *ptr;
  
  g_return_val_if_fail(recset != NULL, NULL);
  
  /* free fields' list */
  ptr = recset->fields;
  while (ptr)
    {
      gda_oracle_field_free((gpointer) ptr->data);
      ptr = g_list_next(ptr);
    }
  g_list_free(recset->fields);
  recset->fields = NULL;
  
  /* traverse all fields in resultset */
  if (recset->stmt_type == OCI_STMT_SELECT)
    {
      gint counter = 1;
      OCIParam *param;
      sb4 parm_status;
      
      parm_status = OCIParamGet(recset->hstmt, OCI_HTYPE_STMT,
                                recset->cnc->herr, (dvoid**) &param, (ub4) counter);
      while (parm_status == OCI_SUCCESS)
        {
          glong col_name_len;
          gchar   name_buffer[ ORA_NAME_BUFFER_SIZE + 1 ];
          gchar*  pgchar_dummy;

          Gda_ORACLE_Field *f = gda_oracle_field_new();
          
          /* fill in field information */
          /* attention - names are not terminated by '\0' */
          OCIAttrGet((dvoid *) param, OCI_DTYPE_PARAM, 
                     &pgchar_dummy,(ub4 *) &col_name_len, OCI_ATTR_NAME,
                     recset->cnc->herr);
          memcpy(name_buffer, pgchar_dummy, col_name_len);
          name_buffer[col_name_len] = '\0';
          f->name = g_strdup(name_buffer);

          OCIAttrGet((dvoid *) param, OCI_DTYPE_PARAM,
                     &(f->sql_type), 0, OCI_ATTR_DATA_TYPE,
                     recset->cnc->herr);

          /* wanna get numeric as string */
          if( SQLT_NUM == f->sql_type
              || SQLT_VNU == f->sql_type )
            {
              f->sql_type = SQLT_AVC;
            }

          //f->c_type = ?

          OCIAttrGet((dvoid *) param, OCI_DTYPE_PARAM,
                     &(f->nullable), 0, OCI_ATTR_IS_NULL,
                     recset->cnc->herr);

          OCIAttrGet((dvoid *) param, OCI_DTYPE_PARAM,
                     &(f->precission), 0, OCI_ATTR_PRECISION,
                     recset->cnc->herr);

          OCIAttrGet((dvoid *) param, OCI_DTYPE_PARAM,
                     &(f->num_scale), 0, OCI_ATTR_SCALE,
                     recset->cnc->herr);

          OCIAttrGet((dvoid *) param, OCI_DTYPE_PARAM,
                     &(f->defined_length), 0, OCI_ATTR_DATA_SIZE,
                     recset->cnc->herr);
          f->actual_length = 0;
          f->malloced = 0;
          f->real_value = g_malloc0(f->defined_length);
          f->hdef = NULL; /* let oracle allocate it */
          f->indicator = 0;
          
          /* bind output columns */
          if (OCIDefineByPos(recset->hstmt,
                             &f->hdef,
                             recset->cnc->herr,
                             counter,
                             f->real_value,
                             f->defined_length,
                             f->sql_type,
                             &f->indicator,
                             0,
                             0,
                             OCI_DEFAULT) != OCI_SUCCESS)
            {
              add_error_to_connection(recset->cnc);
              return (recset);
            }
            
          /* add field to recordset's description */
          recset->fields = g_list_append(recset->fields, f);
          
          /* retrieve next field */
          counter++;
          parm_status = OCIParamGet(recset->hstmt, OCI_HTYPE_STMT,
                                    recset->cnc->herr, (dvoid**)&param, (ub4) counter);
        }
        
      /* if parm_status != OCI_NO_DATA, there was an error */
      if (parm_status != OCI_NO_DATA)
        add_error_to_connection(recset->cnc);
    }
  return (recset);
}

Gda_ORACLE_Recordset *
gda_oracle_cmd_execute (Gda_ORACLE_Command *cmd, Gda_ORACLE_Error *error,
			    const GDA_CmdParameterSeq *params, gulong options)
{
  Gda_ORACLE_Recordset *recset = NULL;
  OCIStmt              *hstmt = NULL;
  sword                 stmt_type;
  
  g_print("gda_oracle_cmd_execute(): enter\n");

  /* return a recordset even an empty one */
  recset = gda_oracle_recset_new();
  recset->fields = NULL;
  recset->cnc = cmd->cnc;

  
  /* initialization */
  if (OCI_SUCCESS == OCIHandleAlloc((dvoid *) cmd->cnc->henv,
                                    (dvoid **) &hstmt,
                                    OCI_HTYPE_STMT, 
                                    (size_t) 0, 
                                    (dvoid **) 0
                                   ))
    {
      if (OCI_SUCCESS == OCIStmtPrepare(hstmt,
                                        cmd->cnc->herr, 
                                        (CONST text *) cmd->cmd,
                                        strlen(cmd->cmd),
                                        OCI_NTV_SYNTAX,
                                        OCI_DEFAULT
                                       ))
        {
          stmt_type = 0;
          OCIAttrGet((dvoid *) hstmt, OCI_HTYPE_STMT, (dvoid *) &stmt_type,
                     NULL, OCI_ATTR_STMT_TYPE, cmd->cnc->herr);

          /* now, really execute the command */
          if ( OCI_SUCCESS == OCIStmtExecute(cmd->cnc->hservice,
                                             hstmt,
                                             cmd->cnc->herr,
                                             (ub4) ((OCI_STMT_SELECT==stmt_type)?0:1),
                                             (ub4) 0,
                                             (CONST OCISnapshot*) NULL,
                                             (OCISnapshot*) NULL,
                                             OCI_DEFAULT) )
            {
              if (OCI_STMT_SELECT==stmt_type)
                { 
                  recset->hstmt = hstmt;
                  recset->stmt_type = stmt_type;
                  return (init_recset_fields(recset));
                }
              else
                return recset;
            }
          else
            add_error_to_connection(recset->cnc);
        }
      else
        add_error_to_connection(recset->cnc);
    }
  g_warning("gda_oracle_cmd_execute(): leave with error\n");
  return NULL;
}
