/* GDA Manager
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <ctype.h>

#include <gnome.h>
#include <gda-server.h>
#include <gda-connection.h>

#include "gda-entries.h"

#define TREE_SPACING 5

static       GList*            providers = 0;
static       GtkWidget*        current_tree;
static       GtkWidget*        provider_pixmap = 0;
static       GtkWidget*        clean_name_pixmap = 0;
static       GtkWidget*        dirty_name_pixmap = 0;
static       Gdamgr_NodeInfo*  current_node = 0;




Gdamgr_ProviderInfo*
gdamgr_provider_add(Gda_Server* server)
{
  Gdamgr_ProviderInfo* retval;

  retval = g_new0(Gdamgr_ProviderInfo, 1);
  retval->server_info = gda_server_copy(server);
  retval->entries = 0;
  providers = g_list_append(providers, retval);
  return retval;
}

void
gdamgr_provider_free(Gdamgr_ProviderInfo* provider)
{
  GList* entrylist;

  entrylist = provider->entries;
  while (entrylist)
    {
      Gdamgr_EntryInfo* entry;
      
      entry = entrylist->data;
      gdamgr_entry_free(entry);
      entrylist = g_list_next(entrylist);
    }
  g_list_free(provider->entries);
  if (provider->server_info)
    gda_server_free(provider->server_info);
  g_free(provider);
}

Gdamgr_ProviderInfo*
gdamgr_provider_find_by_name(gchar* name)
{
  GList* providers_ptr = providers;

  while (providers_ptr)
    {
      Gdamgr_ProviderInfo* provider;

      provider = providers_ptr->data;
      if (strcmp(provider->server_info->name, name) == 0)
	return provider;
      providers_ptr = g_list_next(providers_ptr);
    }
  return 0;
}
	  
    
void
gdamgr_provider_remove(Gdamgr_ProviderInfo* provider)
{
  GList* ptr = g_list_find(providers, provider);
  if (ptr)
    {
      gdamgr_provider_free(provider);
      providers = g_list_remove_link(providers, ptr);
    }
  else
    g_warning("gdamgr_provider_remove: Inconsistence detected!!!!");
}

void
gdamgr_provider_remove_by_name(gchar* name)
{
  GList* ptr = providers;
  while (ptr)
    {
      Gdamgr_ProviderInfo* info;
      
      info = ptr->data;
      if (strcmp(info->server_info->name, name) == 0)
	{
	  gdamgr_provider_free(info);
	  g_list_free(info->entries);
	  providers = g_list_remove_link(providers, ptr);
	  break;
	}
      ptr = g_list_next(ptr);
    }
}

Gdamgr_ProviderInfo*
gdamgr_provider_get_by_name(gchar* name)
{
  GList* ptr = providers;
  while (ptr)
    {
      Gdamgr_ProviderInfo* info;

      info = ptr->data;
      if (strcmp(info->server_info->name, name) == 0)
	return info;
      ptr = g_list_next(ptr);
    }
  return 0;
}


void
gdamgr_provider_add_datasource(Gdamgr_ProviderInfo* provider,
			       Gdamgr_EntryInfo* info)
{
  info->provider = provider;
  provider->entries = g_list_append(provider->entries, info);
}

void
gdamgr_provider_remove_datasource(Gdamgr_ProviderInfo* provider,
				  Gdamgr_EntryInfo* info)
{
  provider->entries = g_list_remove(provider->entries, info);
}

GList*
gdamgr_provider_get_entries(Gdamgr_ProviderInfo* provider)
{
  return provider->entries;
}

void
gdamgr_provider_set_entries(Gdamgr_ProviderInfo* provider, GList* entries)
{
  provider->entries = entries;
}


Gdamgr_EntryInfo*
gdamgr_entry_new()
{
  Gdamgr_EntryInfo* info;

  info = g_new0(Gdamgr_EntryInfo, 1);
  return info;
}

void
gdamgr_entry_free(Gdamgr_EntryInfo* entry)
{
  if (entry->gda_name)
    g_free(entry->gda_name);
  if (entry->dsn)
    g_free(entry->dsn);
  if (entry->description)
    g_free(entry->description);
  if (entry->configurator)
    g_free(entry->configurator);
  g_free(entry);
}

void
gdamgr_entry_remove(Gdamgr_EntryInfo* entry)
{
  Gdamgr_ProviderInfo* provider = entry->provider;
  GList* ptr = g_list_find(gdamgr_provider_get_entries(provider), entry);

  if (ptr)
    {
      gdamgr_entry_free(entry);
      gdamgr_provider_set_entries(provider, g_list_remove_link(gdamgr_provider_get_entries(provider), ptr));
    }
  else
    g_warning("gdamgr_entry_remove: Inconsistence detected!!!!");
}

static GList*
gdamgr_entries_list(void)
{
  GList* providers_ptr = providers;
  GList* retval = 0;

  while (providers_ptr)
    {
      GList* entries;
      
      entries = gdamgr_provider_get_entries(providers_ptr->data);
      while (entries)
	{
	  retval = g_list_append(retval, entries->data);
	  entries = g_list_next(entries);
	}
      providers_ptr = g_list_next(providers_ptr);
    }
  return retval;
}

Gdamgr_NodeInfo*
gdamgr_get_selected_node(void)
{
  return current_node;
}

static void
tree_expand(GtkWidget* tree, GtkCTreeNode* node, gint column)
{
  Gdamgr_NodeInfo* node_info =
    gtk_ctree_node_get_row_data(GTK_CTREE(tree), node);
    
  switch(node_info->ntype)
    {
    case ProviderNode:
      {
	Gdamgr_ProviderInfo* info;
	
	info = (Gdamgr_ProviderInfo*)node_info->node;
	info->expanded = 1;
      }
      break;
    case EntryNode:
      {
	;
      }
      break;
    }
}

static void
unselect_row_callback(GtkWidget* tree, GtkCTreeNode* node, gint column)
{

  current_tree = tree;
  
  switch(current_node->ntype)
    {
    case ProviderNode:
      {
	Gdamgr_ProviderInfo* info;
	
	info = (Gdamgr_ProviderInfo*)current_node->node;
	info->selected = 0;
      }
      break;
    case EntryNode:
      {
	Gdamgr_EntryInfo* info;
	
	info = (Gdamgr_EntryInfo*)current_node->node;
	info->selected = 0;
      }
      break;
    }
}

static void
select_row_callback(GtkWidget* tree, GtkCTreeNode* node, gint column)
{
  GtkWidget* notebook;
  GtkWidget* page;
  gint       pagenr = -1;
  gint*      ptr;

  current_tree = tree;
  current_node = gtk_ctree_node_get_row_data(GTK_CTREE(tree), node);

  notebook = gtk_object_get_data(GTK_OBJECT(tree), "notebook");
  if (!notebook)
    g_error("select_row_callback: can't get notebook from tree");
  switch(current_node->ntype)
    {
    case ProviderNode:
      {
	Gdamgr_ProviderInfo* info;
	GtkWidget*           widget;
	
	info = (Gdamgr_ProviderInfo*)current_node->node;
	info->selected = 1;
	
	ptr = gtk_object_get_data(GTK_OBJECT(tree), "provider-page");
	pagenr = *ptr;
	gtk_notebook_set_page(GTK_NOTEBOOK(notebook), pagenr);
	page = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), pagenr);
	
	if (info->server_info->type == GOAD_SERVER_SHLIB)
	  widget = gtk_object_get_data(GTK_OBJECT(page), "shlib_server");
	else
	  widget = gtk_object_get_data(GTK_OBJECT(page), "exe_server");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), 1);
	
	widget = gtk_object_get_data(GTK_OBJECT(page), "repository ID");
	gtk_entry_set_text(GTK_ENTRY(widget), info->server_info->repo_id);
	
	widget = gtk_object_get_data(GTK_OBJECT(page), "description");
	gtk_entry_set_text(GTK_ENTRY(widget), info->server_info->comment);
	
	widget = gtk_object_get_data(GTK_OBJECT(page), "location");
	gtk_entry_set_text(GTK_ENTRY(widget), info->server_info->location);
      }
      break;
    case EntryNode:
      {
	Gdamgr_EntryInfo* info;
	GtkWidget* widget;

	info = (Gdamgr_EntryInfo*)current_node->node;
	info->selected = 1;
	ptr = gtk_object_get_data(GTK_OBJECT(tree), "dsn-page");
	pagenr = *ptr;
	gtk_notebook_set_page(GTK_NOTEBOOK(notebook), pagenr);
	page = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), pagenr);
	
	widget = gtk_object_get_data(GTK_OBJECT(page), "provider");
	gtk_entry_set_text(GTK_ENTRY(widget), info->provider->server_info->name);
	
	widget = gtk_object_get_data(GTK_OBJECT(page), "DSN");
	gtk_entry_set_text(GTK_ENTRY(widget), info->dsn);
	
	widget = gtk_object_get_data(GTK_OBJECT(page), "description");
	gtk_entry_set_text(GTK_ENTRY(widget), info->description);
	
	widget = gtk_object_get_data(GTK_OBJECT(page), "configuration");
	gtk_entry_set_text(GTK_ENTRY(widget), info->configurator);
	break;
      }
    }

  if (pagenr >= 0)
    gtk_notebook_set_page(GTK_NOTEBOOK(notebook), pagenr);
}


void
accept_provider_changes(GtkWidget* button, gpointer data)
{

  GtkWidget* msg;

  msg = gnome_message_box_new(_("Configuration of GOAD servers not implemented"),
			      GNOME_MESSAGE_BOX_GENERIC, GNOME_STOCK_BUTTON_OK, NULL);
  gtk_widget_show(msg);
}

void
entry_edited(GtkWidget* entry, gpointer data)
{
  Gdamgr_EntryInfo* entry_info;
  Gdamgr_NodeInfo*  node = gdamgr_get_selected_node();
  gchar* txt;
  
  g_print("entry_changed for %p called\n", node);
  if (node->ntype != EntryNode)
    {
      g_error("accept_entry_changes: Inconsistency detected");
    }
  if (!GTK_WIDGET_HAS_FOCUS(entry))
    {
      g_print("Entry has no focus\n");
      return;
    }
  entry_info = (Gdamgr_EntryInfo*)node->node;
  if (!entry_info->edited)
    {
      txt = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
      entry_info->edited = 1;
      gdamgr_refresh_provider_tree(current_tree);
      gtk_entry_set_text(GTK_ENTRY(entry), txt);
      g_free(txt);
    }
}
  
gint
entry_activated(GtkWidget* entry, gpointer data)
{
  Gdamgr_NodeInfo*  node = gdamgr_get_selected_node();
  g_print("entry_activated for %p called\n", node);
  return 0;
}

gint
entry_deactivated(GtkWidget* entry, gpointer data)
{
  Gdamgr_NodeInfo*  node = gdamgr_get_selected_node();

  g_print("entry_deactivated for %p called\n", node);

  return 0;
}

void
accept_entry_changes(GtkWidget* button, gpointer data)
{
  Gdamgr_EntryInfo* entry_info;
  Gdamgr_NodeInfo*  node = gdamgr_get_selected_node();
  GtkWidget*        entry;
  gchar*            txt;
  gint              need_refresh = 0;
  
  if (node->ntype != EntryNode)
    {
      g_error("accept_entry_changes: Inconsistency detected");
    }
  entry_info = (Gdamgr_EntryInfo*)node->node;
  entry = gtk_object_get_data(GTK_OBJECT(data), "provider");
  txt = gtk_entry_get_text(GTK_ENTRY(entry));
  if (strcmp(entry_info->provider->server_info->name, txt) != 0)
    {
      Gdamgr_ProviderInfo* new_provider = gdamgr_provider_find_by_name(txt);
      if (!new_provider)
	{
	  gchar      errormsg[128];
	  GtkWidget* msg;
	  
	  g_snprintf(errormsg, sizeof(errormsg), _("Unknown Provider %s"), txt);
	  msg = gnome_message_box_new(errormsg, GNOME_MESSAGE_BOX_ERROR, GNOME_STOCK_BUTTON_OK, NULL);
	  return ;
	}
      gdamgr_provider_remove_datasource(entry_info->provider, entry_info);
      gdamgr_provider_add_datasource(new_provider, entry_info);
      new_provider->expanded = 1;
      need_refresh = 1;
    }

  entry = gtk_object_get_data(GTK_OBJECT(data), "DSN");
  txt = gtk_entry_get_text(GTK_ENTRY(entry));
  if (strcmp(entry_info->dsn, txt) != 0)
    {
      g_free(entry_info->dsn);
      entry_info->dsn = g_strdup(txt);
      need_refresh = 1;
    }

  entry = gtk_object_get_data(GTK_OBJECT(data), "description");
  txt = gtk_entry_get_text(GTK_ENTRY(entry));
  if (strcmp(entry_info->description, txt) != 0)
    {
      g_free(entry_info->description);
      entry_info->description = g_strdup(txt);
      need_refresh = 1;
    }

  entry = gtk_object_get_data(GTK_OBJECT(data), "configuration");
  txt = gtk_entry_get_text(GTK_ENTRY(entry));
  if (strcmp(entry_info->configurator, txt) != 0)
    {
      g_free(entry_info->configurator);
      entry_info->configurator = g_strdup(txt);
      need_refresh = 1;
    }
  if (need_refresh)
    {
      gdamgr_refresh_provider_tree(current_tree);
      need_save = 1;
    }
}

static void
remove_node(GtkCTree* tree, GtkCTreeNode* node, gpointer data)
{
  Gdamgr_NodeInfo* info = gtk_ctree_node_get_row_data(tree, node);

  g_free(info);
  gtk_ctree_node_set_row_data(tree,node, 0);
  gtk_ctree_remove_node(GTK_CTREE(tree), node);
}

void
gdamgr_refresh_provider_tree(GtkWidget* tree)
{
  static gint           initialized;
  GtkCTreeNode*         parent;
  GtkCTreeNode*         sibling;
  GtkCTreeNode*         selected = 0;
  Gdamgr_ProviderInfo*  provider;
  Gdamgr_EntryInfo*     gda_entry;
  GList*                providers_ptr;
  GList*                gda_entries_ptr;
  gchar*                text[1];
  

  gchar* parent_name;
  gchar* sibling_name;
  
  if (!providers)
    return;
  
  gtk_clist_freeze(GTK_CLIST(tree));

  gtk_ctree_post_recursive(GTK_CTREE(tree), 0, remove_node, 0);

  if (!provider_pixmap)
    provider_pixmap =
      gnome_pixmap_new_from_file(gnome_pixmap_file("gnome-db/gda-servers.png"));
  if (!clean_name_pixmap)
    {
      clean_name_pixmap = gnome_pixmap_new_from_file(gnome_pixmap_file("gnome-db/db-black.png"));
      dirty_name_pixmap = gnome_pixmap_new_from_file(gnome_pixmap_file("gnome-db/db-red.png"));
    }
  providers_ptr = providers;
  parent = 0;
  sibling = 0;
  
  while (providers_ptr)
    {
      Gdamgr_NodeInfo* node_data;

      provider = providers_ptr->data;
      text[0] = provider->server_info->name;
      parent = gtk_ctree_insert_node(GTK_CTREE(tree), 0, sibling,
				     text,
				     TREE_SPACING,
				     GNOME_PIXMAP(provider_pixmap)->pixmap,
				     GNOME_PIXMAP(provider_pixmap)->mask,
				     GNOME_PIXMAP(provider_pixmap)->pixmap,
				     GNOME_PIXMAP(provider_pixmap)->mask,
				     FALSE, provider->expanded ? TRUE
				     : FALSE);
      node_data = g_new0(Gdamgr_NodeInfo, 1);
      node_data->ntype = ProviderNode;
      node_data->node  = provider;
      if (provider->selected)
	selected = parent;
      
      gtk_ctree_node_set_row_data(GTK_CTREE(tree), parent, node_data);
      
      parent_name = provider->server_info->name;
			 
      gda_entries_ptr = gdamgr_provider_get_entries(provider);
      sibling = 0;
      sibling_name = 0;
      while (gda_entries_ptr)
	{
	  GtkWidget* name_pixmap;

	  gda_entry = gda_entries_ptr->data;
	  name_pixmap = gda_entry->edited ? dirty_name_pixmap : clean_name_pixmap;
	  
	  text[0] = gda_entry->gda_name;
	  sibling = gtk_ctree_insert_node(GTK_CTREE(tree),
					  parent, sibling,
					  text,
					  TREE_SPACING,
					  GNOME_PIXMAP(name_pixmap)->pixmap,
					  GNOME_PIXMAP(name_pixmap)->mask,
					  GNOME_PIXMAP(name_pixmap)->pixmap,
					  GNOME_PIXMAP(name_pixmap)->mask,
					  FALSE, TRUE);
	  
	  if (gda_entry->selected)
	    selected = sibling;
	  node_data = g_new0(Gdamgr_NodeInfo, 1);
	  node_data->ntype = EntryNode;
	  node_data->node  = gda_entry;
	  gtk_ctree_node_set_row_data(GTK_CTREE(tree), sibling, node_data);
	  
	  sibling_name = gda_entry->gda_name;
	  gda_entries_ptr = g_list_next(gda_entries_ptr);
	}
      sibling = parent;
      sibling_name = provider->server_info->name;
      providers_ptr = g_list_next(providers_ptr);
    }
  if (!initialized)
    {
      initialized = 1;
      gtk_signal_connect(GTK_OBJECT(tree), "tree_select_row",
			 GTK_SIGNAL_FUNC(select_row_callback), tree); 
      gtk_signal_connect(GTK_OBJECT(tree), "tree_unselect_row",
			 GTK_SIGNAL_FUNC(unselect_row_callback),
			 tree);
      gtk_signal_connect(GTK_OBJECT(tree), "tree_expand",
			 GTK_SIGNAL_FUNC(tree_expand),
			 tree);
    }
  if (selected)
    gtk_ctree_select(GTK_CTREE(tree), selected);
  gtk_clist_thaw(GTK_CLIST(tree));
}

void
new_provider_cmd(void)
{
  ;
}

void
new_datasource_cmd(void)
{
  Gdamgr_EntryInfo* new_entry;
  Gdamgr_ProviderInfo* provider;
  static GtkWidget* gda_prompt_box = 0;
  static GtkWidget* gda_name_entry = 0;
  
  if (!current_node)
    {
      static GtkWidget* box = 0;

      if (!box)
	{
	  box = gnome_message_box_new(_("No Provider selected"),
				      "error");
	  gtk_window_set_modal(GTK_WINDOW(box), TRUE);
	  gnome_dialog_close_hides(GNOME_DIALOG(box), TRUE);
	}
      gtk_widget_show(box);
      return;
    }
  if (!gda_prompt_box)
    {
      GtkWidget* label;
      GtkWidget* table;
      
      gda_prompt_box = gnome_dialog_new(_("Enter GDA Name"),
					GNOME_STOCK_BUTTON_OK, NULL);
      table = gtk_table_new(1, 2, FALSE);
      label = gtk_label_new(_("GDA Name"));
      gtk_misc_set_alignment(GTK_MISC(label), 0.1, 1.0);
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
		       GTK_FILL, 0, 0, 0);
      gda_name_entry = gtk_entry_new();
      gtk_table_attach(GTK_TABLE(table), gda_name_entry, 1, 2, 0, 1,
		       GTK_FILL, 0, 0, 0);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(gda_prompt_box)->vbox),
			 table, TRUE, TRUE, GNOME_PAD);
      gnome_dialog_close_hides(GNOME_DIALOG(gda_prompt_box), TRUE);
      gnome_dialog_set_close(GNOME_DIALOG(gda_prompt_box), TRUE);
    }
  gtk_widget_show_all(gda_prompt_box);
  gnome_dialog_run(GNOME_DIALOG(gda_prompt_box));
  new_entry = g_new0(Gdamgr_EntryInfo, 1);
  new_entry->gda_name = g_strdup(gtk_entry_get_text(GTK_ENTRY(gda_name_entry)));
  new_entry->dsn = g_strdup("DSN");
  new_entry->description = g_strdup("Description");
  new_entry->configurator = g_strdup("None");
  new_entry->expanded = 0;
  new_entry->selected = 1;
  if (current_node->ntype == ProviderNode)
    {
      provider = current_node->node;
      provider->selected = 0;
    }
  else
    provider = ((Gdamgr_EntryInfo*)current_node->node)->provider;
  gdamgr_provider_add_datasource(provider, new_entry);
  gdamgr_refresh_provider_tree(current_tree);
}

void
save_cmd(void)
{
  gdamgr_store_gda_info();
}


void
quit_cmd(void)
{
  gint cancel;
  
  GtkWidget* save_box;


  if (need_save)
    {
      save_box = gnome_message_box_new(_("You have edited the GDA configuration file.\n\nSave the file?"),
				       GNOME_MESSAGE_BOX_QUESTION,
				       GNOME_STOCK_BUTTON_OK,
				       GNOME_STOCK_BUTTON_CANCEL,
				       NULL);
      gnome_message_box_set_modal(GNOME_MESSAGE_BOX(save_box));
      cancel = gnome_dialog_run(GNOME_DIALOG(save_box));
      if (!cancel)
	gdamgr_store_gda_info();
    }
  gtk_main_quit();
}



void
delete_cmd(void)
{
  Gdamgr_NodeInfo* node = gdamgr_get_selected_node();

  switch(node->ntype)
    {
    case ProviderNode:
      gdamgr_provider_remove((Gdamgr_ProviderInfo*)node->node);
      break;
    case EntryNode:
      gdamgr_entry_remove((Gdamgr_EntryInfo*)node->node);
      break;
    }
  gdamgr_refresh_provider_tree(current_tree);
}




static gchar*
trim_string(gchar* s)
{
  gint length = strlen(s);

  while (length && isspace(s[--length]))
    ;
  s[length+1] = '\0';
  return s;
}
  
gint
gdamgr_build_gda_info()
{
  GList*            providers_ptr;
  GList*            providers_tmp;
  gpointer          gda_iterator;
  GString*          path;
  gchar*            provider_name;
  gchar*            gda_name;
  gchar*            section_name;
  Gdamgr_EntryInfo*    entry;
  Gdamgr_ProviderInfo* provider;
    
  providers_ptr = providers;
  path = g_string_new("");
  while (providers_ptr)
    {
      Gdamgr_ProviderInfo* provider;

      provider = providers_ptr->data;
      gdamgr_provider_free(provider);
      providers_ptr = g_list_next(providers_ptr);
    }
  g_list_free(providers);
  providers = 0;
  
  providers_ptr = providers_tmp = gda_server_list();
  while (providers_ptr)
    {
      gdamgr_provider_add(providers_ptr->data);
      providers_ptr = g_list_next(providers_ptr);
    }
  g_list_free(providers_tmp);
  
  gda_iterator = gnome_config_init_iterator("/gdalib/Datasources");
  while ((gda_iterator = gnome_config_iterator_next(gda_iterator, &gda_name, &section_name)))
    {

      entry = gdamgr_entry_new();
      g_string_sprintf(path, "/gdalib/%s/", trim_string(gda_name));
      gnome_config_push_prefix(path->str);

      provider_name = gnome_config_get_string("Provider");
      entry->gda_name = gda_name;
      entry->dsn = gnome_config_get_string("DSN");
      entry->description = gnome_config_get_string("Description");
      entry->configurator = gnome_config_get_string("Configurator");

      provider = gdamgr_provider_get_by_name(provider_name);
      if (!provider)
	g_warning("Provider '%s' as requested from entry '%s' is not known", provider_name, gda_name);
      else
	gdamgr_provider_add_datasource(provider, entry);
      
      gnome_config_pop_prefix();
    }
  g_string_free(path, 1);
  return 0;
}

  
gint
gdamgr_store_gda_info(void)
{
  GList*               datasources;
  GList*               datasources_ptr;
  GString*             path;
  Gdamgr_EntryInfo*    entry;

  gnome_config_push_prefix("/gdalib/Datasources/");
  datasources_ptr = datasources = gdamgr_entries_list();
  while (datasources_ptr)
    {

      entry = datasources_ptr->data;
      gnome_config_set_string(entry->gda_name, entry->gda_name);
      datasources_ptr = g_list_next(datasources_ptr);
    }
  gnome_config_pop_prefix();
  path = g_string_new("");

  datasources_ptr = datasources;
  while (datasources_ptr)
    {
      entry = datasources_ptr->data;
      g_string_sprintf(path, "/gdalib/%s/", entry->gda_name);
      entry->edited = 0;
      gnome_config_push_prefix(path->str);
      gnome_config_set_string("Provider", entry->provider->server_info->name);
      gnome_config_set_string("DSN", entry->dsn);
      gnome_config_set_string("Description", entry->description);
      gnome_config_set_string("Configurator", entry->configurator);
      gnome_config_pop_prefix();
      datasources_ptr = g_list_next(datasources_ptr);
    }
  g_list_free(datasources);
  g_string_free(path, 1);
  gdamgr_refresh_provider_tree(current_tree);
  need_save = 0;
  gnome_config_sync();
  return 0;
}
