/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

static void close_editor_cb (GtkWidget *w, gpointer data);

static void capture_command_cb (GtkWidget *w, gpointer data);
static void close_file_cb (GtkWidget *w, gpointer data);
static void find_file_cb (GtkWidget *w, gpointer data);
static void new_file_cb (GtkWidget *w, gpointer data);
static void open_file_cb (GtkWidget *w, gpointer data);
static void open_shell_cb (GtkWidget *w, gpointer data);
static void save_as_cb (GtkWidget *w, gpointer data);
static void save_file_cb (GtkWidget *w, gpointer data);

static void copy_cb (GtkWidget *w, gpointer data);
static void cut_cb (GtkWidget *w, gpointer data);
static void paste_cb (GtkWidget *w, gpointer data);
static void text_changed_cb (GtkEditable *text, gpointer data);

static void real_close (GtkWidget *w, gpointer data);

static GtkWidget *l_view = NULL;
static GtkWidget *l_notebook = NULL;
static glong      l_count = 0;
static GnomeUIInfo edittoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("New"), N_("Create a new empty file"),
    new_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Open an existing file"),
    open_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save current file"),
    save_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_SAVE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save As..."), N_("Save file with another name"),
    save_as_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_SAVE_AS, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Print"), N_("Print current file"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PRINT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close current file"),
    close_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text"),
    cut_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text to clipboard"),
    copy_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste clipboard's contents"),
    paste_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PASTE, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close this window"),
    close_editor_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_QUIT, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static GtkWidget *
get_current_text (void)
{
  gint current = gtk_notebook_get_current_page(GTK_NOTEBOOK(l_notebook));
  GtkWidget *scroll = gtk_notebook_get_nth_page(GTK_NOTEBOOK(l_notebook), current);
  if (scroll != 0)
    {
      return ((GtkWidget *) gtk_object_get_data(GTK_OBJECT(scroll), 
                                                "FE_Editor_TextWidget"));
    }
  return (0);
}

static gboolean
is_text_changed (GtkText *text)
{
  gchar *prop;
  g_return_val_if_fail(GTK_IS_TEXT(text), FALSE);
  
  prop = gtk_object_get_data(GTK_OBJECT(text), "FE_Editor_Changed");
  if (prop && !strcmp(prop, "1"))
    return (TRUE);
  return (FALSE);
}

static void
real_close (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != NULL);
  
  while (gtk_notebook_get_current_page(GTK_NOTEBOOK(l_notebook)) >= 0)
    {
      close_file_cb(0, 0);
    }
  l_view = 0;
}

/*
 * Callbacks
 */
static void
capture_command_cb (GtkWidget *w, gpointer data)
{
  GtkWidget *dialog, *table, *label;
  GtkWidget *cmd_entry, *dir_entry;
  gint btn;
  
  g_return_if_fail(l_view != NULL);
  
  dialog = gnome_dialog_new(_("Run Command"), 
                            GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL, NULL);
  
  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, GNOME_PAD);
  
  label = gda_ui_new_label_widget(_("Command"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  cmd_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), cmd_entry, 1, 2, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  
  label = gda_ui_new_label_widget(_("Directory"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  dir_entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_table_attach(GTK_TABLE(table), dir_entry, 1, 2, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  gtk_widget_show(table);

  btn = gnome_dialog_run(GNOME_DIALOG(dialog));
  if (btn == 0)
    {
      gchar *cmd, *dir, *old_dir;
      FILE *fp;
      
      old_dir = g_get_current_dir();
      cmd = gtk_entry_get_text(GTK_ENTRY(cmd_entry));
      dir = gtk_entry_get_text(GTK_ENTRY(dir_entry));
      
      if (dir && strlen(dir) > 0)
        {
          if (chdir(dir) == -1)
            {
              gda_ui_show_error(_("Could not change to directory '%s'"), dir);
              return;
            }
        }
      if ((fp = popen(cmd, "r")))
        {
          GtkText *text;
          gchar buffer[1024];
          gchar *header;
          
          fe_status_bar_message(_("Running '%s'"), cmd);
          new_file_cb(NULL, NULL);
          text = get_current_text();
          gtk_text_freeze(text);
          
          /* write header */
          header = g_strdup_printf(_("Output from '%s'\nIn directory '%s'\n\n"),
                                   cmd, dir && strlen(dir) > 0 ? dir : old_dir);
          gtk_text_insert(text, NULL, NULL, NULL, header, strlen(header));
          g_free((gpointer) header);
          
          while (fgets(buffer, sizeof(buffer), fp))
            {
              gtk_text_insert(text, NULL, NULL, NULL, buffer, strlen(buffer));
            }
          gtk_text_thaw(text);
          pclose(fp);
          
          fe_status_bar_message(_("Command '%s' completed"), cmd);
        }
      else gda_ui_show_error(_("Unable to execute '%s'"), cmd);
      
      /* come back to old directory */
      if (dir && strlen(dir) > 0)
        {
          if (chdir(old_dir) == -1)
            {
              gda_ui_show_error(_("Could not change to directory '%s'"), old_dir);
            }
        }
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

static void
close_editor_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != NULL);
  
  fe_destroy_view(l_view);
  l_view = NULL;
}

static void
close_file_cb (GtkWidget *w, gpointer data)
{
  if (l_view != NULL)
    {
      gint current = gtk_notebook_get_current_page(GTK_NOTEBOOK(l_notebook));
      if (current >= 0)
        {
          GtkWidget *text = get_current_text();
          if (text && is_text_changed(GTK_TEXT(text)))
            {
              gchar *file_name = gtk_object_get_data(GTK_OBJECT(text),
                                                     "FE_Editor_FileName");
              if (fe_yes_no_dialog(_("File '%s' has been changed. "
                                     "Do you want to save the changes?"),
                                     file_name ? file_name : "Untitled"))
                {
                  save_file_cb(NULL, NULL);
                }
            }
          gtk_notebook_remove_page(GTK_NOTEBOOK(l_notebook), current);
        }
    }
}

static void
find_file_cb (GtkWidget *w, gpointer data)
{  
  gchar *args[1];
  
  g_return_if_fail(l_view != NULL);

  args[0] = "gsearchtool";  
  if (gnome_execute_async(NULL, 1, args) == -1)
    gda_ui_show_error(_("Unable to execute '%s'"), args[0]);
}

static void
new_file_cb (GtkWidget *w, gpointer data)
{
  if (l_view != 0)
    {
      gchar *label = g_strdup_printf("Untitled%ld", l_count);
      GtkWidget *text = fe_new_text_widget();
      GtkWidget *scroll = gda_ui_new_scrolled_window_widget();
      gtk_signal_connect(GTK_OBJECT(text), "changed",
                         GTK_SIGNAL_FUNC(text_changed_cb), 0);
      gtk_container_add(GTK_CONTAINER(scroll), text);
      gtk_object_set_data(GTK_OBJECT(scroll), "FE_Editor_TextWidget", (gpointer) text);
      
      gtk_notebook_prepend_page(GTK_NOTEBOOK(l_notebook), scroll, 
                                gtk_label_new(label));
      fe_set_notebook_page(GTK_NOTEBOOK(l_notebook), scroll);
      g_free(label);
      l_count++; 
    }
}

static void
open_file_cb (GtkWidget *w, gpointer data)
{
  if (l_view != 0)
    {
      gchar *file_name = fe_get_file_name(_("Open File"));
      if (file_name != 0)
        {
          GtkWidget *text = fe_new_text_widget();
          GtkWidget *scroll = gda_ui_new_scrolled_window_widget();
          gtk_signal_connect(GTK_OBJECT(text), "changed",
                             GTK_SIGNAL_FUNC(text_changed_cb), 0);
          gtk_container_add(GTK_CONTAINER(scroll), text);
          gtk_object_set_data(GTK_OBJECT(scroll), "FE_Editor_TextWidget", (gpointer) text);
          gtk_notebook_prepend_page(GTK_NOTEBOOK(l_notebook), scroll, 
                                    gtk_label_new(g_basename(file_name)));
          fe_set_notebook_page(GTK_NOTEBOOK(l_notebook), scroll);
          fe_load_file_in_text(GTK_TEXT(text), file_name);
          gtk_object_set_data(GTK_OBJECT(text), "FE_Editor_FileName", (gpointer) file_name);
        }
    }
}

static void
save_as_cb (GtkWidget *w, gpointer data)
{
  if (l_view != 0)
    {
      GtkWidget *text = get_current_text();
      if (text != 0)
        {
          gchar *file_name = fe_get_file_name(_("Save As..."));
          if (file_name != 0)
            {
              fe_save_text_to_file(GTK_TEXT(text), file_name);
              gtk_object_set_data(GTK_OBJECT(text), "FE_Editor_FileName", (gpointer) file_name);
              gtk_object_set_data(GTK_OBJECT(text), "FE_Editor_Changed", "0");
            }
        }
    }
}

static void
save_file_cb (GtkWidget *w, gpointer data)
{
  if (l_view != 0)
    {
      GtkWidget *text = get_current_text();
      if (text != NULL && is_text_changed(GTK_TEXT(text)))
        {
          gchar *file_name = (gchar *) gtk_object_get_data(GTK_OBJECT(text),
                                                           "FE_Editor_FileName");
          if (file_name != 0)
            fe_save_text_to_file(GTK_TEXT(text), file_name);
          else
            {
              file_name = fe_get_file_name(_("Save As..."));
              if (file_name != 0)
                {
                  fe_save_text_to_file(GTK_TEXT(text), file_name);
                  gtk_object_set_data(GTK_OBJECT(text), "FE_Editor_FileName", 
                                      (gpointer) file_name);
                  gtk_object_set_data(GTK_OBJECT(text), "FE_Editor_Changed", "0");
                }
            }
        }
    }
}

static void
copy_cb (GtkWidget *w, gpointer data)
{
  if (l_view != NULL)
    {
      GtkWidget *text = get_current_text();
      if (text != NULL)
        {
          gtk_editable_copy_clipboard(GTK_EDITABLE(text));
        }
    }
}

static void
cut_cb (GtkWidget *w, gpointer data)
{
  if (l_view != NULL)
    {
      GtkWidget *text = get_current_text();
      if (text != NULL)
        {
          gtk_editable_cut_clipboard(GTK_EDITABLE(text));
        }
    }
}

static void
paste_cb (GtkWidget *w, gpointer data)
{
  if (l_view != NULL)
    {
      GtkWidget *text = get_current_text();
      if (text != NULL)
        {
          gtk_editable_paste_clipboard(GTK_EDITABLE(text));
        }
    }
}

static void
text_changed_cb (GtkEditable *text, gpointer data)
{
  g_return_if_fail(GTK_IS_TEXT(text));
  gtk_object_set_data(GTK_OBJECT(text), "FE_Editor_Changed", "1");
}

/*
 * Public functions
 */
void
fe_open_editor (GtkWidget *w, gpointer data)
{
  if (l_view == NULL)
    {
      GtkWidget *tools_bar, *button, *box;
      l_view = fe_new_view(_("Editor"), 2, 2, edittoolbar);
      fe_set_view_destroy_func(l_view, (FE_DestroyViewFunc) real_close, NULL);
      
      box = gtk_hbox_new(FALSE, 0);
      fe_add_widget_to_view(l_view, 0, 1, 2, 2, box);
      
      tools_bar = gtk_vbox_new(FALSE, 0);
      button = gda_ui_new_button_widget_with_pixmap(_("Run"), GNOME_STOCK_MENU_JUMP_TO);
      gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(capture_command_cb), (gpointer) 0);
      gtk_box_pack_start(GTK_BOX(tools_bar), button, 0, 0, 0);
      
      button = gda_ui_new_button_widget_with_pixmap(_("Shell"), GNOME_STOCK_MENU_ATTACH);
      gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(fe_run_shell), (gpointer) 0);
      gtk_box_pack_start(GTK_BOX(tools_bar), button, 0, 0, 0);
      
      button = gda_ui_new_button_widget_with_pixmap(_("Find"), GNOME_STOCK_MENU_SEARCH);
      gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(find_file_cb), (gpointer) 0);
      gtk_box_pack_start(GTK_BOX(tools_bar), button, 0, 0, 0);
                              
      gtk_widget_show(tools_bar);
      gtk_box_pack_start(GTK_BOX(box), tools_bar, 0, 0, GNOME_PAD);
      
      l_notebook = fe_new_notebook_widget();
      gtk_box_pack_start(GTK_BOX(box), l_notebook, 1, 1, GNOME_PAD);
      
      l_count = 0;
      new_file_cb(NULL, NULL);
    }
  fe_display_view(l_view);
}

void
fe_run_shell (GtkWidget *w, gpointer data)
{
  gchar *args[1];
  const gchar *default_term = "gnome-terminal";
  
  if (!(args[0] = gnome_config_get_string(FE_CONFIG_GENERAL_SHELL)))
    {
      if (!(args[0] = g_getenv("TERM")))
        {
          args[0] = default_term;
        }
    }
  if (gnome_execute_async(NULL, 1, args) == -1)
    gda_ui_show_error(_("Unable to execute '%s'"), args[0]);
}