/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"

/* function prototypes */
static void close_current_browser (GtkWidget *, gpointer );
static void refresh_grids (GtkWidget *, gpointer );
static void select_list_cb (GtkWidget *, gpointer );

/* global variables */
extern gint l_openconns;
static GList *l_openwins = NULL;

/* structure for schema */
static struct GSQLFE_ObjectInfo 
{
  gchar *string;
  gint gda_sch1;
  gint gda_sch2;
  gint gda_data1;
  gint gda_data2;
} l_objects[] =
{
  { "Views", GDA_Connection_GDCN_SCHEMA_VIEWS, GDA_Connection_no_CONSTRAINT, 0, 0 },
  { "Tables", GDA_Connection_GDCN_SCHEMA_TABLES, GDA_Connection_no_CONSTRAINT, 0, 0},
  { "Indexes", GDA_Connection_GDCN_SCHEMA_INDEXES, GDA_Connection_no_CONSTRAINT, 0, 0 },
  { "Procedures", GDA_Connection_GDCN_SCHEMA_PROCS, GDA_Connection_no_CONSTRAINT, 0, 0 },
  { NULL, 0, 0, 0, 0 }
};

/* private functions */
static void
close_current_browser (GtkWidget *w, gpointer data)
{
  struct GSQLFE_BrowserInfo *info = browser_get_current_window();
  if (info != NULL)
    {
      gint current = gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                           info->tab);
      gtk_notebook_remove_page(GTK_NOTEBOOK(ui_get_work_area()), current);
      l_openwins = g_list_remove(l_openwins, (gpointer) info);
      g_free((gpointer) info);
      l_openconns--;
    }
}

static struct GSQLFE_ObjectInfo *
get_schema_from_string (gchar *str)
{
  register gint cnt;
  for (cnt = 0; l_objects[cnt].string != NULL; cnt++)
    {
      if (!g_strcasecmp(str, l_objects[cnt].string))
	return (&l_objects[cnt]);
    }
}

static void
refresh_grids (GtkWidget *w, gpointer data)
{
  struct GSQLFE_BrowserInfo *info;
  /* select_list_cb() expects the table list widget as its 
     first argument and the current browser as the second one */
  info = browser_get_current_window();
  if (info != NULL)
    {
      select_combo_cb(NULL, (gpointer) info);
      select_list_cb(GTK_WIDGET(info->list), (gpointer) info);
    }
}

static void
select_combo_cb (GtkWidget *w, gpointer data)
{
  struct GSQLFE_BrowserInfo *info = (struct GSQLFE_BrowserInfo *) data;
  if (info != NULL)
    {
      Gda_Recordset *rowset;
      gchar *obj_type = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(info->combo)->entry));
      struct GSQLFE_ObjectInfo *schema = get_schema_from_string(obj_type);
      fprintf(stderr," select_combo_cb: Schema = '%s'\n", obj_type);
      if (schema != NULL)
	{
	  /* show list of objects */
	  rowset = gda_connection_open_schema(info->db, 
					      schema->gda_sch1,
					      schema->gda_sch2, NULL);
	  if (rowset != NULL)
	    {
	      gchar  bfr[128];
	      gulong position = gda_recordset_move(rowset, 1, 0);
	      /* clear the screen list */
	      gtk_list_clear_items(GTK_LIST(info->list), 0, -1);
	      while (position != GDA_RECORDSET_INVALID_POSITION &&
		     !gda_recordset_eof(rowset))
		{
		  Gda_Field *field = gda_recordset_field_idx(rowset, 2);
		  if (field != NULL)
		    {
		      GtkWidget *w = 
			gtk_list_item_new_with_label(gda_stringify_value(bfr, 
						     sizeof(bfr) - 1, field));
		      gtk_container_add(GTK_CONTAINER(info->list), w);
		      gtk_widget_show(w);
		    }
		  /* FETCH next row */
		  position = gda_recordset_move(rowset, 1, 0);
		}
	      /* close the record set */
	      gda_recordset_close(rowset);
	      gda_recordset_free(rowset);
	    }
	  else ui_show_error("Error getting schema");
	}
    }
}

static void
select_list_cb (GtkWidget *w, gpointer data)
{
  struct GSQLFE_BrowserInfo *info = (struct GSQLFE_BrowserInfo *) data;
  if (w != NULL && info != NULL)
    {
      GtkObject *list_item;
      Gda_Recordset *rowset;
      gchar *obj_name, *obj_type;
      struct GSQLFE_ObjectInfo *schema;
      GList *node = g_list_first(GTK_LIST(w)->selection);
      if (node != NULL)
        {
	  /* get object type */
	  obj_type = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(info->combo)->entry));
	  schema = get_schema_from_string(obj_type);
	  if (schema != NULL)
	    {
	      /* get object name */
	      list_item = GTK_OBJECT(node->data);
	      obj_name = gtk_object_get_data(list_item, glb_list_item_data_key);
	      fprintf(stderr,"select_list_cb: obj_name = '%s'\n", obj_name);
	      /* show object's description */
	      rowset = gda_connection_open_schema(info->db, 
				                          schema->gda_data1,
			                              schema->gda_data2, obj_name);
	      if (rowset != NULL)
		{
		  info->grid1 = grid_show_resultset(info->grid1, rowset,
						    GTK_CONTAINER(info->grid_scroll1));
		  gda_recordset_close(rowset);
		  gda_recordset_free(rowset);
		}
	      else
		{
		  info->grid1 = grid_new_empty(info->grid1, 
					       GTK_CONTAINER(info->grid_scroll1));
		  ui_show_error("Error fetching data");
		}
	    }
	}
    }
}

/* close all browser windows related with this connection */
void
browser_close_all (Gda_Connection *gdb)
{
  GList *node = g_list_first(l_openwins);
  while (node != NULL)
    {
      struct GSQLFE_BrowserInfo *info = 
             (struct GSQLFE_BrowserInfo *) node->data;
      /* move to next item in list */
      node = g_list_next(node);
      if (info != NULL && info->db == gdb)
        {
          gint this_one =
              gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                    info->tab);
          gtk_notebook_remove_page(GTK_NOTEBOOK(ui_get_work_area()),
                                   this_one);
          /* free all memory */
          l_openwins = g_list_remove(l_openwins, info);
          g_free((gpointer) info);
        }
    }
}

struct GSQLFE_BrowserInfo *
browser_get_current_window (void)
{
  gint this_one;
  gint current = gtk_notebook_current_page(GTK_NOTEBOOK(ui_get_work_area()));
  if (current >= 0)
    {
      GList *node = g_list_first(l_openwins);
      struct GSQLFE_BrowserInfo *info;
      while (node != NULL)
        {
          info = node->data;
          this_one = gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                           info->tab);
          if (current == this_one) 
            return (info);
          node = g_list_next(node);
        }
    }
  return (NULL);
}

/* open a new browser window */
GnomeUIInfo browsertoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("New"), N_("Create new object"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh grids"),
    refresh_grids, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Script"), 
    N_("Generate script from object's definition"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_INDEX, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close this window"),
    close_current_browser, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};

void
browser_new_window (GtkWidget *gdb, gpointer data)
{
  struct GSQLFE_BrowserInfo *brw_info;
  register gint cnt;
  gchar label[64], *str;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      /* create browser object */
      if ((brw_info = g_malloc(sizeof(struct GSQLFE_BrowserInfo))))
	{
	  brw_info->db = info->db;
	  /* create browser window */
	  brw_info->tab = gtk_table_new(3, 3, FALSE);
	  str = gda_connection_get_dsn(brw_info->db);
	  sprintf(label, "Tables: %s", str);
	  gtk_notebook_append_page(GTK_NOTEBOOK(ui_get_work_area()), brw_info->tab,
				   gtk_label_new(label));
	  g_free((gpointer) str);
	  /* create common toolbar */
	  brw_info->toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, 
					      GTK_TOOLBAR_ICONS);
	  gnome_app_fill_toolbar(GTK_TOOLBAR(brw_info->toolbar),
				 browsertoolbar, NULL);
	  gtk_table_attach(GTK_TABLE(brw_info->tab), brw_info->toolbar, 
			   0, 3, 0, 1,
			   GTK_FILL | GTK_SHRINK,
			   GTK_FILL | GTK_SHRINK, 
			   3, 3);
	  gtk_widget_show(brw_info->toolbar);
	  /* create combo box */
	  brw_info->combo = ui_new_combo_in_table(brw_info->tab,
						  0, 1, 1, 2);
	  for (cnt = 0; l_objects[cnt].string != NULL; cnt++)
	    {
	      GtkWidget *li = gtk_list_item_new_with_label(l_objects[cnt].string);
	      gtk_container_add(GTK_CONTAINER(GTK_COMBO(brw_info->combo)->list), li);
	      gtk_widget_show(li);
	      gtk_object_set_data(GTK_OBJECT(li), glb_list_item_data_key,
				  (gchar *) l_objects[cnt].string);
	    }
	  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(brw_info->combo)->list), 
			     "selection_changed",
			     GTK_SIGNAL_FUNC(select_combo_cb), 
			     (gpointer) brw_info);
	  /* create scrolled window to put list inside */
	  brw_info->scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	  gtk_scrolled_window_set_policy(
			 GTK_SCROLLED_WINDOW(brw_info->scrolled_window),
		      	 GTK_POLICY_AUTOMATIC,
	       		 GTK_POLICY_AUTOMATIC);
	  gtk_table_attach(GTK_TABLE(brw_info->tab), 
			   brw_info->scrolled_window, 0, 1, 2, 3,
			   GTK_FILL | GTK_SHRINK,
			   GTK_SHRINK | GTK_EXPAND | GTK_FILL,
			   3, 3);
	  gtk_widget_show(brw_info->scrolled_window);
	  /* create object list */
	  brw_info->list = gtk_list_new();
	  gtk_signal_connect(GTK_OBJECT(brw_info->list), "selection_changed",
			     GTK_SIGNAL_FUNC(select_list_cb), 
			     (gpointer) brw_info);
	  gtk_list_set_selection_mode(GTK_LIST(brw_info->list), 
				      GTK_SELECTION_SINGLE);
	  gtk_scrolled_window_add_with_viewport(
			    GTK_SCROLLED_WINDOW(brw_info->scrolled_window), 
			    brw_info->list);
	  gtk_widget_show(brw_info->list);
	  /* create grid */
	  brw_info->grid_scroll1 = gtk_scrolled_window_new(NULL, NULL);
	  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(brw_info->grid_scroll1),
					 GTK_POLICY_AUTOMATIC,
					 GTK_POLICY_AUTOMATIC);
	  gtk_table_attach(GTK_TABLE(brw_info->tab), brw_info->grid_scroll1,
			   1, 3, 1, 3, GTK_FILL | GTK_EXPAND,
			   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			   3, 3);
	  gtk_widget_show(brw_info->grid_scroll1);
	  brw_info->grid1 = grid_new_empty(NULL, GTK_CONTAINER(brw_info->grid_scroll1));
	  /* register this window */
	  l_openwins = g_list_append(l_openwins, (gpointer) brw_info);
	  gtk_widget_show(brw_info->tab);
	  ui_set_notebook_page(GTK_NOTEBOOK(ui_get_work_area()), brw_info->tab);
	  l_openconns++;
	}
      else ui_show_error("Cannot allocate %d bytes", 
			 sizeof(struct GSQLFE_BrowserInfo));
    }
}
