/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome.h>
#include <gda.h>


#include "gda-recordset.h"
#include "gda-command.h"

/* per default there's no upper limit on the number of rows which can
 * be fetched from the data source
 */
#define GDA_DEFAULT_MAXROWS 0

/* per default fetch 64 rows at a time */
#define GDA_DEFAULT_CACHESIZE 64



/*
 * this function will change as soon as the client side
 * error objects are here
 */
static gint
Exception(CORBA_Environment *ev)
{
  switch (ev->_major)
    {
    case CORBA_SYSTEM_EXCEPTION:
      fprintf(stderr,"CORBA system exception %s.\n", CORBA_exception_id(ev));
      return -1;
    case CORBA_USER_EXCEPTION:
      fprintf(stderr,"CORBA user exception %s.\n", CORBA_exception_id(ev));
      return -1;
    default:
    }
  return 0;
}


static void
free_chunks(GList* chunks)
{
  GDA_Row* row;
  GList*   ptr;

  ptr = chunks;
  while(ptr)
    {
      row = ptr->data;
      CORBA_free(row);
      ptr = g_list_next(ptr);
    }
}

__inline__ static GDA_Row*
row_by_idx(Gda_Recordset* rs, gint idx)
{
  gint                 offset;
  GDA_Recordset_Chunk* chunk;
  
  if (idx > rs->cachesize)
    offset = rs->cachesize-1;
  offset = idx;
  chunk = g_list_nth(rs->chunks, offset)->data;

  return &chunk->_buffer[0];
}

  
  

/**
 * gda_recordset_new:
 *
 * Allocates space for a new recordset.
 *
 * Returns: the allocated recordset object
 */

Gda_Recordset*
gda_recordset_new(void)
{
  Gda_Recordset* rs;

  rs = g_new0(Gda_Recordset, 1);
  rs->maxrows = GDA_DEFAULT_MAXROWS;
  rs->cachesize = GDA_DEFAULT_CACHESIZE;
  rs->bof = 1;
  rs->eof = 1;
  rs->readonly = 1;
  rs->forwardonly = 1;
  rs->chunks_length = 0;
  rs->open = 0;
  return rs;
}


/**
 * gda_recordset_free:
 * @rs: the recordset which should be destroyed.
 *
 * This function frees all memory allocated by the recordset and
 * destroys all associations with commands and connections.
 *
 */
void
gda_recordset_free(Gda_Recordset* rs)
{
  if (rs->open)
    gda_recordset_close(rs);
  if (rs->internal_cmd)
    {
      gda_command_free(rs->internal_cmd);
    }
  g_free(rs);
}

/**
 * gda_recordset_close:
 * @rs: the recordset to close
 *
 * This function closes the recordset and frees the memory occupied by 
 * the actual data items. The recordset can be opened
 * again, doing the same query as before.
 * It is guaranteeed that the data cached by the recordset is
 * refetched from the server. Use this function is some
 * characteristics of the recordset must be changed
 *
 */
void
gda_recordset_close(Gda_Recordset* rs)
{
  CORBA_Environment ev;

  CORBA_exception_init(&ev);
  
  if (!rs->open)
    return;
  rs->open = 0;
  if (rs->corba_rs)
    {
      fprintf(stderr,"Closing Recordset in the server\n");
      GDA_Recordset_close(rs->corba_rs, &ev);
      Exception(&ev);
    }
  rs->corba_rs = CORBA_OBJECT_NIL;
  if (rs->chunks)
    {
      free_chunks(rs->chunks);
    }
  rs->chunks = 0;
}


/**
 * gda_recordset_bof:
 * @rs: Recordset which should be checked
 *
 * This function is used to check if the recordset cursor is beyond
 * the first row.
 * If this function returns %TRUE any of the functions which actually returns
 * the value of a field an error is returned because the cursor
 * doesn't point to a row. 
 * Returns: 1 if the cursor is beyond the first record or the
 * recordset is empty, 0 otherwise.
 */
gint
gda_recordset_bof(Gda_Recordset* rs)
{
  if (!rs->open)
    return 1;
  if (rs->current_row)
    return rs->bof;
  return 0;
}



/**
 * gda_recordset_eof:
 * @rs:  Recordset which should be checked
 *
 * This function is used to check if the recordset cursor is after
 * the last row. 
 * If this function returns %TRUE any of the functions which actually returns
 * the value of a field an error is returned because the cursor
 * doesn't point to a row. 
 * Returns: 1 if the cursor is after the last record or the
 * recordset is empty, 0 otherwise.
 */
gint
gda_recordset_eof(Gda_Recordset* rs)
{
  if (!rs->open)
    return 1;
  if (rs->current_row)
    return rs->eof;
  return 0;
}

/**
 * gda_recordset_move:
 * @rs: the recordset
 * @count: The number of records to skip
 * @bookmark: if not NULL, the cursor is positioned relative to the
 * record described by this paramter. seee
 * gda_recordset_get_bookmark() how to get bookmark values for
 * records.
 *
 * Moves the cursor of the recordset forward or backward. @count is
 * the number of records to move. If @count is negative the cursor is
 * moved towards the beginning. The function causes the recordset to
 * actually fetch records from the data source. Each fetch
 * from the data source fetches #cachesize rows in one turn. A maximum 
 * of #maxrows rows can be fetched. 
 *
 * If the cursor is on the second row and the @count parameter is -10, 
 * then the cursor is position in front of the first record
 * available. gda_rcordset_bof() will return %TRUE and 
 * the return value of the function is two, because the cursor actually
 * moved two records. 
 *
 * Returns: the number of the record the cursor is addressing after
 * the move or %GDA_RECORDSET_INVALID_POSITION if there was an error
 * fetching the rows.  
 */

gulong
gda_recordset_move(Gda_Recordset* rs, gint count, gpointer bookmark)
{

  CORBA_Environment    ev;
  GDA_Recordset_Chunk* chunk;
  
  g_return_val_if_fail(rs->forwardonly && count >= 0, GDA_RECORDSET_INVALID_POSITION);
  g_return_val_if_fail(rs->corba_rs != NULL, GDA_RECORDSET_INVALID_POSITION);
  g_return_val_if_fail(rs->open, GDA_RECORDSET_INVALID_POSITION);
  
  CORBA_exception_init(&ev);

  if (rs->eof && count >= 0)
    return GDA_RECORDSET_INVALID_POSITION;
  if (rs->bof && count <= 0)
    return GDA_RECORDSET_INVALID_POSITION;
  if (count == 0)
    return 0;
  
  chunk = GDA_Recordset_fetch(rs->corba_rs, 1, &ev);
#if 0
  fprintf(stderr,"chunk->_buffer =  %p\n", chunk->_buffer);
  fprintf(stderr,"chunk->_length =  %d\n", chunk->_length);
  fprintf(stderr,"chunk->_maximum = %d\n", chunk->_maximum);
#endif
  if (Exception(&ev))
    return GDA_RECORDSET_INVALID_POSITION;

  if (chunk->_length == 0)
    {
      if (count > 0)
	rs->eof = 1;
      else
	rs->bof = 1;
      if (!rs->chunks)
	{
	  if (count > 0)
	    rs->bof = 1;
	  else
	    rs->eof = 1;
	  rs->current_index = GDA_RECORDSET_INVALID_POSITION;
	}
      CORBA_free(chunk);
      printf("EOF FOR RECORDSET\n");
      return rs->current_index;
    }
  rs->eof = 0;
  if (!rs->chunks)
    rs->current_index = -1;
  rs->current_index += count;
  if (rs->chunks_length > rs->cachesize)
    {
      GList* head = rs->chunks;
      CORBA_free(head->data);
      rs->chunks = g_list_next(rs->chunks);
      g_list_free(head);
    }
  else
    {
      rs->chunks = g_list_append(rs->chunks, chunk);
    }
  rs->current_row = row_by_idx(rs, rs->current_index);
  return rs->current_index;
}

/**
 * gda_recordset_move_first:
 * @rs: the recordset
 *
 * Moves the cursor of the recordset to the first record. 
 * 
 * If the cursor is already on the the fist record nothing happen.
 *
 * Returns: the position of the cursor, or
 * %GDA_RECORDSET_INVALID_POSITION
 * if there was an error. 
 */

gulong
gda_recordset_move_first(Gda_Recordset* rs)
{

  CORBA_Environment    ev;
  GDA_Recordset_Chunk* chunk;
  
  g_return_val_if_fail(rs != 0, -1);

  CORBA_exception_init(&ev);
  chunk = GDA_Recordset_fetch(rs->corba_rs, rs->cachesize, &ev);
  if (Exception(&ev) < 0)
    return GDA_RECORDSET_INVALID_POSITION;
  return 0;
}

/**
 * gda_recordset_move_last:
 * @rs: the recordset
 *
 * Moves the cursor of the recordset to the first record. 
 * 
 * If the cursor is already on the the fist record nothing happen.
 *
 * Returns: the position of the cursor, or
 * %GDA_RECORDSET_INVALID_POSITION
 * if there was an error. 
 */

gulong
gda_recordset_move_last(Gda_Recordset* rs)
{

  return GDA_RECORDSET_INVALID_POSITION;
}


/**
 * gda_recordset_move_first:
 * @rs: the recordset
 *
 * Moves the cursor of the recordset to the first record. 
 * 
 * If the cursor is already on the the fist record nothing happen.
 *
 * Returns: the position of the cursor, or
 * %GDA_RECORDSET_INVALID_POSITION
 * if there was an error. 
 */

gulong
gda_recordset_move_next(Gda_Recordset* rs)
{

  return gda_recordset_move(rs, 1, 0);
}

/**
 * gda_recordset_field_idx:
 * @rs: the recordset
 * @idx: the index of the field in the current row
 *
 * Returns a pointer to the field at position @idx of
 * the current row of the recordset.
 *
 * Returns: a pointer to a field structor or NULL if 
 * @idx is out of range
 */

GDA_Field*
gda_recordset_field_idx(Gda_Recordset* rs, gint idx)
{
  g_return_val_if_fail(rs != NULL, 0);
  g_return_val_if_fail(rs->chunks != NULL, 0);
  g_return_val_if_fail(rs->current_row != NULL, 0);
  g_return_val_if_fail(idx >= 0, 0);
  g_return_val_if_fail(rs->open, 0);
  
  return &rs->current_row->_buffer[idx];
}

/**
 * gda_recordset_rowsize:
 *
 * @rs: the recordset
 *
 * Returns the number of fields in a row of the current recordset
 *
 * Returns: the number of rows in the recordset or
 * 0 if there was an error.
 */
gint
gda_recordset_rowsize(Gda_Recordset* rs)
{
  g_return_val_if_fail(rs != NULL, 0);
  g_return_val_if_fail(rs->current_row != NULL, 0);
  
  return rs->current_row->_length;
}
  
