/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <memory.h>

#include "odbc.h"
#include "sqldriver.h"

#include "gda-command.h"
#include "gda-recset.h"
#include "gda-error.h"
#include "gda-connection.h"
#include "gda-field.h"

Gda_ODBC_Command*
gda_odbc_cmd_new(void)
{
  Gda_ODBC_Command* rc = g_new0(Gda_ODBC_Command, 1);
  return rc;
}


void
gda_odbc_cmd_free(Gda_ODBC_Command* cmd)
{
  GList* ptr;
  
  if (cmd->cmd)
    g_free(cmd->cmd);

  cmd->cmd = 0;

  ptr = cmd->recsets;
  while(ptr)
    {
      gda_odbc_recset_free(ptr->data);
      ptr = g_list_next(ptr);
    }
  g_list_free(cmd->recsets);
  cmd->recsets = 0;
  g_free(cmd);
	
}



Gda_ODBC_Connection*
gda_odbc_cmd_set_connection(Gda_ODBC_Command* cmd,
			    Gda_ODBC_Connection* cnc)
{
  Gda_ODBC_Connection* rc = cmd->cnc;

  cmd->cnc = cnc;
  return rc;
}



gint
gda_odbc_gda2sqlctype(GDA_ValueType tag)
{
  
  switch (tag)
    {
    case GDA_TypeNull:
      g_warning("GOT GDA_TYPE_NULL");
      abort();
      break;
    case GDA_TypeBigint:
      return SQL_C_CHAR;	/* unsigned char* */
      break;
    case GDA_TypeBinary:
      return SQL_C_BINARY;	/* unsigned char*  */
      break;
    case GDA_TypeBoolean:	/* unsigned char */
      return SQL_C_BIT;
      break;
    case GDA_TypeBstr:
      return SQL_C_BINARY;	/* unsigned char* */
      break;
    case GDA_TypeChar:
      return SQL_C_CHAR;	/* unsigned char*  */
      break;
    case GDA_TypeCurrency:
      return SQL_C_CHAR;	/* unsigned char* */
      break;
    case GDA_TypeDate:
      return SQL_C_LONG;	/* time_t */
      break;
    case GDA_TypeDbDate:
      return SQL_C_DATE;	/* struct SQLDATE */
      break;
    case GDA_TypeDbTime:
      return SQL_C_TIME;	/* struct SQLTIME */
      break;
    case GDA_TypeDbTimestamp:
      return SQL_C_TIMESTAMP;	/* struxct sqltimestamp */
      break;
    case GDA_TypeDecimal:	
      return SQL_C_CHAR;	/* unsigned char* */
      break;
    case GDA_TypeDouble:
      return SQL_C_DOUBLE;	/* double */
    case GDA_TypeError:
      return SQL_C_CHAR;	/* unsigned char* */
      break;
    case GDA_TypeInteger:
      return SQL_C_SLONG;	/* unsigned long */
    case GDA_TypeLongvarbin:
      return SQL_C_BINARY;	/* unsigned char* */
    case GDA_TypeLongvarchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeLongvarwchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeNumeric:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeSingle:
      return SQL_C_FLOAT;	/* float */
    case GDA_TypeSmallint:
      return SQL_C_SSHORT;	/* short */
    case GDA_TypeTinyint:
      return SQL_C_STINYINT;	/* unsigned char */
    case GDA_TypeUBigint:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeUSmallint:
      return SQL_C_USHORT;	/* unsigned short */
    case GDA_TypeVarchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeVarbin:
      return SQL_C_BINARY;	/* unsigned char* */
    case GDA_TypeVarwchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeFixchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeFixbin:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeFixwchar:
      return SQL_C_CHAR;	/* unsigned char* */
    }
  g_warning("gda_odbc_gda2sqlctype: reached end of function, no matching GDA type for %d found\n", tag);
  abort();
  return -1;
}

static void*
value_2_ptr(GDA_Value* v)
{
  
  switch (v->_d)
    {
    case GDA_TypeNull:
      g_warning("GOT GDA_TYPE_NULL");
      abort();
      break;
    case GDA_TypeBigint:
      return v->_u.lvc;		/* unsigned char */
      break;
    case GDA_TypeBinary:
      return v->_u.lvb._buffer;	/* unsigned char*  */
      break;
    case GDA_TypeBoolean:	/* unsigned char */
      return &v->_u.b;
      break;
    case GDA_TypeBstr:
      return v->_u.lvb._buffer;	/* unsigned char* */
      break;
    case GDA_TypeChar:
      return v->_u.lvc;		/* unsigned char*  */
      break;
    case GDA_TypeCurrency:
      return v->_u.lvc;		/* unsigned char* */
      break;
    case GDA_TypeDate:
      return &v->_u.d; /* struct SQLDATE */
      break;
    case GDA_TypeDbDate:
      return &v->_u.dbd; /* struct SQLDATE */
      break;
    case GDA_TypeDbTime:
      return &v->_u.dbt;	/* struct SQLTIME */
      break;
    case GDA_TypeDbTimestamp:
      return &v->_u.dbtstamp;	/* struct SQLTIMESTAMP */
      break;
    case GDA_TypeDecimal:	
      return v->_u.dec.value._buffer; /* unsigned char* */
      break;
    case GDA_TypeDouble:
      return &v->_u.dp;		/* double */
    case GDA_TypeError:
      return &v->_u.lvc;	/* unsigned char* */
      break;
    case GDA_TypeInteger:
      return &v->_u.i;		/* CORBA_long */
    case GDA_TypeLongvarbin:
      return v->_u.lvb._buffer; /* unsigned char* */
    case GDA_TypeLongvarchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeLongvarwchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeNumeric:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeSingle:
      return &v->_u.f;		/* float */
    case GDA_TypeSmallint:
      return &v->_u.si;		/* short */
    case GDA_TypeTinyint:
      return &v->_u.c;		/* unsigned char */
    case GDA_TypeUBigint:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeUSmallint:
      return &v->_u.us;		/* unsigned short */
    case GDA_TypeVarchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeVarbin:
      return v->_u.lvb._buffer;	/* unsigned char* */
    case GDA_TypeVarwchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeFixchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeFixbin:
      return v->_u.lvb._buffer; /* unsigned char* */
    case GDA_TypeFixwchar:
      return v->_u.lvc;		/* unsigned char* */
    }
  g_warning("value_2_ptr: reached end of function, no matching GDA type found\n");
  abort();
  return (void*)0;
}
      
static long
value_2_inputsize(GDA_Value* v)
{
  
  switch (v->_d)
    {
    case GDA_TypeNull:
      g_warning("GOT GDA_TYPE_NULL");
      abort();
      break;
    case GDA_TypeBigint:
      return SQL_NTS;		/* unsigned char */
      break;
    case GDA_TypeBinary:
      return v->_u.lvb._length;	/* unsigned char*  */
      break;
    case GDA_TypeBoolean:	/* unsigned char */
      return sizeof(v->_u.b);
      break;
    case GDA_TypeBstr:
      return v->_u.lvb._length;	/* unsigned char* */
      break;
    case GDA_TypeChar:
      return SQL_NTS;		/* unsigned char*  */
      break;
    case GDA_TypeCurrency:
      return SQL_NTS;		/* unsigned char* */
      break;
    case GDA_TypeDate:
      return sizeof(v->_u.d); /* struct SQLDATE */
      break;
    case GDA_TypeDbDate:
      return sizeof(v->_u.dbd);	/* struct SQLDATE */
      break;
    case GDA_TypeDbTime:
      return sizeof(v->_u.dbt);	/* struct SQLTIME */
      break;
    case GDA_TypeDbTimestamp:
      return sizeof(v->_u.dbtstamp); /* struct SQLTIMESTAMP */
      break;
    case GDA_TypeDecimal:	
      return v->_u.dec.value._length;		/* unsigned char* */
      break;
    case GDA_TypeDouble:
      return sizeof(v->_u.dp);		/* double */
    case GDA_TypeError:
      return SQL_NTS;	/* unsigned char* */
      break;
    case GDA_TypeInteger:
      return sizeof(v->_u.i);		/* CORBA_long */
    case GDA_TypeLongvarbin:
      return v->_u.lvb._length; /* unsigned char* */
    case GDA_TypeLongvarchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeLongvarwchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeNumeric:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeSingle:
      return sizeof(v->_u.f);	/* float */
    case GDA_TypeSmallint:
      return sizeof(v->_u.si);	/* short */
    case GDA_TypeTinyint:
      return sizeof(v->_u.c);	/* unsigned char */
    case GDA_TypeUBigint:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeUSmallint:
      return sizeof(v->_u.us);	/* unsigned short */
    case GDA_TypeVarchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeVarbin:
      return v->_u.lvb._length;	/* unsigned char* */
    case GDA_TypeVarwchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeFixchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeFixbin:
      return v->_u.lvb._length; /* unsigned char* */
    case GDA_TypeFixwchar:
      return SQL_NTS;		/* unsigned char* */
    }
  g_warning("value_2_inputsize: reached end of function, no matching GDA type found\n");
  abort();
  return 0;
}
      

Gda_ODBC_Recordset*
gda_odbc_init_recset_for_hstmt(Gda_ODBC_Recordset* rec, Gda_ODBC_Error* e)
{
  gint   rc;
  gshort ncols;
  gint   i;
  gchar  colname[128];
  gshort colnamelen;
  
  rec->at_end = 0;
  rec->at_begin = 1;
  rec->pos = 0;

  fprintf(stderr,"gda_init_recset_for_hstmt: entered, recset = 0x%p\n", rec);
  rc = SQLNumResultCols(rec->hstmt, &ncols);
  fprintf(stderr,"gda_init_recset_for_hstmt: number of columns: rc = %d, ncols = %d\n", rc, ncols);
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(e, rec, rec->cmd->cnc , "SQLNumResultCols");
      g_log("GDA ODBC", G_LOG_LEVEL_DEBUG, "SQLNumResultCols error\n");
      return 0;
    }
  if (ncols == 0)
    return 0;

  for (i = 0; i < ncols; i++)
    {
      Gda_ODBC_Field* f;

      f = gda_odbc_field_new();

      rc = SQLDescribeCol(rec->hstmt,
			  i+1,
			  colname, sizeof(colname), &colnamelen,
			  &f->ll_type,
			  &f->defined_length,
			  &f->num_scale,
			  &f->nullable);
      if (rc == SQL_ERROR)
	{
	  g_log("GDA ODBC", G_LOG_LEVEL_DEBUG, "SQLDescribeCol error\n");
	  gda_odbc_error_make(e, rec, rec->cmd->cnc, "SQLDescribeCol");
	  return 0;
	}
      colname[colnamelen] = '\0';
      f->name = g_strdup(colname);
      fprintf(stderr,"gda_init_recset_for_hstmt: describing field '%s'\n", f->name);
      rec->fields = g_list_append(rec->fields, f);
    }
  return rec;
}





Gda_ODBC_Recordset*
gda_odbc_cmd_execute(Gda_ODBC_Command* cmd, Gda_ODBC_Error* error, const GDA_CmdParameterSeq* params,
		     gulong options)
{
  gshort                 nparams;
  gint                   rc;
  int                    i;
  static long            sizes[1000];
  Gda_ODBC_Recordset*    recset;
  GDA_CmdParameter*      current_parameter;

  recset = gda_odbc_recset_new();
  fprintf(stderr,"gda_odbc_cmd_execute: enter\n");

  rc = SQLAllocStmt(cmd->cnc->hdbc, &recset->hstmt);

  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(error, recset, cmd->cnc, "SQLAllocStmt");
      gda_odbc_recset_free(recset);
      return 0;
    }
#if 0
  rc = SQLSetScrollOptions(recset->hstmt, SQL_CONCUR_ROWVER,  SQL_SCROLL_KEYSET_DRIVEN, 20);
  if (rc != SQL_SUCCESS)
    {
      gda_odbc_error_make(error, recset, cmd->cnc, "SQLPrepare");
      gda_odbc_recset_free(recset);
      return 0;
    }
#endif
  rc = SQLPrepare(recset->hstmt, cmd->cmd, SQL_NTS);
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(error, recset, cmd->cnc, "SQLPrepare");
      gda_odbc_recset_free(recset);
      return 0;
    }
  rc = SQLNumParams(recset->hstmt, &nparams);
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(error, recset, cmd->cnc, "SQLNumParams");
      gda_odbc_recset_free(recset);
      return 0;
    }
  if (params && ((nparams && params->_length == 0) || (!nparams && params->_length)))
    {
      if (nparams)
	{
	  g_log("GDA ODBC", G_LOG_LEVEL_ERROR,"Statement needs parameters, but none passed\n");
	}
      else
	{
	  g_log("GDA ODBC", G_LOG_LEVEL_ERROR, "Statement doesn't need parameters, but params availble\n");
	}
    }
  fprintf(stderr,"Number of parameters from parser: %d\n", nparams);
  fprintf(stderr,"Number od parameters from CORBA:  %d\n", params->_length);
  if (nparams)
    {
      
      for (i = 0; i < nparams; i++)
	{
	  gushort sqlType;
	  gulong precision;
	  gshort scale;
	  gshort nullable;
	  unsigned long ctype;

	  current_parameter = &(params->_buffer[i]);
	  
	  if (current_parameter->value._d)
	    {
	      g_warning ("NULL PARAMETER NYI");
	      abort();
	    }

	  ctype = gda_odbc_gda2sqlctype(current_parameter->value._u.v._d);
	  
	  rc = SQLDescribeParam(recset->hstmt,
				i + 1,
				&sqlType,
				&precision,
				&scale,
				&nullable);
	  if (rc != SQL_SUCCESS)
	    {
	      gda_odbc_error_make(error, recset, cmd->cnc, "SQLNumParams");
	      gda_odbc_recset_free(recset);
	      return 0;
	    }
	  sizes[i] = value_2_inputsize(&current_parameter->value._u.v);
	  fprintf(stderr,"gda_odbc_cmd_execute: before: sizes[%d] = %ld\n", i,
		  sizes[i]);
	  fprintf(stderr,"Calling SQLBindParameter for param %d\n", i+1);
	  fprintf(stderr,"                            ctype = %ld\n", ctype);
	  fprintf(stderr,"                          sqltype = %d\n", sqlType);
	  
	  rc = SQLBindParameter(recset->hstmt,
				i + 1,
				SQL_PARAM_INPUT,
				ctype,
				sqlType,
				precision,
				scale,
				value_2_ptr(&current_parameter->value._u.v),
				0,
				&sizes[i]
				);
	  
	  if (rc != SQL_SUCCESS)
	    {
	      gda_odbc_error_make(error, recset,cmd->cnc, "SQLNumParams");
	      gda_odbc_recset_free(recset);
	      return 0;
	    }
	}
    }
  rc = SQLExecute(recset->hstmt);
  for (i = 0; i < nparams; i++)
    {
      fprintf(stderr,"gda_odbc_cmd_execute: after sizes[%d] = %ld\n", i,
	      sizes[i]);
    }
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(error, recset, cmd->cnc, "SQLExecute");
      gda_odbc_recset_free(recset);
      return 0;
    }
  recset->cmd = cmd;
  return gda_odbc_init_recset_for_hstmt(recset, error);
}




  
