/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gconf;

import java.util.ArrayList;
import java.util.List;

import org.gnu.glib.Boxed;
import org.gnu.glib.Handle;

/**
 * Represents a dynamically-typed value in GConf.  
 */
public class ConfValue extends Boxed {
	
	/**
	 * Construct a new ConfValue object of the type provided.
	 * @param type
	 */
	public ConfValue(ConfValueType type) {
		super(gconf_value_new(type.getValue()));
	}
	
	public ConfValue(Handle handle) {
		super(handle);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		gconf_value_free(handle);
	}
	
	public String getString() {
		return gconf_value_get_string(handle);
	}
	
	public void setString(String theString) {
		gconf_value_set_string(handle, theString);
	}
	
	public int getInt() {
		return gconf_value_get_int(handle);
	}
	
	public void setInt(int theInt) {
		gconf_value_set_int(handle, theInt);
	}
	
	public double getDouble() {
		return gconf_value_get_float(handle);
	}
	
	public void setDouble(double theDouble) {
		gconf_value_set_float(handle, theDouble);
	}
	
	public boolean getBoolean() {
		return gconf_value_get_bool(handle);
	}
	
	public void setBoolean(boolean theBoolean) {
		gconf_value_set_bool(handle, theBoolean);
	}

	public ConfValueType getListType() {
		return ConfValueType.intern(gconf_value_get_list_type(handle));
	}
	
	public void setListType(ConfValueType type) {
		gconf_value_set_list_type(handle, type.getValue());
	}
	
	public List getList() {
		List list = new ArrayList();
		Object[] objs = gconf_value_get_list(handle);
		for (int i = 0; i < objs.length; i++) {
			list.add(objs[i]);
		}
		return list;
	}
	
	public void setList(List theList) {
		Object[] objs = new Object[theList.size()];
		for (int i = 0; i < theList.size(); i++) {
			objs[i] = theList.get(i);
		}
		gconf_value_set_list(handle, objs);
	}
	
	public ConfValue getCar() {
		return new ConfValue(gconf_value_get_car(handle));
	}
	
	public void setCar(ConfValue theCar) {
		gconf_value_set_car(handle, theCar.getHandle());
	}
	
	public ConfValue getCdr() {
		return new ConfValue(gconf_value_get_cdr(handle));
	}
	
	public void setCdr(ConfValue theCdr) {
		gconf_value_set_cdr(handle, theCdr.getHandle());
	}
	
	public ConfSchema getSchema() {
		return new ConfSchema(gconf_value_get_schema(handle));
	}
	
	public void setSchema(ConfSchema theSchema) {
		gconf_value_set_schema(handle, theSchema.getHandle());
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return gconf_value_to_string(handle);
	}
	
	native static final protected Handle gconf_value_new(int type);
	native static final protected Handle gconf_value_copy(Handle source);
	native static final protected void gconf_value_free(Handle value);
	native static final protected String gconf_value_get_string(Handle value);
	native static final protected int gconf_value_get_int(Handle value);
	native static final protected double gconf_value_get_float(Handle value);
	native static final protected int gconf_value_get_list_type(Handle value);
	native static final protected Object[] gconf_value_get_list(Handle value);
	native static final protected Handle gconf_value_get_car(Handle value);
	native static final protected Handle gconf_value_get_cdr(Handle value);
	native static final protected boolean gconf_value_get_bool(Handle value);
	native static final protected Handle gconf_value_get_schema(Handle value);
	native static final protected void gconf_value_set_int(Handle value, int theInt);
	native static final protected void gconf_value_set_string(Handle value, String theString);
	native static final protected void gconf_value_set_bool(Handle value, boolean theBool);
	native static final protected void gconf_value_set_float(Handle value, double theFloat);
	native static final protected void gconf_value_set_schema(Handle value, Handle theSchema);
	native static final protected void gconf_value_set_car(Handle value, Handle theCar);
	native static final protected void gconf_value_set_cdr(Handle value, Handle theCdr);
	native static final protected void gconf_value_set_list_type(Handle value, int type);
	native static final protected void gconf_value_set_list(Handle value, Object[] theList);
	native static final protected String gconf_value_to_string(Handle value);
}
