#include <stdio.h>
#include <X11/X.h>
#include <Xm/Xm.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/LabelG.h>
#include <Xm/PushBG.h>
#include <Xm/ToggleBG.h>
#include <Xm/ToggleB.h>
#include <Xm/SeparatoG.h>
#include <Xm/SelectioB.h>
#include "../util/misc.h"
#include "XY.h"
#include "XYDialogs.h"

typedef struct {
    Widget form;
    Widget sizeShell;
    Widget lineBtns[XY_N_LINE_STYLES];
    Widget markBtns[XY_N_MARK_STYLES];
    Widget sizeBtns[XY_N_MARK_SIZES];
    Pixmap linePixmaps[XY_N_LINE_STYLES];
    Pixmap markPixmaps[XY_N_MARK_STYLES];
    Pixmap sizePixmaps[XY_N_MARK_SIZES];
    Widget sizeDlogBtns[XY_N_MARK_SIZES];
    Widget *lineOptMenus;
    Widget *markOptMenus;
    Widget *sizeOptMenus;
    Pixmap *samplePixmaps;
    Widget *sampleLabels;
    XYCurve *origCurves;
    XYCurve *curves;
    XYCallbackProc okCB;
    XYCallbackProc applyCB;
    XYCallbackProc cancelCB;
    void *okArg;
    void *applyArg;
    void *cancelArg;
    GC gc;
    int nCurves;
    Pixel fgColor, bgColor;
} markDialog;

enum resultFlags {NO_RESULT, RESULT_OK, RESULT_CANCEL};

static void stylesOKCB(XYCurve *curves, int, void *resultFlag);
static void stylesCancelCB(XYCurve *curves, int, void *resultFlag);
static void update(markDialog *dialog);
static void optMenuCB(Widget w, markDialog *dialog, caddr_t callData);
static void clearLinesCB(Widget w, markDialog *dialog, caddr_t callData);
static void clearMarksCB(Widget w, markDialog *dialog, caddr_t callData);
static void defaultLinesCB(Widget w, markDialog *dialog, caddr_t callData);
static void defaultMarksCB(Widget w, markDialog *dialog, caddr_t callData);
static void okCB(Widget w, markDialog *dialog, caddr_t callData);
static void applyCB(Widget w, markDialog *dialog, caddr_t callData);
static void cancelCB(Widget w, markDialog *dialog, caddr_t callData);
void destroyCB(Widget w, markDialog *dialog, caddr_t callData);
static void copyStyles(XYCurve *fromCurves, XYCurve *toCurves, int nCurves);
static void updateSample(markDialog *dialog, int index);
static void updateTimerProc(XtPointer clientData, XtIntervalId *id);
static void markSizeCB(Widget w, markDialog *dialog, caddr_t callData);
static void createMarkSizeDialog(markDialog *dialog);
static void sizeOKCB(Widget w, markDialog *dialog, caddr_t callData);
static void sizeCancelCB(Widget w, markDialog *dialog, caddr_t callData);

/*
** XYEditStyles
**
** Prompt user for marker and line styles for a set of curves for display
** by the XY widget.  This is a simpler version of XYCreateStylesDialog,
** which presents the same dialog, except with no Apply button.  The routine
** returns when the user has selected a new set of styles for the curves,
** and alters the array curves to reflect the user's changes.
**
** Parameters
**
**	parent		Parent widget for dialog
**	curves		Array of XYCurves data structures containing style
**			information to alter.  XYEditStyles modifies
**			this data to match the style information selected
**			by the user.
**	nCurves		The number of curves data structures in curves above.
**
** Returns True if styles were changed
*/
int XYEditStyles(Widget parent, XYCurve *curves, int nCurves)
{
    int resultFlag = NO_RESULT;
    Widget dialog;
    
    dialog = XYCreateStylesDialog(parent, curves, nCurves, stylesOKCB,
    	    &resultFlag, NULL, NULL, stylesCancelCB, &resultFlag);
    while (resultFlag == 0)
        XtAppProcessEvent(XtWidgetToApplicationContext(dialog), XtIMAll);
    
    return resultFlag == RESULT_OK;
}

static void stylesOKCB(XYCurve *curves, int nCurves, void *resultFlag)
{
    *(int *)resultFlag = RESULT_OK;
}
static void stylesCancelCB(XYCurve *curves, int nCurves, void *resultFlag)
{
    *(int *)resultFlag = RESULT_CANCEL;
}

/*
** XYCreateStylesDialog
**
** Display a dialog for setting marker and line styles for a set of curves
** for display by the XY widget.  Returns immediately but leaves up the
** dialog with callbacks attached to the ok, apply, and cancel buttons.
** When the user presses the OK or Apply buttons, the original set of curves
** structures passed to XYCreateStylesDialog is altered and passed back to
** the apply or ok callback routine.
** 
** The dialog may be popped down by the caller by destroying the widget
** returned by this routine.  Since the ok and cancel buttons automatically
** handle popping down the dialog, this is usually not necessary.
**
** Parameters
**
**	parent		Parent widget for dialog
**	curves		Array of XYCurves data structures containing style
**			information to alter.  XYCreateStylesDialog modifies
**			this data when the user makes his selections and
**			presses ok or apply, therefore, this must data must
**			be preserved after the call to this routine, until
**			the styles dialog is popped down.
**	nCurves		The number of curves data structures in curves above.
**	okCallback	Routine to call when the user presses OK in the
**			dialog.  Signals that the dialog has been removed
**			and set of curves has been altered with the new
**			style information.
**	okArg		Arbitrary data to pass to the ok callback routine.
**	applyCallback	Routine to call when the user presses Apply in the
**			styles dialog.  Signals that the data in the curves
**			array has been altered, but the dialog is still up.
**			If this is passed as NULL, no apply button is
**			displayed.
**	applyArg	Arbitrary data to pass to the ok callback routine.
**	cancelCallback	Routine to call when the user presses Cancel in the
**			styles dialog.  Signals that the dialog has been
**			popped down without altering the curve structures.
**	cancelArg	Arbitrary data to pass to the ok callback routine.
*/
Widget XYCreateStylesDialog(Widget parent, XYCurve *curves, int nCurves,
	XYCallbackProc okCallback, void *okArg, XYCallbackProc applyCallback,
	void *applyArg, XYCallbackProc cancelCallback, void *cancelArg)
{
    Widget form, lineMenu, markMenu, sizeMenu, clearMarks, clearLines, sep;
    Widget defaultMarks, defaultLines, markSize, ok, apply, cancel;
    Widget lineOptMenu, markOptMenu, sizeOptMenu, sampleLabel, name, topWidget;
    XtArgVal topAttach;
    int i, n;
    Arg args[12];
    markDialog *dialog;
    Display *display = XtDisplay(parent);
    int depth = DefaultDepthOfScreen(XtScreen(parent));
    int hasApply = applyCallback != NULL;
    Pixel fgColor, bgColor;
    GC drawGC;
    Pixmap pm;
    XmString s1;
    
    dialog = (markDialog *)XtMalloc(sizeof(markDialog));
    dialog->origCurves = curves;
    dialog->curves = (XYCurve *)XtMalloc(sizeof(XYCurve)*nCurves);
    memcpy(dialog->curves, curves, sizeof(XYCurve)*nCurves);
    dialog->nCurves = nCurves;
    dialog->lineOptMenus = (Widget *)XtMalloc(sizeof(Widget)*nCurves);
    dialog->markOptMenus = (Widget *)XtMalloc(sizeof(Widget)*nCurves);
    dialog->sizeOptMenus = (Widget *)XtMalloc(sizeof(Widget)*nCurves);
    dialog->samplePixmaps = (Pixmap *)XtMalloc(sizeof(Pixmap)*nCurves);
    dialog->sampleLabels = (Widget *)XtMalloc(sizeof(Widget)*nCurves);
    dialog->gc = drawGC = XCreateGC(display, XtWindow(parent), 0, NULL);
    XtVaGetValues(parent, XmNforeground, &fgColor, XmNbackground, &bgColor, 0);
    dialog->okCB = okCallback;
    dialog->applyCB = applyCallback;
    dialog->cancelCB = cancelCallback;
    dialog->okArg = okArg;
    dialog->applyArg = applyArg;
    dialog->cancelArg = cancelArg;
    dialog->fgColor = fgColor;
    dialog->bgColor = bgColor;
    
    n = 0;
    XtSetArg(args[n], XmNtitle, "Mark and Line Styles"); n++;
    XtSetArg(args[n], XmNautoUnmanage, False); n++;
    form = XmCreateFormDialog(parent, "XYMarkLineDialog", args, n);
    XtVaSetValues(form, XmNshadowThickness, 0, 0);
    dialog->form = form;
    XtAddCallback(XtParent(form), XmNdestroyCallback,
    	    (XtCallbackProc)destroyCB, dialog);
        
    lineMenu = XmCreatePulldownMenu(form, "lineMenu", NULL, 0);
    for (i=0; i<XY_N_LINE_STYLES; i++) {
        pm = XCreatePixmap(display, XtWindow(parent), 50, 12, depth);
        dialog->linePixmaps[i] = pm;
        XSetForeground(display, drawGC, bgColor);
        XFillRectangle(display, pm, drawGC, 0, 0, 50, 12);
        XYDrawLine(display, pm, drawGC, i, fgColor, 5, 6, 45, 6);
        dialog->lineBtns[i] = XtVaCreateManagedWidget("lineStyle",
        	xmPushButtonGadgetClass, lineMenu,
    		XmNlabelType, XmPIXMAP,
    		XmNlabelPixmap, pm, 0);
    }
    XtAddCallback(lineMenu, XmNentryCallback, (XtCallbackProc)optMenuCB,
    	    dialog);
    
    markMenu = XmCreatePulldownMenu(form, "markMenu", NULL, 0);
    for (i=0; i<XY_N_MARK_STYLES; i++) {
        pm = XCreatePixmap(display, XtWindow(parent), 12, 12, depth);
        dialog->markPixmaps[i] = pm;
        XSetForeground(display, drawGC, bgColor);
        XFillRectangle(display, pm, drawGC, 0, 0, 12, 12);
        XYDrawMarker(display, pm, drawGC, XY_LARGE, i, fgColor, 6, 6);
        dialog->markBtns[i] = XtVaCreateManagedWidget("markStyle",
        	xmPushButtonGadgetClass, markMenu,
    		XmNlabelType, XmPIXMAP,
    		XmNlabelPixmap, pm, 0);
    }
    XtAddCallback(markMenu, XmNentryCallback, (XtCallbackProc)optMenuCB,
    	    dialog);
    
    sizeMenu = XmCreatePulldownMenu(form, "sizeMenu", NULL, 0);
    for (i=0; i<XY_N_MARK_SIZES; i++) {
        pm = XCreatePixmap(display, XtWindow(parent), 12, 12, depth);
        dialog->sizePixmaps[i] = pm;
        XSetForeground(display, drawGC, bgColor);
        XFillRectangle(display, pm, drawGC, 0, 0, 12, 12);
        XYDrawMarker(display, pm, drawGC, i, XY_CIRCLE_MARK, fgColor, 6, 6);
        dialog->sizeBtns[i] = XtVaCreateManagedWidget("markSize",
        	xmPushButtonGadgetClass, sizeMenu,
    		XmNlabelType, XmPIXMAP,
    		XmNlabelPixmap, pm, 0);
    }
    XtAddCallback(sizeMenu, XmNentryCallback, (XtCallbackProc)optMenuCB,
    	    dialog);
    
    topWidget = NULL;
    topAttach = XmATTACH_FORM;
    for (i=0; i<nCurves; i++) {

	n = 0;
	XtSetArg(args[n], XmNspacing, 0); n++;
	XtSetArg(args[n], XmNmarginWidth, 0); n++;
	XtSetArg(args[n], XmNtopAttachment, topAttach); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg(args[n], XmNtopWidget, topWidget); n++;
	XtSetArg(args[n], XmNsubMenuId, lineMenu); n++;
	XtSetArg(args[n], XmNmenuHistory,
		dialog->lineBtns[curves[i].lineStyle]); n++;
	lineOptMenu = XmCreateOptionMenu(form, "lineOptMenu", args, n);
	XtManageChild(lineOptMenu);
	dialog->lineOptMenus[i] = lineOptMenu;
	XtVaSetValues(XmOptionButtonGadget(lineOptMenu), XmNmarginHeight, 0, 0);
	
	n = 0;
	XtSetArg(args[n], XmNspacing, 0); n++;
	XtSetArg(args[n], XmNmarginWidth, 0); n++;
	XtSetArg(args[n], XmNtopAttachment, topAttach); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopWidget, topWidget); n++;
	XtSetArg(args[n], XmNleftWidget, lineOptMenu); n++;
	XtSetArg(args[n], XmNsubMenuId, markMenu); n++;
	XtSetArg(args[n], XmNmenuHistory,
		dialog->markBtns[curves[i].markerStyle]); n++;
	markOptMenu = XmCreateOptionMenu(form, "markOptMenu", args, n);
	dialog->markOptMenus[i] = markOptMenu;
	XtManageChild(markOptMenu);
	XtVaSetValues(XmOptionButtonGadget(markOptMenu), XmNmarginHeight, 0, 0);
	
	n = 0;
	XtSetArg(args[n], XmNspacing, 0); n++;
	XtSetArg(args[n], XmNmarginWidth, 0); n++;
	XtSetArg(args[n], XmNtopAttachment, topAttach); n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg(args[n], XmNtopWidget, topWidget); n++;
	XtSetArg(args[n], XmNleftWidget, markOptMenu); n++;
	XtSetArg(args[n], XmNsubMenuId, sizeMenu); n++;
	XtSetArg(args[n], XmNmenuHistory,
		dialog->sizeBtns[curves[i].markerSize]); n++;
	sizeOptMenu = XmCreateOptionMenu(form, "sizeOptMenu", args, n);
	dialog->sizeOptMenus[i] = sizeOptMenu;
	XtManageChild(sizeOptMenu);
	XtVaSetValues(XmOptionButtonGadget(sizeOptMenu), XmNmarginHeight, 0, 0);

        pm = XCreatePixmap(display, XtWindow(parent), 52, 12, depth);
        dialog->samplePixmaps[i] = pm;
        sampleLabel = XtVaCreateManagedWidget("markStyle",
        	xmLabelGadgetClass, form,
    		XmNlabelType, XmPIXMAP,
    		XmNlabelPixmap, pm,
    		XmNtopAttachment, topAttach,
    		XmNtopWidget, topWidget,
    		XmNleftAttachment, XmATTACH_WIDGET,
    		XmNleftWidget, sizeOptMenu,
    		XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
    		XmNbottomWidget, markOptMenu, 0);
	dialog->sampleLabels[i] = sampleLabel;
        updateSample(dialog, i);
	
	name = XtVaCreateManagedWidget("name", xmLabelGadgetClass, form,
    		XmNlabelString, curves[i].name,
    		XmNalignment, XmALIGNMENT_BEGINNING,
    		XmNtopAttachment, topAttach,
    		XmNtopWidget, topWidget,
    		XmNleftAttachment, XmATTACH_WIDGET,
    		XmNleftWidget, sampleLabel,
    		XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
    		XmNbottomWidget, markOptMenu,
    		XmNrightAttachment, XmATTACH_POSITION,
    		XmNrightPosition, 99, 0);
    	
    	topAttach = XmATTACH_WIDGET;
    	topWidget = lineOptMenu;
    }
    
    clearMarks = XtVaCreateManagedWidget("clearMarks",
    	    xmPushButtonGadgetClass, form,
    	    XmNlabelString, s1=XmStringCreateSimple("Clear Marks"),
    	    XmNtopAttachment, topAttach,
    	    XmNtopWidget, topWidget,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, 1,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, 33,
    	    XmNtopOffset, 6, 0);
    XmStringFree(s1);
    XtAddCallback(clearMarks, XmNactivateCallback,
    	    (XtCallbackProc)clearMarksCB, dialog);
    defaultMarks = XtVaCreateManagedWidget("defaultMarks",
    	    xmPushButtonGadgetClass, form,
    	    XmNlabelString, s1=XmStringCreateSimple("Default Marks"),
    	    XmNtopAttachment, topAttach,
    	    XmNtopWidget, topWidget,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, 34,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, 66,
    	    XmNtopOffset, 6, 0);
    XmStringFree(s1);
    XtAddCallback(defaultMarks, XmNactivateCallback,
    	    (XtCallbackProc)defaultMarksCB, dialog);
    markSize = XtVaCreateManagedWidget("markSize", xmPushButtonGadgetClass,form,
    	    XmNlabelString, s1=XmStringCreateSimple("Mark Size"),
    	    XmNtopAttachment, topAttach,
    	    XmNtopWidget, topWidget,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, 67,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, 99,
    	    XmNtopOffset, 6, 0);
    XmStringFree(s1);
    XtAddCallback(markSize, XmNactivateCallback, (XtCallbackProc)markSizeCB,
    	    dialog);
    clearLines = XtVaCreateManagedWidget("clearLines",
    	    xmPushButtonGadgetClass, form,
    	    XmNlabelString, s1=XmStringCreateSimple("Clear Lines"),
    	    XmNdefaultButtonShadowThickness, 0,
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, clearMarks,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, 1,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, 33, 0);
    XmStringFree(s1);
    XtAddCallback(clearLines, XmNactivateCallback, (XtCallbackProc)clearLinesCB,
    	    dialog);
    
    defaultLines = XtVaCreateManagedWidget("defaultLines",
    	    xmPushButtonGadgetClass, form,
    	    XmNlabelString, s1=XmStringCreateSimple("Default Lines"),
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, clearMarks,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, 34,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, 66, 0);
    XmStringFree(s1);
    XtAddCallback(defaultLines, XmNactivateCallback,
    	    (XtCallbackProc)defaultLinesCB, dialog);
    
    sep = XtVaCreateManagedWidget("sep", xmSeparatorGadgetClass, form,
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, clearLines,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, 0,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, 100,
    	    XmNtopOffset, 9, 0);
    
    ok = XtVaCreateManagedWidget("ok", xmPushButtonGadgetClass, form,
    	    XmNlabelString, s1=XmStringCreateSimple("OK"),
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, sep,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, hasApply ? 4 : 20,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, hasApply ? 30 : 40,
    	    XmNtopOffset, 5, 0);
    XmStringFree(s1);
    XtAddCallback(ok, XmNactivateCallback, (XtCallbackProc)okCB, dialog);
    XtVaSetValues(form, XmNdefaultButton, ok, 0);
    if (hasApply) {
	apply = XtVaCreateManagedWidget("apply", xmPushButtonGadgetClass, form,
    		XmNlabelString, s1=XmStringCreateSimple("Apply"),
    		XmNtopAttachment, XmATTACH_WIDGET,
    		XmNtopWidget, sep,
    		XmNleftAttachment, XmATTACH_POSITION,
    		XmNleftPosition, 37,
    		XmNrightAttachment, XmATTACH_POSITION,
    		XmNrightPosition, 63,
    		XmNtopOffset, 5, 0);
	XmStringFree(s1);
	XtAddCallback(apply, XmNactivateCallback, (XtCallbackProc)applyCB,
		dialog);
    }
    cancel = XtVaCreateManagedWidget("cancel", xmPushButtonGadgetClass, form,
    	    XmNlabelString, s1=XmStringCreateSimple("Cancel"),
    	    XmNtopAttachment, XmATTACH_WIDGET,
    	    XmNtopWidget, sep,
    	    XmNleftAttachment, XmATTACH_POSITION,
    	    XmNleftPosition, hasApply ? 70 : 60,
    	    XmNrightAttachment, XmATTACH_POSITION,
    	    XmNrightPosition, hasApply ? 96 : 80,
    	    XmNtopOffset, 5, 0);
    XmStringFree(s1);
    XtAddCallback(cancel, XmNactivateCallback, (XtCallbackProc)cancelCB,
    	    dialog);
    
    /* stop the specialty buttons from showing the default button shadow */
    XtVaSetValues(clearMarks, XmNdefaultButtonShadowThickness, 0, 0);
    XtVaSetValues(clearLines, XmNdefaultButtonShadowThickness, 0, 0);
    XtVaSetValues(defaultMarks, XmNdefaultButtonShadowThickness, 0, 0);
    XtVaSetValues(defaultLines, XmNdefaultButtonShadowThickness, 0, 0);
    XtVaSetValues(markSize, XmNdefaultButtonShadowThickness, 0, 0);
    
    XtManageChild(form);
    return XtParent(form);
}

static void updateSample(markDialog *dialog, int index)
{
    XYCurve *curve = &dialog->curves[index];
    Pixmap pm = dialog->samplePixmaps[index];
    Display *display = XtDisplay(dialog->form);
    GC gc = dialog->gc;
    
    XSetForeground(display, gc, dialog->bgColor);
    XFillRectangle(display, pm, gc, 0, 0, 52, 12);
    XYDrawMarker(display, pm, gc, curve->markerSize,
            curve->markerStyle, curve->markerPixel, 46, 6);
    if (curve->lineStyle != XY_NO_LINE) {
	XYDrawMarker(display, pm, gc, curve->markerSize,
        	curve->markerStyle, curve->markerPixel, 6, 6);
        XYDrawLine(display, pm, gc, curve->lineStyle,
            	curve->linePixel, 6, 6, 46, 6);
    }
    XtVaSetValues(dialog->sampleLabels[index], XmNlabelPixmap, 
    	    dialog->linePixmaps[0], 0);
    XtVaSetValues(dialog->sampleLabels[index], XmNlabelPixmap, pm, 0);
}

static void update(markDialog *dialog)
{
    int i, j;
    XYCurve *curve;
    Widget *w, selectedW;
    
    for (i=0, curve=dialog->curves; i<dialog->nCurves; i++, curve++) {
        XtVaGetValues(dialog->lineOptMenus[i], XmNmenuHistory, &selectedW, 0);
        for (j=0, w=dialog->lineBtns; j<XY_N_LINE_STYLES; j++, w++) {
            if (*w == selectedW && j != curve->lineStyle) {
        	curve->lineStyle = j;
        	updateSample(dialog, i);
        	break;
            }
        }
        XtVaGetValues(dialog->markOptMenus[i], XmNmenuHistory, &selectedW, 0);
        for (j=0, w=dialog->markBtns; j<XY_N_MARK_STYLES; j++, w++) {
            if (*w == selectedW && j != curve->markerStyle) {
        	curve->markerStyle = j;
        	updateSample(dialog, i);
        	break;
            }
        }
        XtVaGetValues(dialog->sizeOptMenus[i], XmNmenuHistory, &selectedW, 0);
        for (j=0, w=dialog->sizeBtns; j<XY_N_MARK_SIZES; j++, w++) {
            if (*w == selectedW && j != curve->markerSize) {
        	curve->markerSize = j;
        	updateSample(dialog, i);
        	break;
            }
        }
    }
}

static void optMenuCB(Widget w, markDialog *dialog, caddr_t callData)
{
    XtAppAddTimeOut(XtWidgetToApplicationContext(dialog->form),
    	    0, updateTimerProc, dialog);
}

static void clearLinesCB(Widget w, markDialog *dialog, caddr_t callData)
{
    int i;
    
    for (i=0; i<dialog->nCurves; i++) {
	XtVaSetValues(dialog->lineOptMenus[i], XmNmenuHistory,
		dialog->lineBtns[0], 0);
	update(dialog);
    }
}

static void clearMarksCB(Widget w, markDialog *dialog, caddr_t callData)
{
    int i;
    
    for (i=0; i<dialog->nCurves; i++) {
	XtVaSetValues(dialog->markOptMenus[i], XmNmenuHistory,
		dialog->markBtns[0], 0);
	update(dialog);
    }
}

static void defaultLinesCB(Widget w, markDialog *dialog, caddr_t callData)
{
    int i;
    
    for (i=0; i<dialog->nCurves; i++)
	XtVaSetValues(dialog->lineOptMenus[i], XmNmenuHistory,
		dialog->lineBtns[(i % XY_4_DOT_DASH) + 1], 0);
    update(dialog);
}

static void defaultMarksCB(Widget w, markDialog *dialog, caddr_t callData)
{
    int i;
    
    for (i=0; i<dialog->nCurves; i++)
	XtVaSetValues(dialog->markOptMenus[i], XmNmenuHistory,
		dialog->markBtns[(i % XY_SOLID_CIRCLE_MARK) + 1], 0);
    update(dialog);
}

static void markSizeCB(Widget w, markDialog *dialog, caddr_t callData)
{
    createMarkSizeDialog(dialog);
}

static void okCB(Widget w, markDialog *dialog, caddr_t callData)
{
    copyStyles(dialog->curves, dialog->origCurves, dialog->nCurves);
    if (dialog->okCB != NULL)
    	(*dialog->okCB)(dialog->origCurves, dialog->nCurves, dialog->okArg);
    XtDestroyWidget(XtParent(dialog->form));
}

static void applyCB(Widget w, markDialog *dialog, caddr_t callData)
{
    copyStyles(dialog->curves, dialog->origCurves, dialog->nCurves);
    if (dialog->applyCB != NULL)
    	(*dialog->applyCB)(dialog->origCurves, dialog->nCurves,
    		dialog->applyArg);
}

static void cancelCB(Widget w, markDialog *dialog, caddr_t callData)
{
    if (dialog->cancelCB != NULL)
    	(*dialog->cancelCB)(dialog->origCurves, dialog->nCurves,
    		dialog->cancelArg);
    XtDestroyWidget(XtParent(dialog->form));
}

void destroyCB(Widget w, markDialog *dialog, caddr_t callData)
{
    int i;
    Display *display = XtDisplay(dialog->form);
    
    /* Free all resources used by the dialog */
    for (i=0; i<XY_N_MARK_STYLES; i++)
    	XFreePixmap(display, dialog->markPixmaps[i]);
    for (i=0; i<XY_N_LINE_STYLES; i++)
    	XFreePixmap(display, dialog->linePixmaps[i]);
    for (i=0; i<XY_N_MARK_SIZES; i++)
    	XFreePixmap(display, dialog->sizePixmaps[i]);
    XtFree((char *)dialog->curves);
    XtFree((char *)dialog->lineOptMenus);
    XtFree((char *)dialog->markOptMenus);
    XtFree((char *)dialog->sizeOptMenus);
    XtFree((char *)dialog->samplePixmaps);
    XtFree((char *)dialog->sampleLabels);;
    XFreeGC(display, dialog->gc);
    XtFree((char *)dialog);
}

static void copyStyles(XYCurve *fromCurves, XYCurve *toCurves, int nCurves)
{
    int i;
    XYCurve *from, *to;
    
    for(i=0, from=fromCurves, to=toCurves; i<nCurves; i++, from++, to++) {
	to->markerStyle = from->markerStyle;
	to->markerSize = from->markerSize;
	to->lineStyle = from->lineStyle;
	to->markerPixel = from->markerPixel;
	to->linePixel = from->linePixel;
	to->nPoints = from->nPoints;
    }
}

static void updateTimerProc(XtPointer clientData, XtIntervalId *id)
{
    update((markDialog *)clientData);
}

static void createMarkSizeDialog(markDialog *dialog)
{
    Widget selBox, checkBox;
    XmString s1;
    int i, n, avgSize, sum = 0;
    Arg args[5];

    /* find the average size of markers for setting default */
    for (i=0; i<dialog->nCurves; i++)
    	sum += dialog->curves[i].markerSize;
    avgSize = sum / dialog->nCurves;
    
    n = 0;
    XtSetArg(args[n], XmNselectionLabelString,
    	    s1=XmStringCreateSimple("Set all markers to size:")); n++;
#if XmVersion >= 1002
    XtSetArg(args[n], XmNchildPlacement, XmPLACE_BELOW_SELECTION); n++;
#endif
    selBox = XmCreatePromptDialog(dialog->form, "markSizeDialog", args, n);
    XmStringFree(s1);
    dialog->sizeShell = XtParent(selBox);
    XtAddCallback(selBox, XmNokCallback, (XtCallbackProc)sizeOKCB, dialog);
    XtAddCallback(selBox, XmNcancelCallback, (XtCallbackProc)sizeCancelCB,
    	    dialog);
    XtUnmanageChild(XmSelectionBoxGetChild(selBox, XmDIALOG_TEXT));
    XtUnmanageChild(XmSelectionBoxGetChild(selBox, XmDIALOG_HELP_BUTTON));
    XtVaSetValues(XtParent(selBox), XmNtitle, "Marker Size", 0);
    AddMotifCloseCallback(XtParent(selBox), (XtCallbackProc)sizeCancelCB,
 	    dialog);

    /* create the size checkbox */
    n = 0;
    XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
    checkBox = XmCreateRadioBox(selBox, "checkBox", args, n);
    XtManageChild(checkBox);
    for (i=0; i<XY_N_MARK_SIZES; i++) {
    	dialog->sizeDlogBtns[i] = XtVaCreateManagedWidget("size",
    		xmToggleButtonGadgetClass, checkBox, XmNlabelType, XmPIXMAP,
    		XmNlabelPixmap, dialog->sizePixmaps[i], XmNset, i==avgSize, 0);
    }
#if XmVersion >= 1002
    XtVaSetValues(checkBox, XmNinitialFocus, dialog->sizeDlogBtns[avgSize], 0);
#endif
    
    XtManageChild(selBox);
}

static void sizeOKCB(Widget w, markDialog *dialog, caddr_t callData)
{
    int i, index;
    
    /* Find out which size was selected */
    for (i=0; i<XY_N_MARK_SIZES; i++)
    	if (XmToggleButtonGetState(dialog->sizeDlogBtns[i]))
    	    index = i;
    
    /* Set all of the option menus to show that size */
    for (i=0; i<dialog->nCurves; i++)
	XtVaSetValues(dialog->sizeOptMenus[i], XmNmenuHistory,
		dialog->sizeBtns[index], 0);
    update(dialog);
    
    /* Get rid of the dialog */
    XtDestroyWidget(dialog->sizeShell);
}

static void sizeCancelCB(Widget w, markDialog *dialog, caddr_t callData)
{
    XtDestroyWidget(dialog->sizeShell);
}
