#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * Everything dealing with encoding data
 *
 * Deals with serialization and network encoding
 * of primative data types (char, short, int, long,
 * string, buffer).  Sorry, no float yet (unless
 * there is a htonf routine somewhere).
 *
 * Is little/big endian clean, but requires
 * 4 byte ints/longs. All the world's a Sun??
 */

#include "sos.h"

/* Pack a variable of type type into buf.  Worry about non-word aligned buf.  Update buf ptr */
#define PACK(type, buf, val) do { type foo = val; memcpy(buf,(char *)&foo,sizeof(type)); buf+=sizeof(type); } while(0)

/* Pack a variable of type type into buf.  Store pval result to conversion may reference it numerous times. */
#define CPACK(type, buf, pval, conv) do { type tmp = pval; tmp=conv(tmp); memcpy(buf,(char *)&tmp,sizeof(type)); buf+=sizeof(type); } while(0)

/* Copy a variable from src to dest.  Update dest ptr */
#define MPACK(dest, src, len) do { memcpy(dest, src, len); dest += len; } while (0)

/* Unpack a variable from buf into a variable of type type while doing a conv conversion. */
#define UNPACK(type, buf, val, conv) do { memcpy((char *)&val, buf, sizeof(type)); val = conv(val); buf+=sizeof(type); } while(0)

/* Unpack a variable from buf into a dereferenced variable of type type while doing conversion. */
#define PUNPACK(type, buf, pval, conv) do { type *packptr = pval; memcpy((char *)packptr, buf, sizeof(type)); *packptr = conv(*packptr); buf+=sizeof(type); } while(0)

/* Set src pointer to point into dest.  Update src ptr */
#define MUNPACK(dest, src, len) do { char **packptr = dest; *packptr = src; src += len; } while (0)

/* Don't even ask why we do this. You're better off not knowing. */
/* BUT DON'T MESS WITH IT UNLESS YOU REALLY KNOW WHAT YOU SHOULDN'T :-) */
#define SOS_XDR_HTONQ SOS_HTONQ
#define SOS_XDR_NTOHQ SOS_NTOHQ

/* 
 * Debug "levels"
 * up to 20 -- Various specifics
 * 2 sos_string length computation
 * 4 sos_buffer unpacking
 * 
 * 64 -- General info
 * 128 -- Useless verbosity :-)
 * (use 192 for both types of messages. If you are setting 128 you probably
 * mean 192).
 */



/* Buffer containing information to/from wire */
static sos_string staticspace;

/* Private functions */
static sos_string *sos_xdr_sencode_int(char *fmt, va_list args);
static int sos_xdr_sdecode_int(char *buf, char *fmt, va_list args);
static int sos_xdr_compute_arg_number_length (char *fmt, va_list args, int *argcount, int *length );
static int sos_xdr_packbuf(char *fmt, va_list saveargs, char *curloc);
static int sos_xdr_unpackbuf(char *fmt, va_list args,  char **curloc, int *pargfound);
static int sos_xdr_compare_fmt_buf(char *fmt, char *curloc, int lenleft);
static sos_string *sos_xdr_sappend_int(sos_string *buf, char *fmt, va_list args);
static sos_string *sos_xdr_sprepend_int(sos_string *buf, char *fmt, va_list args);
static int sos_xdr_speek_int(sos_string *buf, char **pcurloc, char *fmt, va_list args);
static void sos_xdr_print_quad(sos_quad_t num);



/*
 * Encode data represented by fmt into a buffer
 * which is returned.  The buffer will be freed
 * if you call sos_xdr_sencode, sos_xdr_wencode,
 * sos_xdr_wdecode, or sos_xdr_freebuf
 */
sos_string *
sos_xdr_sencode(char *fmt, ...)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_sencode", NULL);
  va_list args;
  sos_string *ret;

  va_start(args,fmt);
  ret = sos_xdr_sencode_int(fmt, args);
  va_end(args);

  SOS_RETURN(ret);
}



/*
 * Internal varargs version--string encode
 */
static sos_string *
sos_xdr_sencode_int(char *fmt, va_list args)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_sencode_int", "Format is: %s", fmt); 
  char *curloc;
  int length = 0;
  int err = 0;
  char *ptr;
  int argcount = 0;
  int tmplength=0;
  va_list saveargs;

  if ( !fmt || !args )
    {
      sos_error_printf("Illegal Arguments\n");
      SOS_RETURN(NULL);
    }

  sos_xdr_freebuf();

  length += 4;			/* Number of arguments */

  saveargs = args;

  err=sos_xdr_compute_arg_number_length(fmt, args, &argcount, &tmplength);

  if (err)
    {
      sos_error_printf("Invalid format: %s\n",fmt);
      SOS_RETURN(NULL);
    }

  length += tmplength;


  sos_debug_printf_and(64,"sos_xdr_sencode: Arcount: %d -- length: %d\n",
		      argcount, length);

  if ((staticspace.str = (char *)malloc(length)) == NULL)
    {
      sos_error_printf("Mallocing new string for staticspace: %s\n",strerror(errno));
      SOS_RETURN(NULL);
    }

  sos_debug_printf_and(128,"sos_xdr_sencode(): Mallocing %d bytes\n",length);

  curloc = staticspace.str;
  staticspace.len = length;

  PACK(int, curloc, SOS_HTONI(argcount));

  err = sos_xdr_packbuf(fmt, args, curloc);

  if (err)
    {
      sos_error_printf("sos_xdr_sencode(): Type buffer invalid (How did it get past the first check??\n");
      sos_xdr_freebuf();
      SOS_RETURN(NULL);
    }

  va_end(args);
  va_end(saveargs);

  SOS_RETURN(&staticspace);
}



/*
 * Encode the data formatted by fmt and send it over the wire (fd) using wfun
 *
 * Returns 0 on write returning 0
 * Returns -1 on any error
 * Returns number of bytes written
 *
 * (if fd is NBIO, you must provide a wfun which takes care of that)
 */
int
sos_xdr_wencode(int fd, int (*wfun)(int fd, caddr_t buf, __SIZE_TYPE__ len), char *fmt, ...)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_wencode", NULL);
  va_list args;
  sos_string *buf;
  int ret;

  va_start(args,fmt);

  buf = sos_xdr_sencode_int(fmt, args);
  va_end(args);

  ret = sos_xdr_sput(fd, buf, wfun);

  sos_xdr_freebuf();
  SOS_RETURN(ret);
}



/*
 * Decode a buffer containing xdred data in fmt
 *
 * Returns -1 on error
 * Returns number of arguments decoded
 */
int
sos_xdr_sdecode(char *buf, char *fmt, ...)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_sdecode", NULL);
  va_list args;
  int ret;

  va_start(args,fmt);
  ret = sos_xdr_sdecode_int(buf, fmt, args);
  va_end(args);

  SOS_RETURN(ret);
}



/*
 * Internal varargs string decode
 */
int sos_xdr_sdecode_int(char *buf, char *fmt, va_list args)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_sdecode_int", NULL);
  char *curloc;
  int err;
  int argcount = 0;
  int argfound = 0;
  char type;

  if ( !buf || !fmt || !args )
    {
      sos_error_printf("Illigal Arguments\n");
      SOS_RETURN(-1);
    }

  curloc = buf;

  UNPACK(int, curloc, argcount, SOS_NTOHI);

  err = sos_xdr_unpackbuf(fmt, args, &curloc, &argfound);

  if (err)
    {
      sos_error_printf("Invalid format: %s\n",fmt);
      SOS_RETURN(-1);
    }

  if (argfound != argcount)
    {
      sos_error_printf("Number of unpacked args != reported value (%d %d)\n",argfound,argcount);
      SOS_RETURN(-1);
    }

  va_end(args);

  SOS_RETURN(argfound);
}



/*
 * Decode from wire
 *
 * Read in the data, unpack it, return number of arguments parsed to user
 *
 * Returns -1 on any error
 * Returns 0 on EOF on fd
 *
 * All pointer types (string, sos_string) are set to point into a
 * space which is freed if you call sos_xdr_sencode, sos_xdr_wencode,
 * sos_xdr_wdecode, or sos_xdr_freebuf People wanting permanence
 * should copy the data.
 */
int
sos_xdr_wdecode(int fd, int (*rfun)(int fd, caddr_t buf, __SIZE_TYPE__ len), char *fmt, ...)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_wdecode", NULL);
  va_list args;
  int cnt = 0;
  int ret; 
  int netlen;
  sos_string buf;

  sos_xdr_freebuf();

  if ((ret = sos_xdr_sget(fd, &buf, rfun)) < 1)
    {
      sos_error_printf("Could not read from wire\n");
      SOS_RETURN(-1);
    }

  va_start(args,fmt);
  ret = sos_xdr_sdecode_int(buf.str, fmt, args);
  va_end(args);

  sos_debug_printf_and(128,"Returning %d\n",ret);

  if (ret < 0)
    sos_xdr_freebuf();

  SOS_RETURN(ret);
}



/*
 * Free buffer allocated by sos_xdr_wdecode()
 */
void 
sos_xdr_freebuf(void)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_freebuf", NULL);
  if (staticspace.str)
    {
      free(staticspace.str);
      staticspace.str = NULL;
      staticspace.len = 0;
    }
  SOS_VRETURN();
}



/*
 * Compute the number of arguments and the length of the data indicated
 * by the fmtptr.
 * 
 * Returns 0 on success and >0 on error. Argcount and length are sideeffects.
 */
static int
sos_xdr_compute_arg_number_length (char *fmt, va_list args, 
				   int *pargcount, int *plength )
{
  SOS_ENTRY("sos_xdr", "sos_xdr_compute_arg_number_length", NULL);
  int err=0;
  char *tmpstr;
  sos_string *tmpbuf;
  int argcount = 0;
  int length =0;

   for(;*fmt;fmt++)
    {
      switch(*fmt)
	{
	case 'i':		/* Int */
	  argcount++;
	  (void)va_arg(args,int);
	  length += sizeof(int);	/* Size of data */
	  length++;		/* data type */
	  break;

#ifndef SOS_NOQUAD	  
	case 'q':		/* quad */
	  argcount++;
	  (void)va_arg(args,sos_quad_t);
	  length += sizeof(sos_quad_t);	/* Size of data */
#if defined SOS_QUAD_IS_LONG
	  length += sizeof(sos_quad_t);
#endif /* SOS_QUAD_IS_LONG */
	  length++;		/* data type */
	  break;
#endif /* SOS_NOQUAD */

	case 'l':		/* Long */
	  argcount++;
	  (void)va_arg(args,long);
	  length += sizeof(long);	/* Size of data */
	  length++;		/* data type */
	  break;

	case 's':		/* Short */
	  argcount++;
	  (void)va_arg(args,short);
	  length += sizeof(short);		/* Size of data */
	  length++;		/* data type */
	  break;

	case 'c':		/* Char */
	  argcount++;
	  (void)va_arg(args,char);
	  length += sizeof(char);		/* Size of data */
	  length++;		/* data type */
	  break;

	case 'S':		/* String */
	  argcount++;
	  tmpstr = va_arg(args,char *);
	  length += strlen(tmpstr)+1; /* Need to find the length */
	  length += sizeof(char *);		/* Length of length :-) */
	  sos_debug_printf_eq(2, "len tmpstr = %d (len is %d)\n",strlen(tmpstr),length);
	  length++;		/* Data type */
	  break;

	case 'B':		/* Buffer */
	  argcount++;
	  tmpbuf = va_arg(args,sos_string *);
	  length += tmpbuf->len;/* Need to find the length */
	  sos_debug_printf_and(2,"len tmpbuf = %d (len is %d)\n",tmpbuf->len,length);
	  length += sizeof(char *);		/* Length of length :-) */
	  length++;		/* Data type */
	  break;
	  
	default:
	  err++;		/* Invalid format */
	  sos_error_printf("Format character %c is invalid\n",*fmt);
	}
    }
  
  *pargcount = argcount;
  *plength = length;

  sos_debug_printf_and(64,"Returning: argcount: %d -- length: %d\n", argcount, length);

  SOS_RETURN(err);
}



/*
 * Encode supplied data into one buffer
 */
static int
sos_xdr_packbuf(char *fmt, va_list saveargs, char *curloc)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_packbuf", NULL);
  int err=0;
  char *tmpstr;
  sos_string *tmpbuf;
  int tmplen;

  /* XDR encoding */
  for(;*fmt;fmt++)
    {
      switch(*fmt)
	{
	case 'i':		/* Int */
	  PACK(char, curloc, *fmt);
	  PACK(int, curloc, SOS_HTONI(va_arg(saveargs, int)));
	  break;

	  /* 
	   * Quad packing. If quads are long then pack up a leading long of
	   * 0 so that a receiving machine which *does* support real quads 
	   * will do the right thing. Of course when it tries to send to 
	   * a quad *back* .....
	   */

#ifndef SOS_NOQUAD	  
	case 'q':		/* quad */
	  PACK(char, curloc, *fmt);
#ifdef SOS_QUAD_IS_LONG
	  CPACK(sos_quad_t, curloc, 0L,SOS_HTONQ);
#endif /* SOS_QUAD_IS_LONG */
	  CPACK(sos_quad_t, curloc, (va_arg(saveargs, sos_quad_t)),SOS_HTONQ);
	  break;
#endif /* SOS_NOQUAD */

	case 'l':		/* Long */
	  PACK(char, curloc, *fmt);
	  PACK(long, curloc, SOS_HTONL(va_arg(saveargs, long)));
	  break;

	case 's':		/* Short */
	  PACK(char, curloc, *fmt);
	  PACK(short, curloc, (short)SOS_HTONS(va_arg(saveargs, int)));
	  break;

	case 'c':		/* Char */
	  PACK(char, curloc, *fmt);
	  PACK(char, curloc, (char)(va_arg(saveargs, int)));
	  break;

	case 'S':		/* String */
	  PACK(char, curloc, *fmt);
	  tmpstr = va_arg(saveargs,char *);
	  tmplen = strlen(tmpstr)+1;
	  PACK(int, curloc, SOS_HTONI(tmplen));
	  MPACK(curloc,tmpstr,tmplen);
	  break;

	case 'B':		/* Buffer */
	  PACK(char, curloc, *fmt);
	  tmpbuf = va_arg(saveargs,sos_string *);
	  PACK(int, curloc, SOS_HTONI(tmpbuf->len));
	  MPACK(curloc, tmpbuf->str, tmpbuf->len);
	  break;

	default:
	  sos_error_printf("Format character %c is invalid\n",*fmt);
	  err++;		/* Invalid format */
	}
    }

  SOS_RETURN(err);
}



/*
 * Unpack encoded data into args
 */
static int
sos_xdr_unpackbuf(char *fmt, va_list args, char **pcurloc, int *pargfound)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_unpackbuf", NULL);
  char *curloc=*pcurloc;
  int err=0;
  char type;
  int argfound=0;
  int tmplen;
  sos_string *tmpbuf;
  
  /* XDR decoding */
  for(;*fmt;fmt++)
    {
      switch(*fmt)
	{
	case 'i':		/* Int */
	  UNPACK(char, curloc, type, (char));
	  if (type != *fmt) { err++; break; }
	  PUNPACK(int, curloc, (va_arg(args, int *)), SOS_NTOHI);
	  argfound++;
	  break;

	  /* 
	   * Quad unpacking. If quads are long we're up the creek (unless
	   * our peer is similarly crippled), just read in the first 
	   * sizeof(long) bytes and throw them away.
	   */
#ifndef SOS_NOQUAD	  
	case 'q':		/* quad */
	  UNPACK(char, curloc, type, (char));
	  if (type != *fmt) { err++; break; }
#if defined  SOS_QUAD_IS_LONG
	  {
	    sos_quad_t tmp;
	    sos_quad_t *tmpp=&tmp;
	    PUNPACK(sos_quad_t, curloc, tmpp, SOS_XDR_NTOHQ);
	  }
#endif /* SOS_QUAD_IS_LONG */
	  PUNPACK(sos_quad_t, curloc, (va_arg(args, sos_quad_t *)), SOS_XDR_NTOHQ);
	  argfound++;
	  break;
#endif /* SOS_NOQUAD */

	case 'l':		/* Long */
	  UNPACK(char, curloc, type, (char));
	  if (type != *fmt) { err++; break; }
	  PUNPACK(long, curloc, (va_arg(args, long *)), SOS_NTOHL);
	  argfound++;
	  break;

	case 's':		/* Short */
	  UNPACK(char, curloc, type, (char));
	  if (type != *fmt) { err++; break; }
	  PUNPACK(short, curloc, (va_arg(args, short *)), SOS_NTOHS);
	  argfound++;
	  break;

	case 'c':		/* Char */
	  UNPACK(char, curloc, type, (char));
	  if (type != *fmt) { err++; break; }
	  PUNPACK(char, curloc, (va_arg(args, char *)), (char));
	  argfound++;
	  break;

	case 'S':		/* String */
	  UNPACK(char, curloc, type, (char));
	  if (type != *fmt) { err++; break; }
	  UNPACK(int, curloc, tmplen, SOS_NTOHI);
	  MUNPACK((va_arg(args, char **)), curloc, tmplen);
	  argfound++;
	  break;

	case 'B':		/* Buffer */
	  UNPACK(char, curloc, type, (char));
	  if (type != *fmt) { err++; break; }
	  tmpbuf = va_arg(args, sos_string *);
	  UNPACK(int, curloc, tmpbuf->len, SOS_NTOHI);
	  if ( sos_debug_and(4) )
	    {
	      char *p;
	      int i;
	      fprintf(stderr,"sos_xdr_unpackbuf(): Decoding Buffer:\n");
	      fprintf(stderr,"\tLength: %d\n\tBuffer: **", tmpbuf->len);
	      for (p=curloc,i=0; i < tmpbuf->len; p++,i++)
		fprintf(stderr,"%c", *p);
	      fprintf(stderr,"**\n");
	    }
	  MUNPACK((&(tmpbuf->str)), curloc, tmpbuf->len);
	  argfound++;
	  break;

	default:
	  sos_error_printf("Format character %c is invalid\n",*fmt);
	  err++;		/* Invalid format */
	}
      if (err) break;
    }

  if (pargfound)
    *pargfound += argfound;
  *pcurloc = curloc;

  SOS_RETURN(err);
}



/*
 * Unpack encoded data into args
 */
static int
sos_xdr_compare_fmt_buf(char *fmt, char *curloc, int lenleft)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_compare_fmt_buf", NULL);
  int err=0;
  char type;
  int argfound=0;
  int tmplen;
  sos_string *tmpbuf;
  
#define DECREMENT_LENLEFT(down) do { if ((lenleft -= down) < 0) { sos_error_printf("Over bounds of buf\n"); SOS_RETURN(-1); } } while (0)

  /* XDR decoding */
  for(;*fmt;fmt++)
    {
      switch(*fmt)
	{
	case 'i':		/* Int */
	  UNPACK(char, curloc, type, (char));
	  DECREMENT_LENLEFT(1);
	  if (type != *fmt) { err++; break; }
	  curloc += sizeof(int);
	  DECREMENT_LENLEFT(sizeof(int));
	  argfound++;
	  break;

#ifndef SOS_NOQUAD	  
	case 'q':		/* quad */
	  UNPACK(char, curloc, type, (char));
	  DECREMENT_LENLEFT(1);
	  if (type != *fmt) { err++; break; }
	  curloc += sizeof(sos_quad_t);
	  DECREMENT_LENLEFT(sizeof(sos_quad_t));
	  argfound++;
	  break;
#endif /* SOS_NOQUAD */

	case 'l':		/* Long */
	  UNPACK(char, curloc, type, (char));
	  DECREMENT_LENLEFT(1);
	  if (type != *fmt) { err++; break; }
	  curloc += sizeof(long);
	  DECREMENT_LENLEFT(sizeof(long));
	  argfound++;
	  break;

	case 's':		/* Short */
	  UNPACK(char, curloc, type, (char));
	  DECREMENT_LENLEFT(1);
	  if (type != *fmt) { err++; break; }
	  curloc += sizeof(short);
	  DECREMENT_LENLEFT(sizeof(short));
	  argfound++;
	  break;

	case 'c':		/* Char */
	  UNPACK(char, curloc, type, (char));
	  DECREMENT_LENLEFT(1);
	  if (type != *fmt) { err++; break; }
	  curloc += sizeof(char);
	  DECREMENT_LENLEFT(sizeof(char));
	  argfound++;
	  break;

	case 'S':		/* String */
	  UNPACK(char, curloc, type, (char));
	  DECREMENT_LENLEFT(1);
	  if (type != *fmt) { err++; break; }
	  UNPACK(int, curloc, tmplen, SOS_NTOHI);
	  DECREMENT_LENLEFT(sizeof(int));
	  curloc += tmplen;
	  DECREMENT_LENLEFT(tmplen);
	  argfound++;
	  break;

	case 'B':		/* Buffer */
	  UNPACK(char, curloc, type, (char));
	  DECREMENT_LENLEFT(1);
	  if (type != *fmt) { err++; break; }
	  UNPACK(int, curloc, tmplen, SOS_NTOHI);
	  DECREMENT_LENLEFT(sizeof(int));
	  curloc += tmplen;
	  DECREMENT_LENLEFT(tmplen);
	  argfound++;
	  break;

	default:
	  sos_error_printf("Format character %c is invalid\n",*fmt);
	  err++;		/* Invalid format */
	}
      if (err) break;
    }

  if (err)
    SOS_RETURN(-1);

  SOS_RETURN(argfound);
}


/*
 * Write to wire (blocking version)  (Compatable with NBIO)
 */
int
sos_xdr_sput (int fd, sos_string *buf, int (*wfun)(int fd, caddr_t buf, __SIZE_TYPE__ len))
{
  SOS_ENTRY("sos_xdr", "sos_xdr_sput", NULL);
  int netlen;
  int cnt = 0;
  int ret;
    
  if (!buf || !wfun || fd < 0)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-2);
    }

  sos_debug_printf_and(64,"XDRBUFLEN = %d: Sending\n",buf->len);

  /* Send over length of data */
  
  netlen = SOS_HTONI(buf->len);
  while (cnt < sizeof(int))
    {
      sos_wait_output(fd, NULL);
      ret = (*wfun)(fd, ((char *)(&netlen)) + cnt, sizeof(int) - cnt);

      if (ret < 1)
	{
	  sos_error_printf("write function failed (ret %d): %s\n", ret, strerror(errno));
	  SOS_RETURN(ret);
	}

      cnt += ret;
    }

  sos_debug_printf_and(64,"Sending over data\n");

  /* Send over data */
  cnt = 0;
  while (cnt < buf->len)
    {
      sos_wait_output(fd, NULL);
      ret = (*wfun)(fd, buf->str + cnt, buf->len - cnt);

      if (ret < 1)
	{
	  sos_error_printf("write function failed (ret %d): %s\n", ret, strerror(errno));
	  SOS_RETURN(ret);
	}

      cnt += ret;
    }

  SOS_RETURN(cnt);
}



/*
 * Read from wire (blocking version)  (Compatable with NBIO)
 * Returns ``static'' xdr space
 */
int
sos_xdr_sget(int fd, sos_string *buf, int (*rfun)(int fd, caddr_t buf, __SIZE_TYPE__ len))
{
  SOS_ENTRY("sos_xdr", "sos_xdr_sget", NULL);
  int netlen;
  int cnt = 0;
  int ret;
    
  if (!buf || !rfun || fd < 0)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-2);
    }

  sos_xdr_freebuf();

  /* read the length of the xdr buffer */
  cnt = 0;
  while (cnt < sizeof(int))
    {
      sos_wait_input(fd,NULL);
      ret = (*rfun)(fd, ((char *)(&netlen)) + cnt, sizeof(int) - cnt);

      if (ret < 1)
	{
	  sos_error_printf("Error reading length: %d\n",ret);
	  SOS_RETURN(ret);
	}

      cnt += ret;
    }

  staticspace.len = SOS_NTOHI(netlen);
  sos_debug_printf_and(64,"XDR buflen = %d\n",staticspace.len);

  if (sos_allocstr(&staticspace, staticspace.len) == NULL)
    {
      sos_error_printf("Could not allocate staticspace buffer (sos_buffer): %s\n",strerror(errno));
      SOS_RETURN(-1);
    }

  /* 
   * Checking (!buf) is redundant (args already verified).
   */
  buf->str = staticspace.str;
  buf->len = staticspace.len;
  
  sos_debug_printf_and(64,"B4read\n");

  /* read the buffer */
  cnt = 0;
  while (cnt < staticspace.len)
    {
      sos_wait_input(fd,NULL);
      ret = (*rfun)(fd, staticspace.str + cnt, staticspace.len - cnt);

      if (ret < 1)
	{
	  sos_error_printf("Error reading length: %d: %s\n",ret,strerror(errno));
	  sos_xdr_freebuf();
	  SOS_RETURN(ret);
	}

      cnt += ret;
    }

  SOS_RETURN(cnt);
}



/*
 * Append data to a previously constructed XDR buffer
 */
sos_string *sos_xdr_sappend(sos_string *buf,char *fmt, ...)
 {
   SOS_ENTRY("sos_xdr", "sos_xdr_sappend", NULL);
   va_list args;
   sos_string *ret = NULL;

  va_start(args,fmt);
  ret = sos_xdr_sappend_int(buf,fmt, args);
  va_end(args);

  SOS_RETURN(ret);
}



/*
 * Internal append--varargs version
 */
static sos_string *sos_xdr_sappend_int(sos_string *buf, char *fmt, va_list args)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_sappend_int", "Format is %s", fmt);
  va_list saveargs;
  char *curloc;
  sos_string *tbuf;
  int length = 0;
  int err = 0;
  int argcount = 0;
  char *tmpbuf;			/* Just used for realloc */
  int tmpint;
  int tmplength=0;

  if ( !fmt )
    {
      sos_error_printf("Ilegal Arguments\n");
      SOS_RETURN(NULL);
    }

  if ( !buf )
    {
      /* 
       * It is quite possible that the caller simply hasn't made a call
       * to sos_xdr_sencode yet. In that case treat *this* call as the initial
       * call.
       */
      tbuf=sos_xdr_sencode_int(fmt, args);
      SOS_RETURN(tbuf);
    }

  length = buf->len;

  saveargs = args;

  err=sos_xdr_compute_arg_number_length(fmt, args, &argcount, &tmplength);

  if (err)
    {
      sos_error_printf("Invalid format: %s\n",fmt);
      SOS_RETURN(NULL);
    }

  sos_debug_printf_and(64,"Arcount: %d -- length: %d\n",argcount, length);

  length += tmplength;

  tmpbuf = buf->str;
  if ((buf->str = (char *)realloc(tmpbuf,length)) == NULL)
    {
      free (tmpbuf);		/* Don't leak memory please. */
      sos_error_printf("Reallocing XDR buffer: %s\n", strerror(errno));
      SOS_RETURN(NULL);
    }

  sos_debug_printf_and(128,"Reallocing %d bytes\n",length);

  /* Update total argument count. Get old value and repack with update. */
  curloc = buf->str;
  UNPACK(int, curloc, tmpint, SOS_NTOHI);
  curloc = buf->str;
  PACK(int, curloc, SOS_HTONI(argcount+tmpint));

  curloc = buf->str + buf->len;
  buf->len = length;

  err = sos_xdr_packbuf(fmt, args, curloc);

  if (err)
    {
      sos_error_printf("Invalid format: %s\n",fmt);
      sos_xdr_freebuf();
      SOS_RETURN(NULL);
    }

  va_end(saveargs);

  SOS_RETURN(buf);
}



/*
 * Prepend data to a previously constructed XDR buffer
 */
sos_string *sos_xdr_sprepend(sos_string *buf,char *fmt, ...)
 {
   SOS_ENTRY("sos_xdr", "sos_xdr_sprepend", NULL);
   va_list args;
   sos_string *ret = NULL;

  va_start(args,fmt);
  ret = sos_xdr_sprepend_int(buf,fmt, args);
  va_end(args);

  SOS_RETURN(ret);
}



/*
 * Internal prepend--varargs version
 */
static sos_string *sos_xdr_sprepend_int(sos_string *buf, char *fmt, va_list args)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_sprepend_int", "Format is %s", fmt);
  va_list saveargs;
  char *curloc;
  sos_string *tbuf;		/* This will point to malloc'ed space if used. */
  int length = 0;
  int err = 0;
  int argcount = 0;
  char *tmpbuf;			/* Just used for realloc */
  int tmpint;
  int tmplength=0;

  if ( !fmt )
    {
      sos_error_printf("Ilegal Arguments\n");
      SOS_RETURN(NULL);
    }

  if ( !buf )
    {
      /* 
       * It is quite possible that the caller simply hasn't made a call
       * to sos_xdr_sencode yet. In that case treat *this* call as the initial
       * call.
       */
      tbuf=sos_xdr_sencode_int(fmt, args);
      SOS_RETURN(tbuf);
    }

  length = buf->len;

  saveargs = args;

  err=sos_xdr_compute_arg_number_length(fmt, args, &argcount, &tmplength);

  if (err)
    {
      sos_error_printf("Invalid format: %s\n",fmt);
      SOS_RETURN(NULL);
    }

  sos_debug_printf_and(64,"Arcount: %d -- length: %d\n",argcount, tmplength);

  length += tmplength;

  tmpbuf = buf->str;
  if ((buf->str = (char *)realloc(tmpbuf,length)) == NULL)
    {
      free (tmpbuf);		/* Don't leak memory please. */
      sos_error_printf("Reallocing XDR buffer: %s\n", strerror(errno));
      SOS_RETURN(NULL);
    }

  sos_debug_printf_and(128,"Reallocing %d bytes\n",length);

  /* Update total argument count. Get old value and repack with update. */
  curloc = buf->str;
  UNPACK(int, curloc, tmpint, SOS_NTOHI);
  curloc = buf->str;
  PACK(int, curloc, SOS_HTONI(argcount+tmpint));

  sos_debug_printf_and(128, "Moving some data for prepending -- Size: %d\n", buf->len-sizeof(int));

  SOS_OVERLAPPING_MOVE(curloc+tmplength, curloc, buf->len-sizeof(int));
  buf->len = length;

  err = sos_xdr_packbuf(fmt, args, curloc);

  if (err)
    {
      sos_error_printf("Invalid format: %s\n",fmt);
      sos_xdr_freebuf();
      SOS_RETURN(NULL);
    }

  va_end(saveargs);

  SOS_RETURN(buf);
}



/*
 * Pull out the next few arguments from an XDR string, staring at position
 * curloc. no error checking. Buf is currently unused.  curloc is updated 
 * automatically.
 */
int 
sos_xdr_speek(sos_string *buf, char **pcurloc, char *fmt, ...)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_speek", NULL);
  va_list args;
  int ret;
  
  va_start(args,fmt);
  ret = sos_xdr_speek_int(buf, pcurloc, fmt, args);
  va_end(args);

  SOS_RETURN(ret);
}



/*
 * Internal peek--varargs
 */
static int
sos_xdr_speek_int(sos_string *buf, char **pcurloc, char *fmt, va_list args)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_speek_int", NULL);
  int err;
  int argcount = 0;
  int argfound = 0;
  char *curloc=NULL;
  int length;
  char type;


  if ( !buf || !fmt )
    {
      sos_error_printf("Ilegal Arguments\n");
      SOS_RETURN(-1);
    }

  if ( pcurloc )
    {
      curloc = *pcurloc;
    }

  if ( !curloc ) 
    {
      sos_debug_printf_and(128,"Initializing the curloc\n");
      curloc = buf->str;
      sos_xdr_get_argcount(buf, &curloc);
    }

  
  if ((argcount = sos_xdr_get_argcount(buf, NULL)) < 0)
    {
      sos_error_printf("Invalid XDR buffer\n");
      SOS_RETURN(-1);
    }

  if ((buf->len - (curloc-buf->str)) == 0)
    {
      sos_debug_printf_and(128,"Perfect End Of Buffer\n");
      SOS_RETURN(0);				/* EOB */
    }

  /*
   * Let's see if the supplied format matches the encoded format
   * and let's make sure we don't run off the end of our buffer
   */
  if (sos_xdr_compare_fmt_buf(fmt,curloc,buf->len - (curloc-buf->str)) < 0)
    {
      sos_error_printf("Some error during comparison of buffer to format string\n");
      SOS_RETURN(-1);
    }

  err = sos_xdr_unpackbuf(fmt, args, &curloc, &argfound);

  if (err)
    {
      sos_error_printf("Invalid format: %s\n",fmt);
      /* We will leave the current location UNMODIFIED */
      SOS_RETURN(-1);
    }

  if (pcurloc)
    *pcurloc=curloc;

  va_end(args);

  /* Update current location */
  SOS_RETURN(argfound);
}



/*
 * Slurp up the argcount from an XDR buffer if you are doing self-parsing.
 */
int
sos_xdr_get_argcount(sos_string *buf, char **pcurloc)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_argcount", NULL);
  int argcount;
  char *curloc;

  if (!buf && (!pcurloc || !*pcurloc))
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if (!pcurloc || !(curloc = *pcurloc))
    {
      curloc = buf->str;
    }

  UNPACK(int, curloc, argcount, SOS_NTOHI);
  
  if (pcurloc) *pcurloc=curloc;
  SOS_RETURN(argcount);
}



#ifndef SOS_NOQUAD
/*
 * Quad print
 */
static void
sos_xdr_print_quad(sos_quad_t num)
{
  SOS_ENTRY("sos_xdr", "sos_xdr_print_quad", NULL);
  unsigned char *tquadp;

#if defined (SOS_QUAD_IS_LONG)
printf("quad: %ld\n", num);
#else
  tquadp=(char *)&num;
  printf ("quad: %02x %02x %02x %02x %02x %02x %02x %02x\n",
	  tquadp[0],
	  tquadp[1],
	  tquadp[2],
	  tquadp[3],
	  tquadp[4],
	  tquadp[5],
	  tquadp[6],
	  tquadp[7]);

  SOS_VRETURN();
#endif /* SOS_QUAD_IS_LONG */
}
#endif SOS_NOQUAD


