#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * SOS NBIO routines
 *
 * Perform chunk I/O over non-blocking streams
 *
 *
 * Debug levels:
 *
 */

#include "sos.h"


#define NS_READY	0
#define NS_LEN		1
#define NS_BUF		2

#define MAX_NORMAL_MESSAGE_SIZE 65536	/* 2^16 bytes, arbitrary */


/* Internal read function--handle common error conditions */
#define INTERNAL_READ(tmp,fd,iofun,buf,len) do \
{ \
  tmp = (*(iofun))((fd),(buf),(len)); \
 \
  if (tmp == 0)		/* EOF */ \
    { \
      sos_error_printf("Signaled EOF during NBIO read on fd %d\n",(fd)); \
      SOS_RETURN(SOS_NBIO_EOF); \
    } \
 \
  if (tmp < 0)		/* Error? */ \
    { \
      if (errno == EWOULDBLOCK || errno == EAGAIN) \
	{ \
	  SOS_RETURN(SOS_NBIO_NOPROGRESS); \
	} \
      sos_error_printf("Error occurred during NBIO read on fd %d: %s\n",(fd),strerror(errno)); \
      SOS_RETURN(SOS_NBIO_IOFUN_ERROR); \
    } \
 \
  if (tmp > (len))		/* Got more than I asked for */ \
    { \
      sos_error_printf("%d > %d--I got more data than I wanted\n",tmp,(len)); \
      SOS_RETURN(SOS_NBIO_ADMIN_ERROR); \
    } \
} while (0)



/* Internal write funciton--handle common error conditions */
#define INTERNAL_WRITE(tmp,fd,iofun,buf,len) do \
{ \
  tmp = (*(iofun))((fd),(buf),(len)); \
 \
  if (tmp == 0)			/* EOF */ \
    { \
      sos_error_printf("Signaled EOF during NBIO write of fd %d\n",(fd)); \
      SOS_RETURN(SOS_NBIO_EOF); \
    } \
 \
  if (tmp < 0) \
    { \
      if (errno == EWOULDBLOCK || errno == EAGAIN) \
	{ \
	  SOS_RETURN(ret); \
	} \
      sos_error_printf("Error occurred during NBIO write of fd %d: %s\n",(fd),strerror(errno)); \
      SOS_RETURN(SOS_NBIO_IOFUN_ERROR); \
    } \
 \
  if (tmp > (len)) \
    { \
      sos_error_printf("%d > %d--I got more data than I wanted\n",tmp,(len)); \
      SOS_RETURN(SOS_NBIO_ADMIN_ERROR); \
    } \
} while (0)



/*
 * Take a previously opened file descriptor, plus other information,
 * and create a SOS_NBIO_HANDLE which will handle all information about
 * the non-blocking output buffer.
 */
sos_nbio_handle *sos_nbio_fdopen(int fd, int (*iofun) (int fd, caddr_t buf, __SIZE_TYPE__ nbyte), int flags)
{
  SOS_ENTRY("sos_nbio","sos_nbio_fdopen",NULL);
  sos_nbio_handle *ret = NULL;
  int fdflags, newflags;

  if (fd < 0 || !iofun || (flags != O_RDONLY && flags != O_WRONLY))
    {
      sos_error_printf("Invalid arguments (%d,%x,%d)\n",fd,iofun,flags);
      SOS_RETURN(NULL);
    }

  /* Find current flags */
  if ((fdflags = fcntl(fd,F_GETFL,(FCNTL_ARG_T)NULL)) < 0)
    {
      sos_error_printf("Could not get descriptor status flags on fd %d: %s\n",fd,strerror(errno));
      SOS_RETURN(NULL);
    }

  /* Get handle */
  if (!(ret = (sos_nbio_handle *)malloc(sizeof(sos_nbio_handle))))
    {
      sos_error_printf("Could not malloc sos_nbio_handle (%d bytes): %s\n",sizeof(sos_nbio_handle),strerror(errno));
      SOS_RETURN(NULL);
    }

  /* Common initialization */
  ret->fd = fd;
  ret->iofun = iofun;
  ret->fdflags = fdflags;

  if (flags == O_RDONLY)
    {				/* Read handle */
      ret->type = O_RDONLY;
      ret->i.r.state = NS_READY;
      ret->i.r.bytes_so_far = 0;
      ret->i.r.bytes_desired = 0;
      ret->i.r.buf = NULL;
    }
  else
    {				/* Write handle */
      ret->type = O_WRONLY;
      ret->i.w.state = NS_READY;
      ret->i.w.bytes_so_far = 0;
      if (!(ret->i.w.queue = dll_create(NULL, NULL, DICT_UNORDERED, NULL)))
	{
	  sos_error_printf("Could not create NBIO DLL queue: %s\n",strerror(errno));
	  sos_nbio_cleanup(ret);
	  SOS_RETURN(NULL);
	}
    }

  /* Add NBIO */
  if (fcntl(ret->fd,F_SETFL,(FCNTL_ARG_T)(ret->fdflags|O_NONBLOCK)) < 0)
    {
      sos_error_printf("Could not set descriptor status flags on fd %d to %d: %s\n",ret->fd,ret->fdflags,strerror(errno));
      sos_nbio_cleanup(ret);
      SOS_RETURN(NULL);
    }

  SOS_RETURN(ret);
}



/*
 * Get rid of a handle by freeing memory,
 * undoing fdflag munging, and destroying anything
 * needed
 */
void sos_nbio_cleanup(sos_nbio_handle *snh)
{
  SOS_ENTRY("sos_nbio","sos_nbio_cleanup",NULL);
  sos_string *tmp;

  if (!snh)
    SOS_VRETURN();

  if (fcntl(snh->fd,F_SETFL,(FCNTL_ARG_T)(snh->fdflags)) < 0)
    sos_error_printf("Could not reset descriptor status flags on fd %d to %d: %s\n",snh->fd,snh->fdflags,strerror(errno));

  if (snh->type == O_RDONLY)
    {
      if (snh->i.r.buf)
	free(snh->i.r.buf);
    }
  else
    {
      if (snh->i.w.queue)
	{
	  for(tmp = (sos_string *)dll_maximum(snh->i.w.queue);tmp;tmp = (sos_string *)dll_maximum(snh->i.w.queue))
	    {
	      dll_delete(snh->i.w.queue, tmp);
	      free(tmp->str);
	      free(tmp);
	    }
	  dll_destroy(snh->i.w.queue);
	}
    }
  free(snh);
  SOS_VRETURN();
}



/*
 * Find out how much space is buffered by the NBIO routines
 *
 * This can be a little off because of the effects of
 * sending/receiving the length of the buffer, but it should be a good
 * order of magnitude.
 */
int sos_nbio_length(sos_nbio_handle *snh)
{
  SOS_ENTRY("sos_nbio","sos_nbio_length",NULL);
  int ret = 0;

  if (!snh)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if (snh->type == O_RDONLY)
    {
      ret = snh->i.r.bytes_so_far;
    }
  else
    {
      sos_string *tmp;

      /* Look at all of the output buffers in the queue */
      for(tmp = (sos_string *)dll_maximum(snh->i.w.queue);tmp;tmp = (sos_string *)dll_predecessor(snh->i.w.queue, tmp))
	{
	  ret += tmp->len;
	}
      ret -= snh->i.w.bytes_so_far;
      if (ret < 0)		/* huh? */
	ret = 0;
    }

  SOS_RETURN(ret);
}



/*
 * read data from a NBIO handle--may not return data right away because
 * of partial reads due to NBIO
 *
 * Return values
 *
 * -2 administrative error
 * -1 iofun error
 *  0 iofun EOF
 *  1 no forward progress made
 *  2 forward progress made
 *  3 complete buffer available
 */
int sos_nbio_read(sos_nbio_handle *snh, sos_string *bufin)
{
  SOS_ENTRY("sos_nbio","sos_nbio_read",NULL);
  int tmp;

  if (!snh || !bufin || snh->type != O_RDONLY)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
    }

  /* Loop for complete state transitions */
  while (1)
    switch(snh->i.r.state)
      {
      case NS_READY:
	INTERNAL_READ(tmp,snh->fd,snh->iofun,(caddr_t)&snh->i.r.bytes_desired,sizeof(snh->i.r.bytes_desired));
	if (tmp < 4)
	  {			/* Partial read of size length (YUK!) */
	    snh->i.r.state = NS_LEN;
	    snh->i.r.bytes_so_far = tmp;
	    SOS_RETURN(SOS_NBIO_PROGRESS);
	  }

	/* Got complete length */
	snh->i.r.bytes_desired = ntohl(snh->i.r.bytes_desired);
	if (!(snh->i.r.buf = malloc(snh->i.r.bytes_desired)))
	  {
	    sos_error_printf("Could not malloc %d bytes for incoming NBIO chunk\n",snh->i.r.bytes_desired);
	    /*
	     * XXX - We have seen problems where people telnet to a
	     * port controlled by NBIO and have typed characters causing
	     * NBIO to think the length of the message is truely large
	     * >>2^24 bytes, causing malloc failures, causing ADMIN_ERROR
	     * causing the outside program to think everything was fucked
	     * causing the program to die.
	     *
	     * I cannot think of a truely good solution (even
	     * handshaking to ensure a minimally complient application
	     * is not good enough), so instead of solving the problem,
	     * we will just say that allocation failures over a
	     * certain size will not be considered OS error, but
	     * rather I/O errors.  Whatever.
	     */
	    if (snh->i.r.bytes_desired > MAX_NORMAL_MESSAGE_SIZE)
	      SOS_RETURN(SOS_NBIO_IOFUN_ERROR);
	    else
	      SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
	  }
	snh->i.r.bytes_so_far = 0;
 	snh->i.r.state = NS_BUF;
	break;			/* Loop into buffer read */


      case NS_LEN:		/* Got partial length (YUK) */
	INTERNAL_READ(tmp,snh->fd,snh->iofun,((caddr_t)&snh->i.r.bytes_desired + snh->i.r.bytes_so_far),sizeof(snh->i.r.bytes_desired) - snh->i.r.bytes_so_far);

	if (tmp + snh->i.r.bytes_so_far < 4)
	  {			/* Partial read of size length (YUK!) */
	    snh->i.r.bytes_so_far += tmp;
	    SOS_RETURN(SOS_NBIO_PROGRESS);
	  }

	/* Got complete length */
	snh->i.r.bytes_desired = ntohl(snh->i.r.bytes_desired);
	if (!(snh->i.r.buf = malloc(snh->i.r.bytes_desired)))
	  {
	    sos_error_printf("Could not malloc %d bytes for incoming NBIO chunk\n",snh->i.r.bytes_desired);
	    /* XXX - See above diatribe about this problem */
	    if (snh->i.r.bytes_desired > MAX_NORMAL_MESSAGE_SIZE)
	      SOS_RETURN(SOS_NBIO_IOFUN_ERROR);
	    else
	      SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
	  }
	snh->i.r.bytes_so_far = 0;
	snh->i.r.state = NS_BUF;
	break;			/* Loop into buffer read */


      case NS_BUF:		/* Trying to read real data */
	INTERNAL_READ(tmp,snh->fd,snh->iofun,snh->i.r.buf + snh->i.r.bytes_so_far,snh->i.r.bytes_desired - snh->i.r.bytes_so_far);

	snh->i.r.bytes_so_far += tmp;

	/* Partial read */
	if (snh->i.r.bytes_so_far < snh->i.r.bytes_desired)
	  {
	    SOS_RETURN(SOS_NBIO_PROGRESS);
	  }

	/* We have a complete buffer!! */
	bufin->str = snh->i.r.buf;
	bufin->len = snh->i.r.bytes_desired;
	snh->i.r.buf = NULL;
	snh->i.r.bytes_desired = 0;
	snh->i.r.bytes_so_far = 0;
	snh->i.r.state = NS_READY;
	SOS_RETURN(SOS_NBIO_BUFCOMPLETE);
      }

  sos_error_printf("Impossible condition--I should never get here\n");
  SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
}



/*
 * write data to a NBIO handle--data may be queued for an indefinite period
 * because of NBIO concerns
 *
 * Return values
 *
 * -2 administrative error
 * -1 iofun error
 *  0 iofun EOF
 *  1 no forward progress made
 *  2 forward progress made
 *  3 at least one chunk finished
 */
int sos_nbio_write(sos_nbio_handle *snh, sos_string *bufout, int flags, fd_set *writefds, u_int *byteswritten)
{
  SOS_ENTRY("sos_nbio","sos_nbio_write",NULL);
  int ret = SOS_NBIO_NOPROGRESS;
  int bytesout;
  int tmp;
  sos_string *cur;

  if (!snh || snh->type != O_WRONLY || (bufout && (!bufout->str || bufout->len < 1)))
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
    }

  /* Assume there is work to do--we will reset if this changes */
  if (writefds)
    FD_SET(snh->fd,writefds);

  if (bufout)			/* Append buffer to output queue */
    {
      sos_string *newbuf = bufout;

      if (flags != SOS_NBIO_MAY_FREE)
	{			/* Need to make copy of input buffer */
	  /* Get sos_string framework */
	  if (!(newbuf = malloc(sizeof(sos_string))))
	    {
	      sos_error_printf("Could not malloc sos_string (%d bytes): %s\n",sizeof(sos_string),strerror(errno));
	      SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
	    }

	  /* Allocate space for duplicate buffer */
	  if (!(newbuf->str = malloc(bufout->len)))
	    {
	      sos_error_printf("Could not malloc duplicate output buffer (%d bytes): %s\n",bufout->len,strerror(errno));
	      free(newbuf);
	      SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
	    }

	  /* Copy data into new buffer */
	  newbuf->len = bufout->len;
	  memcpy(newbuf->str,bufout->str,bufout->len);
	}

      /* Insert buffer into outbut queue */
      if (dll_insert(snh->i.w.queue,newbuf) != DICT_OK)
	{
	  sos_error_printf("Could not insert buffer into output queue (DLL failure): %s\n",strerror(errno));
	  SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
	}
    }


  /* Loop while getting forward progress */
  while (1)
    { 
      /* Check for work to do... */
      if (!(cur = dll_maximum(snh->i.w.queue)))
	{
	  if (writefds)
	    FD_CLR(snh->fd,writefds);

	  SOS_RETURN(ret);
	}
     
      switch(snh->i.w.state)
	{
	case NS_READY:
	  snh->i.w.bytes_desired = htonl(cur->len);
	  snh->i.w.bytes_so_far = 0;
	  INTERNAL_WRITE(tmp,snh->fd,snh->iofun,(caddr_t)&snh->i.w.bytes_desired,sizeof(snh->i.w.bytes_desired));

	  if (byteswritten) *byteswritten += tmp;
	  snh->i.w.bytes_so_far += tmp;
	  if (ret < SOS_NBIO_PROGRESS) ret = SOS_NBIO_PROGRESS;

	  if (snh->i.w.bytes_so_far < sizeof(snh->i.w.bytes_desired))
	    {			/* Partial write of size--YUK */
	      snh->i.w.state = NS_LEN;
	      SOS_RETURN(ret);
	    }

	  /* Sent complete length */
	  snh->i.w.bytes_so_far = 0;
	  snh->i.w.state = NS_BUF;
	  break;		/* Loop into buffer write */

	case NS_LEN:
	  INTERNAL_WRITE(tmp,snh->fd,snh->iofun,((caddr_t)&snh->i.w.bytes_desired + snh->i.w.bytes_so_far),(sizeof(snh->i.w.bytes_desired) - snh->i.w.bytes_so_far));

	  if (byteswritten) *byteswritten += tmp;
	  snh->i.w.bytes_so_far += tmp;
	  if (ret < SOS_NBIO_PROGRESS) ret = SOS_NBIO_PROGRESS;

	  if (snh->i.w.bytes_so_far < sizeof(snh->i.w.bytes_desired))
	    {			/* Partial write of size--YUK */
	      SOS_RETURN(ret);
	    }

	  /* Sent complete length */
	  snh->i.w.bytes_so_far = 0;
	  snh->i.w.state = NS_BUF;
	  break;		/* Loop into buffer write */
	  
	case NS_BUF:
	  INTERNAL_WRITE(tmp,snh->fd,snh->iofun,(cur->str + snh->i.w.bytes_so_far),(cur->len - snh->i.w.bytes_so_far));

	  if (byteswritten) *byteswritten += tmp;
	  snh->i.w.bytes_so_far += tmp;
	  if (ret < SOS_NBIO_PROGRESS) ret = SOS_NBIO_PROGRESS;
	  
	  if (snh->i.w.bytes_so_far < cur->len)
	    {			/* Partial write of buffer */
	      SOS_RETURN(ret);
	    }

	  /* Complete write of buffer!! */
	  ret = SOS_NBIO_BUFCOMPLETE;

	  /* Nuke this element! */
	  dll_delete(snh->i.w.queue,cur);
	  free(cur->str);
	  free(cur);
	  snh->i.w.bytes_so_far = 0;
	  snh->i.w.bytes_desired = 0;
	  snh->i.w.state = NS_READY;
	  break;		/* Go around again to see if there is more work */
	}
    }

  sos_error_printf("Impossible condition--I should never get here\n");
  SOS_RETURN(SOS_NBIO_ADMIN_ERROR);
}
