#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * Read a line from a file descriptor.
 *
 * (Performing primitive editing, multiple
 * EOL sequences, various echoing options, etc)
 *
 * 1 - I/O debugging
 * 2 - Escape mode
 */

#include "sos.h"



/*
 * Gather a line's worth of data
 *
 * This performs character-by-character processing, but it uses getc
 * You need to specify echo mode,
 * You need to handle line editing,
 * You need to handle escape characters (telnet options)
 *
 * If the EOL sequence has special (editing, escape, etc) characters
 * in it, they will be in the sequence *and* be interpreted.  Be warned
 *
 * String will be null terminated
 * String will have eolseq in it
 * Will exit -1 on error
 * Will exit 0 on EOF
 * Will return # of characters read if all is well
 * Will return early if line fills up and return code will be negative
 * (-charread-1) to leave -1 free for error indication
 */
int 
sos_readline(FILE * in, FILE * out, sos_string * line, struct sos_readline_t *opts)
{
  SOS_ENTRY("sos_io","sos_readline",NULL);
  int eol_state[MAX_EOLSEQ];
  int eol_termstate[MAX_EOLSEQ];
  int numchar = 0;
  sos_string delseq;		/* Deletion sequence */
  int c;
  int exitstate = 0;

  if (!opts)
    {
      static struct sos_readline_t defaultopts;

      defaultopts.echo = -1;
      defaultopts.wantalert = 0;
      defaultopts.wantedit = 0;
      defaultopts.wantnull = 0;
      defaultopts.escape_char = 0;
      defaultopts.escape_mode = 0;
      defaultopts.eolseq[0].str = "\r\n";
      defaultopts.eolseq[0].len = 2;
      defaultopts.eolseq[1].str = "\r\0";
      defaultopts.eolseq[1].len = 2;
      defaultopts.eolseq[2].str = "\n";
      defaultopts.eolseq[2].len = 1;
      defaultopts.numeolseq = 3;

      opts = &defaultopts;
    }

  if (!in)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if (!out && opts->echo != -1)
    {
      sos_error_printf("No output stream yet echoing is turned on!!\n");
      SOS_RETURN(-1);
    }

  /* Initialize exit conditions */
  for (c = 0; c < opts->numeolseq; c++)
    {
      eol_state[c] = 0;
      eol_termstate[c] = opts->eolseq[c].len;
    }

  delseq.str = "\010 \010";
  delseq.len = strlen(delseq.str);

  /* Loop until we have a full line */
  while ((!exitstate) && (numchar < line->len))
    {
      char input;
      int ret;
      int dooutput;

      dooutput = 1;

      sos_debug_printf_and(1,"About to getc");

      /* Read one character--in better not be NBIO */
      ret = getc(in);

      sos_debug_printf_and(1,"gotc :%c:%d:%d:", (unsigned char)ret, (unsigned char)ret, opts->escape_char);

      if (ret == EOF || feof(in))
	{
	  line->str[numchar] = '\0';
	  SOS_RETURN(0);
	}
      input = (unsigned char)ret;

      /* If this might be part of the EOL sequence */
      for (c = 0; c < opts->numeolseq; c++)
	{
	tryagain:
	  if (input == opts->eolseq[c].str[eol_state[c]])
	    {
	      eol_state[c]++;
	      if (eol_state[c] >= eol_termstate[c])
		exitstate = 1;	/* We have SEEN the EOL */
	    }
	  else
	    {
	      if (eol_state[c] != 0)
		{
		  eol_state[c] = 0;
		  goto tryagain;
		}
	      eol_state[c] = 0;
	    }
	}

      if (!opts->wantnull && !input)
	continue;		/* Null does not make sense */

      /* Command line editing, SOS style */
      if (opts->wantedit)
	switch (input)
	  {
	  case 0:		/* Null does not make sense in a string */
	    continue;
	  case 4:		/* EOF */
	    if (out && write(fileno(out), "EOF\r\n", 5) != 5)
	      {
		sos_error_printf("Could not write EOF: %s\n",strerror(errno));
		SOS_RETURN(-1);
	      }
	  case 3:		/* INTR */
	    numchar = 0;
	    line->str[numchar] = '\0';
	    SOS_RETURN(0);
	  case 8:		/* Delete */
	  case 127:
	    if (numchar > 0)
	      {
		if (out && write(fileno(out), delseq.str, delseq.len) != delseq.len)
		  {
		    sos_error_printf("Could not write delete sequence: %s\n",strerror(errno));
		    SOS_RETURN(-1);
		  }
		numchar--;
	      }
	    dooutput = 0;
	    continue;
	  case 21:		/* ^U kill line */
	    for (; numchar > 0; numchar--)
	      if (out && write(fileno(out), delseq.str, delseq.len) != delseq.len)
		{
		  sos_error_printf("Could not write delete sequence: %s\n",strerror(errno));
		  SOS_RETURN(-1);
		}
	    numchar = 0;
	    continue;
	  }

      /*
       * Check for protocol escape sequences
       *
       * Currently, nothing uses this, I believe
       */
      if (opts->escape_mode)
	if ((int)(unsigned char)input == opts->escape_char)
	  {
	    sos_debug_printf_and(2, "  Is a escape mode");
	    if ((ret = (*opts->escape_mode) (in, out, (unsigned char)input)) < 0)
	      {
		sos_error_printf("Escape mode requested error\n");
		SOS_RETURN (-1);
	      }
	    sos_debug_printf_and(2, "  Out of escape mode with %d",ret);
	    if (ret == 0)		/* Don't do anything */
	      continue;
	    if (ret > 0)
	      input = ret;	/* Insert this character */
	  }

      /* Add to input line */
      line->str[numchar++] = input;

      /* What kind of output options we have */
      if (dooutput)
	{
	  unsigned char echochar = (unsigned char)opts->echo;

	  ret = 1;
	  if (opts->echo == 0)	/* echo the input */
	    ret = write(fileno(out), &input, 1);
	  if (opts->echo > 0)	/* echo the specified character */
	    ret = write(fileno(out), &echochar, 1);

	  if (ret != 1)		/* echo naught */
	    {
	      sos_error_printf("Could not echo character: %s\n",strerror(errno));
	      SOS_RETURN(ret);
	    }
	}
    }

  line->str[numchar] = '\0';

  /*
   * Under certain circumstances (i.e. it makes sense)
   * print the first EOL sequence to make sure we have
   * performed a CRLF (or whatever)
   */
  if (opts->echo > 0 || (opts->echo == 0 && eol_termstate[0] != eol_state[0]))
    if (write(fileno(out), opts->eolseq[0].str, eol_termstate[0]) != eol_termstate[0])
      {
	sos_error_printf("Could not write EOL seq: %s\n",strerror(errno));
	SOS_RETURN (-1);
      }

  /* We are exiting with errors (usually timeouts or line too long) */
  if (!exitstate)
    {
      /* Alert (ring bell) that we don't like what is happening */
      if (opts->wantalert)
	if (write(fileno(out), "\a", 3) != 3)
	  {
	    sos_error_printf("Could not write alert seq: %s\n",strerror(errno));
	    SOS_RETURN (-1);
	  }

      /* XXX - return number of characters encoded to avoid -1 */
      SOS_RETURN (-numchar - 1);
    }

  /* return number of characters read */
  SOS_RETURN (numchar);
}



/*
 * Standard readline, but setup a signal handler
 * to handle timeouts
 */
int
sos_readline_timeout(FILE * in, FILE * out, sos_string * line, struct sos_readline_t *opts, struct sigaction *readtimeout, int timeout)
{
  SOS_ENTRY("sos_io","sos_readline_timeout",NULL);
  struct sigaction old_sigaction;
  int ret_val;
  struct itimerval newit,oldit;

  if (timeout <= 0)
    {
      sos_error_printf("Could not setsid: %s\n",strerror(errno));
      SOS_RETURN(-1);
    }

  sigaction(SIGALRM, readtimeout, (struct sigaction *)&old_sigaction);
  newit.it_interval.tv_sec = newit.it_interval.tv_usec = 0;
  newit.it_value.tv_sec = timeout; newit.it_value.tv_usec = 0;
  setitimer(ITIMER_REAL, &newit, &oldit);


  ret_val = sos_readline(in, out, line, opts);

  setitimer(ITIMER_REAL, &oldit, NULL);
  sigaction(SIGALRM, (struct sigaction *)&old_sigaction, NULL);

  SOS_RETURN(ret_val);
}



/*
 * flush any data pending on a FD
 *
 * Returns number of bytes flushed (or -1 for error)
 */
int sos_io_flush(int fd)
{
  SOS_ENTRY("sos_io","sos_io_flush",NULL);
  char tmp[512];
  long available;
  int ret;
  int sum = 0;

  while (((ret = ioctl(fd, FIONREAD, &available)) == 0) && available)
    {
      ret = read(fd, tmp, 512);	/* Should be NBIO */
      if (ret < 0)
	{
	  sos_error_printf("Could not read data which FIONREAD said was there: %s\n",strerror(errno));
	  SOS_RETURN(-1);
	}
      sum += ret;
    }

  SOS_RETURN(sum);
}



/*
 * Wait for data to show up on fd, with timeout
 */
int sos_wait_input(int fd, struct timeval *timeout)
{
  SOS_ENTRY("sos_io","sos_wait_input",NULL);
  fd_set rfds;

  FD_ZERO(&rfds);
  FD_SET(fd,&rfds);

  SOS_RETURN(select(fd+1,&rfds,NULL,NULL,timeout));
}



/*
 * Wait for data to ready for fd output, with timeout
 */
int sos_wait_output(int fd, struct timeval *timeout)
{
  SOS_ENTRY("sos_io","sos_wait_output",NULL);
  fd_set wfds;

  FD_ZERO(&wfds);
  FD_SET(fd,&wfds);

  SOS_RETURN(select(fd+1,NULL,&wfds,NULL,timeout));
}
