#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER
 * DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER
 * DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER
 *
 * SOS did not develop the *TT800* routines.  The underlying routines
 * written by M. Matsumoto <matumoto@math.keio.ac.jp>
 *
 * Additionally, the concept of truerand was borrowed from AT&T Bell
 * Labs, and the PMMCG concept was borrowed from some ACM article.
 *
 * DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER
 * DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER
 * DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER DISCLAIMER
 */

/*
 * Everything dealing with randomness
 *
 * The security of our random numbers are based upon (in order of importance):
 *
 * The cryptographical security and diffusion properties of our hash
 * function (currently MD5).
 *
 * The granularity of the system clock (*truerand* stuff)
 *
 * The prime modulus multiplicative congruential generator
 *
 *
 * Note: The private and semi-private routines are not in the sos_fun trace
 * because of speed concerns and to help hide timing information.  This should
 * be pondered upon...
 */

#include "sos.h"
#include "md5.h"


/* Constants for prime modulus multiplicative congruential generator */
#define RNG_M 2147483647L  /* m = 2^31 - 1 */
#define RNG_A 16807L
#define RNG_Q 127773L      /* m div a */
#define RNG_R 2836L        /* m mod a */

/* Num words before you must refresh random pool */
#define REFRESH_POOL_LIMIT	4096
#define ADDRANDS_PER_REFRESH	4


static sos_buffer privpool;
static char statpool[16];
static unsigned volatile int sos_truerand_interrupted=0;
static unsigned volatile int sos_truerand_count = 0;

static void sos_truerand_interrupt(int sig);
static void sos_truerand_addrand(sos_buffer *rand);
static void sos_truerand_refresh(sos_buffer *rand);



/*
 * Generate random numbers
 *
 * len must be multiple of 4 bytes
 */
int
sos_get_rand(unsigned char *buf, int len)
{
  SOS_ENTRY("sos_getrand","sos_get_rand",NULL);
  int c;
  int intlen = sizeof(int);
  char *origbuf = NULL;
  int origlen = 0;
  char minbuf[4];

  if (!buf || len < 1)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  /* If buffer is smaller than 4 bytes, round up :-) */
  if (len < 4)
    {
      origlen = len;
      origbuf = buf;
      buf = minbuf;
      len = 4;
    }

  if (len % 4)
    {
      sos_error_printf("Length of random number buffer must be a multiple of four (%d)\n",len);
      SOS_RETURN(-1);
    }

  /* Throw data into buf a word at a time */
  for (c = 0; c < len; c += intlen)
    {
      *((unsigned int *)(buf + c)) = sos_get_intrand();
    }

  if (origlen)
    {
      memcpy(origbuf, buf, origlen);
    }

  SOS_RETURN(len / intlen);
}



/*
 * Return a random integer based on pool
 * update pool to keep things flowing
 */
unsigned int
sos_get_intrand(void)
{
  SOS_ENTRY("sos_getrand","sos_get_intrand",NULL);
  static signed short poolexpiry = REFRESH_POOL_LIMIT;
  static unsigned char inited = 0;
  struct timeval beforetime, aftertime;
  unsigned int retval = 0;
  unsigned int new[4];
  int count;
  MD5_CTX context;

  gettimeofday(&beforetime, NULL);

  if (!inited)
    {
      inited = 1;
      privpool.str = statpool;
      privpool.len = sizeof(statpool);
    }

  /* Update random pool every once in a while */
  if (++poolexpiry > REFRESH_POOL_LIMIT)
    {
      poolexpiry = 0;
      sos_truerand_refresh(&privpool);
    }

  /* Our returned random number is first word in privpool */
  memcpy((char *)&retval,privpool.str,sizeof(retval));

  /*
   * Use prime modulus multiplicative congruential generator
   * to generate new pseudo-random data for MD5 to munge
   * into current pool
   */
  for(count=0;count<4;count++)
    {
      new[count] = sos_rand_pmmcg(*(((int *)privpool.str)+count));
    }

  gettimeofday(&aftertime, NULL);

  /* Stir the pot */
  MD5Init(&context);
  MD5Update(&context, privpool.str, privpool.len);
  MD5Update(&context, (char *)new, sizeof(new));
  MD5Update(&context, (char *)&beforetime, sizeof(beforetime));
  MD5Update(&context, (char *)&aftertime, sizeof(aftertime));
  MD5Final(&context);

  /* Update the pool */
  memcpy(privpool.str,&(context.digest[0]), 16);

  SOS_RETURN(retval);
}



/*
 * prime modulus multiplicative congruential generator
 * with period 2^31 - 1 (not very important)
 * and range [1 - 2^31 - 1] (important)
 */
unsigned int
sos_rand_pmmcg(unsigned int seed)
{
  register long low, high, test;

  /* Normalize to between 1 and m-1 */
  seed = (seed % (RNG_M - 1)) + 1;

  high = (seed) / RNG_Q;
  low = (seed) % RNG_Q;
  test = RNG_A * low - RNG_R * high;
  if (test > 0)
    (seed) = test;
  else
    (seed) = test + RNG_M;

  return(seed);
}



/*
 * Schedule an alarm call in a very
 * short time (hopefully the
 * granularity of the system clock)
 *
 * XXX - should this have sos_fun?
 */
static void
sos_truerand_set_alarm(void)
{
  struct itimerval next;
  unsigned int tps;

  if ((tps = sysconf(_SC_CLK_TCK)) < 1)
    {
      tps = 60;
    }

  tps = 1000000 / tps;		/* usec / (tick/sec) == usecs/tick */
  tps--;			/* Best value for randomness */

  timerclear(&next.it_interval);
  next.it_value.tv_sec = 0;
  next.it_value.tv_usec = tps;
  if (setitimer(ITIMER_REAL, &next, NULL) < 0)
    {
      /* Something drastic */
    }
}



/*
 * Interrupt handler for tick alarm
 *
 * XXX - should this have sos_fun?
 */
static void
sos_truerand_interrupt(int sig)
{
  if (sos_truerand_count)
    {
      sos_truerand_interrupted = 1;
      return;
    }

  /* Got interrupted before we counted anything.  Hmm. */
  (void)sos_signal_intr(SIGALRM, sos_truerand_interrupt);
  sos_truerand_set_alarm();
}



/*
 * Add ``some'' (hopefully enough) bits of random data into the random
 * pool
 *
 * (Somewhere between 8 and 14 bits, from tests, from sos_truerand_count
 * so assume 8 bits to be conservative)
 *
 * XXX - should this have sos_fun?
 */
static void
sos_truerand_addrand(sos_buffer *rand)
{
  MD5_CTX context;
  struct timeval beforetime, aftertime;
  sos_sigfunc *oldsig;
  struct itimerval olditimer;

  sos_truerand_interrupted = 0;
  sos_truerand_count = 0;

  gettimeofday(&beforetime, NULL);

  oldsig = sos_signal_intr(SIGALRM, sos_truerand_interrupt);
  getitimer(ITIMER_REAL, &olditimer);
  sos_truerand_set_alarm();

  /*
   * Get an int with ``some'' random data in it
   */
  while (!sos_truerand_interrupted)
    sos_truerand_count++;

  (void)sos_signal(SIGALRM, oldsig);
  setitimer(ITIMER_REAL, &olditimer, NULL);
  gettimeofday(&aftertime, NULL);


  /* Create the new pool based on the old one plus the new bits */
  MD5Init(&context);
  MD5Update(&context, rand->str, rand->len);
  MD5Update(&context, (char *)&sos_truerand_count, sizeof(sos_truerand_count));
  MD5Update(&context, (char *)&beforetime, sizeof(beforetime));
  MD5Update(&context, (char *)&aftertime, sizeof(aftertime));
  MD5Final(&context);

  /* Update the pool */
  memcpy(rand->str,&(context.digest[0]), 16);
}



/*
 * Refresh the pool of random data
 *
 * XXX - should this have sos_fun?
 */
static void
sos_truerand_refresh(sos_buffer *rand)
{
  int x;

  for(x=0;x<ADDRANDS_PER_REFRESH;x++) {
    sos_truerand_addrand(rand);
  }
}



/*
 * A C-program for TT800 : July 8th 1996 Version
 * by M. Matsumoto, email: matumoto@math.keio.ac.jp
 * genrand() generate one pseudorandom number with double precision
 * which is uniformly distributed on [0,1]-interval
 * for each call.  One may choose any initial 25 seeds
 * except all zeros.
 *
 * See: ACM Transactions on Modelling and Computer Simulation,
 * Vol. 4, No. 3, 1994, pages 254-266.
 *
 * TT800, a twisted GFSR generator proposed by Matsumoto and Kurita in
 * the ACM Transactions on Modelling and Computer Simulation, Vol. 4,
 * No. 3, 1994, pp. 254-266.  This 'little monster' has a period of
 * 2^800 - 1 and excellent equidistribution properties up to dimension
 * 25. The 'big monster', a TGFSR with a period of more than 2^11000,
 * is currently under construction by M. Matsumoto and T. Nishimura.
 */

/*
 * SOS has modified the code to allow you to get u_ints or doubles.
 * Additionally, you can call the initialization routine to load the
 * initial seeds with truerand data
 *
 * You can probably blame us for any errors
 */

#define N 25
#define M 7
static unsigned long seed[N] =
{ /* initial 25 seeds, change as you wish */
  0x95f24dab, 0x0b685215, 0xe76ccae7, 0xaf3ec239, 0x715fad23,
  0x24a590ad, 0x69e4b5ef, 0xbf456141, 0x96bc1b7b, 0xa7bdf825,
  0xc1de75b7, 0x8858a9c9, 0x2da87693, 0xb657f9dd, 0xffdc8a9f,
  0x8121da71, 0x8b823ecb, 0x885d05f5, 0x4e20cd47, 0x5a9ad5d9,
  0x512c0c03, 0xea857ccd, 0x4cc1d30f, 0x8891a8a1, 0xa6b7aadb
};


/*
 * Load the seed with truerand data
 */
int sos_rand_tt800i()
{
  SOS_ENTRY("sos_getrand","sos_rand_tt800i",NULL);
  int x;

  for(x=0;x<N;x++)
    seed[x] = sos_get_intrand();

  SOS_RETURN(0);
}



/*
 * Get a random u_int
 */
u_int sos_rand_tt800u()
{
  SOS_ENTRY("sos_getrand","sos_rand_tt800u",NULL);
  unsigned long y;
  static int k = 0;
  static unsigned long mag01[2] =
  { 
    0x0, 0x8ebfd028 /* this is magic vector `a', don't change */
  };

  if (k==N)
    {						/* generate N words at one time */
      int kk;

      for (kk=0;kk<N-M;kk++)
	{
	  seed[kk] = seed[kk+M] ^ (seed[kk] >> 1) ^ mag01[seed[kk] % 2];
	}

      for (; kk<N;kk++)
	{
	  seed[kk] = seed[kk+(M-N)] ^ (seed[kk] >> 1) ^ mag01[seed[kk] % 2];
	}
      k=0;
    }

  y = seed[k++];
  y ^= (y << 7) & 0x2b5b2500;			/* s and b, magic vectors */
  y ^= (y << 15) & 0xdb8b0000;			/* t and c, magic vectors */
#ifdef WORDSIZE64
  y &= 0xffffffff;				/* you may delete this line if word size = 32 */
#endif /* WORDSIZE64 */

  /*
   * the following line was added by Makoto Matsumoto in the 1996 version
   * to improve lower bit's corellation.
   * Delete this line to o use the code published in 1994.
   */
  y ^= (y >> 16); /* added to the 1994 version */

  SOS_RETURN(y);
}  



/*
 * Get a random double
 */
double sos_rand_tt800d()
{
  SOS_ENTRY("sos_getrand","sos_rand_tt800d",NULL);

  SOS_ORETURN((double)sos_rand_tt800u()/ (unsigned long) 0xffffffff);
}  
