#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * SOS General routines to perform host/service lookups
 * given any form of host/service identification
 * (e.g. name, number, etc)
 */

#include "sos.h"


/*
 * Get a hostent no matter what we are passed
 *
 * Requires DNS
 *
 * (Use getmcbyfoo for !DNS version, sort of)
 */
struct hostent *
sos_gethbyfoo(char *x)
{
  SOS_ENTRY("sos_gethbyfoo","sos_gethbyfoo",NULL);
  struct hostent *ret;

  if (!x)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  if (isdigit(*x))
    {				/* Might be a dotted quad */
      struct in_addr val;

      if (inet_aton(x, &val) == 1)
	{			/* Valid address */
	  ret = gethostbyaddr((char *)&(val.s_addr),sizeof(val.s_addr),AF_INET);

	  if (!ret)
	    {
	      sos_error_printf("Could not find hostent for name %s: %s\n",x,strerror(errno));
	    }

	  SOS_RETURN(ret);
	}
    }

  /* It is not an address--might be a hostname */
  ret = gethostbyname(x);

  if (!ret)
    {
      sos_error_printf("Could not find hostent for name %s: %s\n",x,strerror(errno));
    }

  SOS_RETURN(ret);
}



/*
 * Get a host address no matter what we are passed
 */
int 
sos_getabyfoo(char *x, struct in_addr *iaddr)
{
  SOS_ENTRY("sos_gethbyfoo","sos_getabyfoo", "Address Candidate: %s", x);
  struct hostent *ret;

  if (!iaddr || !x)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  /* If it is a raw IP address, we return it immediately */
  if (isdigit(*x))
    {				/* Might be a dotted quad */
      struct in_addr val;

      sos_debug_printf_and(128,"[%s] may be a dotted quad", x);

      if (inet_aton(x,&val) == 1)
	{			/* Valid address */
	  memcpy((char *)iaddr, (char *)&val, sizeof(val));
	  sos_debug_printf_and(128,"[%s] IS a dotted quad", x);
	  SOS_RETURN(0);
	}

      sos_debug_printf_and(128,"[%s] is NOT a dotted quad", x);

    }

  if ((ret = sos_gethbyfoo(x)) == NULL)
    {
      sos_error_printf("Could not find address for name %s\n",x);
      SOS_RETURN(-1);
    }

  /* Should we verify this somehow? */
  iaddr->s_addr = **(unsigned long **)ret->h_addr_list;

  SOS_RETURN(0);
}



/*
 * Get a make-conn type of hostent
 *
 * Like gethbyfoo, except that explicit IP addresses get
 * a fake hostent and do NOT go through DNS
 */
struct hostent *
sos_getmcbyfoo(char *x)
{
  SOS_ENTRY("sos_gethbyfoo","sos_getmcbyfoo", NULL);
  static struct hostent stat_host;
  static unsigned long stat_addr, *paddr[2];
  static char *halias[1];
  struct hostent *ret;

  if (!x)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  if (isdigit(*x))
    {				/* Might be a dotted quad */
      struct in_addr val;

      if (inet_aton(x, &val) == 1)
	{			/* Valid address */
	  stat_addr = val.s_addr;
	  paddr[0] = &(stat_addr);
	  paddr[1] = NULL;
	  halias[0] = NULL;
	  stat_host.h_name = x;
	  stat_host.h_aliases = halias;
	  stat_host.h_addrtype = AF_INET;
	  stat_host.h_length = sizeof(struct in_addr);
	  stat_host.h_addr_list = (char **)paddr;

	  SOS_RETURN(&stat_host);
	}
    }

  /* It is not an address--might be a hostname */
  ret = gethostbyname(x);

  if (!ret)
    {
      sos_error_printf("Could not find hostent for name %s: %s\n",x,strerror(errno));
    }

  SOS_RETURN(ret);
}



/*
 * Get a hostname no matter what we are passed
 */
char *sos_getnbyfoo(char *x, char *buf, int blen)
{
  SOS_ENTRY("sos_gethbyfoo","sos_getnbyfoo", NULL);
  struct hostent *ret;
  char *result;

  if (!x)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  if ((ret = sos_gethbyfoo(x)) == NULL)
    {
      /* If it is a raw IP address without a PTR, we return that */
      if (isdigit(*x))
	{			/* might be a dotted quad */
	  struct in_addr val;

	  if (inet_aton(x, &val) == 1)
	    {			/* Valid address */
	      result = x;
	      goto getnbyfoo_end;
	    }
	}
      sos_error_printf("Invalid host named %s: %s\n",x,strerror(errno));
      SOS_RETURN(NULL);
    }

  result = (char *)ret->h_name;

getnbyfoo_end:

  /* Should we verify this somehow? */
  if (!buf)
    SOS_RETURN(result);

  strncpy(buf,result,blen);
  SOS_RETURN(buf);
}



/*
 * Generate an SOS-style hostname (FQDN [q.u.a.d]) from a in_addr into a static buffer
 */
char *sos_getnbyaddr(struct in_addr in)
{
  SOS_ENTRY("sos_gethbyfoo","sos_getnbyaddr", NULL);
  struct hostent *ret;
  char *quad = inet_ntoa(in);
  static char sspace[MAXHOSTNAMELEN + 18 + 1];

  if ((ret = sos_gethbyfoo(quad)) == NULL)
    {
      /* No reverse DNS (sigh) */
      snprintf(sspace, (MAXHOSTNAMELEN + 18 + 1), "[%s]",quad);
    }
  else
    {
      snprintf(sspace,(MAXHOSTNAMELEN + 18 + 1), "%s [%s]",ret->h_name,quad);
    }
  SOS_RETURN(sspace);
}



/*
 * Get a service number no matter what we are passed
 */
int sos_getsbyfoo(char *x)
{
  SOS_ENTRY("sos_gethbyfoo","sos_getsbyfoo", NULL);
  struct servent *serv;
  int ret;

  if (!x)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  /* check for numeric ID */
  if (isdigit(*x)) 
    {
      char *stop;

      ret = (int)strtol(x,&stop,10);

      if (stop && *stop == '\0')
	SOS_RETURN(ret);
    }

  if (!(serv=getservbyname(x,"tcp")))
    {
      sos_error_printf("Could not get service name for %s: %s\n",x,strerror(errno));
      SOS_RETURN(-1);
    }

  /* Getservbyname returns port in network order */
  ret = SOS_NTOHS(serv->s_port);

  SOS_RETURN(ret);
}



/*
 * Get a udp service number no matter what we are passed
 */
int sos_getusbyfoo(char *x)
{
  SOS_ENTRY("sos_gethbyfoo","sos_getusbyfoo", NULL);
  struct servent *serv;
  int ret;

  if (!x)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  /* check for numeric ID */
  if (isdigit(*x)) 
    {
      char *stop;

      ret = (int)strtol(x,&stop,10);

      if (stop && *stop == '\0')
	SOS_RETURN(ret);
    }

  if (!(serv=getservbyname(x,"udp")))
    {
      sos_error_printf("Could not get service name for %s: %s\n",x,strerror(errno));
      SOS_RETURN(-1);
    }

  /* Getservbyname returns port in network order */
  ret = SOS_NTOHS(serv->s_port);

  SOS_RETURN(ret);
}



/*
 * Get hostname for peer
 *
 * Note:  sourceinfo.printhost is strdupped (mallocs memory)
 */
int sos_getpeername(int sourcefd, struct sos_conninfo *sourceinfo)
{
  SOS_ENTRY("sos_gethbyfoo","sos_getpeername", NULL);
  struct sockaddr name;
  int namelen = sizeof(name);
  
  if (!sourceinfo)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  sourceinfo->fd = sourcefd;
  sourceinfo->port = -1;

  if (getpeername(sourcefd, &name, &namelen) < 0)
    {
      sos_error_printf("Could not find peer for %d: %s\n",sourcefd,strerror(errno));
      SOS_RETURN(-1);
    }
  else
    {
      struct sockaddr_in my_inetaddr;

      if (name.sa_family != AF_INET)
	{
	  sos_error_printf("Not an Internet socket %d\n",sourcefd,strerror(errno));
	  SOS_RETURN(-1);
	}

      memcpy((char *)&my_inetaddr, (char *)&name, sizeof(my_inetaddr));

      sourceinfo->port = SOS_NTOHS(my_inetaddr.sin_port);
      memcpy((char *)&(sourceinfo->addr),(char *)&(my_inetaddr.sin_addr),sizeof(struct in_addr));

      sourceinfo->printhost = strdup(sos_getnbyaddr(my_inetaddr.sin_addr));
    }

  SOS_RETURN(0);
}



/*
 * Make a struct sos_conninfo from an arbitrary string (uses sos_gethbyfoo
 * as an intermediary. Returns *first* address in the list (UGGH!!!). 
 *
 * Mallocs memory (strdup)
 */
int 
sos_makeprinthost(int fd, 
		  int port, 
		  char *hoststr, 
		  struct sos_conninfo *conninfo)
{
  SOS_ENTRY("sos_gethbyfoo","sos_makeprinthost", NULL);
  struct hostent *h;		/* Intermediary form of address. */
  char *tmp;

  if (!hoststr || !conninfo)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if ( (h = sos_gethbyfoo(hoststr)) ==  (struct hostent *)NULL)
    {
      struct in_addr val;
      if (isdigit(*hoststr))
	{			/* Might be a dotted quad */
	  if (inet_aton(hoststr, &val) == 1)
	    {			/* Valid address */
	      memcpy((char *)&(conninfo->addr), 
		     (char *)&(val.s_addr),
		     sizeof(sizeof(struct in_addr)));
	    }
	  else
	    {			/* Not a valid dotted quad */
	      sos_error_printf("Not a valid name: %s\n",hoststr);
	      SOS_RETURN(-1);
	    }
	}
      else
	{			/* Not a dotted quad */
	  sos_error_printf("Not a valid name: %s\n",hoststr);
	  SOS_RETURN(-1);
	}
    }
  else
    {
      memcpy((char *)&(conninfo->addr), 
	     h->h_addr_list[0],
	     sizeof(struct in_addr));
    }

  memset(conninfo, (char)0, sizeof(conninfo));

  conninfo->fd = fd;
  conninfo->port = port;

  if (!(tmp = sos_getnbyaddr(conninfo->addr)))
    {
      sos_error_printf("Could not get address--strange--gethbyfoo worked\n");
      SOS_RETURN(-1);
    }

  if (!(conninfo->printhost = strdup(tmp)))
    {
      sos_error_printf("Could not duplicate string %s: %s\n",tmp, strerror(errno));
      SOS_RETURN(-1);
    }

  SOS_RETURN(0);
}



/*
 * Make a struct sos_conninfo from a sockaddr
 *
 * Mallocs memory (strdup)
 */
int 
sos_makeprinthostfromaddr(int fd, 
		  int port, 
		  struct sockaddr_in *addr,
		  struct sos_conninfo *conninfo)
{
  SOS_ENTRY("sos_gethbyfoo","sos_makeprinthostfromaddr", NULL);
  char *tmp;

  if (!addr || !conninfo)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }
  
  memset(conninfo, (char)0, sizeof(conninfo));

  conninfo->fd = fd;
  conninfo->port = port;
  memcpy((char *)&(conninfo->addr), 
	 (char *)&(addr->sin_addr),
	 sizeof(addr->sin_addr));

  if (!(tmp = sos_getnbyaddr(conninfo->addr)))
    {
      sos_error_printf("Could not get name for supplied address\n");
      SOS_RETURN(-1);
    }

  if (!(conninfo->printhost = strdup(tmp)))
    {
      sos_error_printf("Could not duplicate string %s: %s\n",tmp,strerror(errno));
      SOS_RETURN(-1);
    }

  SOS_RETURN(0);
}
