#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * SOS function functions
 *
 * Routines to help make programming and more especially diagnostics
 * by allow function tracing and debug level setting
 *
 * Debug levels:
 *
 * 1 - Function entry
 * 3 - Function entry--extended data
 * 4 - Function exit tracing
 * 8 - onreturn tracing
 * 16 - upbug tracing
 * 32 - Errors (we cannot use sos_error)
 */

#include "sos.h"


struct sos_fun_stuff *sos_cur_fun = NULL;
static struct sos_fun_stuff *saved_fun = NULL;

struct fun_callback
{
  void (*fn)(void *args);
  void *args;
};

static dict_h fun_list = NULL;
static int lots_of_fun = 1;


/*
 * Disable FUN (typically for performance reasons)
 */
void sos_no_fun(void)
{
  lots_of_fun = 0;
  saved_fun = sos_cur_fun;
  sos_cur_fun = NULL;
}



/*
 * Enable FUN (in case it was disabled somehow)
 */
void sos_more_fun(void)
{
  lots_of_fun = 1;
  if (!sos_cur_fun && saved_fun)
    sos_cur_fun = saved_fun;
  saved_fun = NULL;
}



/*
 * Function list initialization
 */
static void sos_fun_init()
{
  if ((fun_list = dll_create(NULL, NULL, DICT_UNORDERED, NULL)) == NULL)
    {
      sos_debug_printf_and(32,"Cannot create SOS function list via dll_create in sos_fun_init()\n");
    }
}



/*
 * Function to be run EVERY TIME you enter a function you wish to know
 * about (e.g. almost any function imaginable).  Implements a 
 * ``sos_stack'' to allow for debugging.  Also, sets current function's
 * debugging level.
 */
sos_fun_handle sos_fun_entry(char *package, char *funname, char *fmt, ...)
{
  va_list args;
  struct sos_fun_stuff *cur;
  char outbuf[1024];
  int fun_debug = sos_debug_query("sos_fun","sos_fun_entry");

  if (!lots_of_fun) return(NULL);

  if (!package || !funname)
    {
      sos_debug_printf_and(32,"sos_fun_entry called with invalid arguments\n");
      return(NULL);
    }

  if (!fun_list) sos_fun_init();

  if ((cur = (struct sos_fun_stuff *)malloc(sizeof(struct sos_fun_stuff))) == NULL)
    {
     sos_debug_printf_and(32,"Cannot malloc %d bytes in sos_fun_entry\n",sizeof(struct sos_fun_stuff));
      return(NULL);
    }


  cur->package = strdup(package);
  cur->funname = strdup(funname);
  cur->other = NULL;
  cur->onreturns = NULL;

  if (fmt)
    {
      va_start(args, fmt);
      vsnprintf(outbuf, 1024, fmt, args);
      va_end(args);
      cur->other = strdup(outbuf);
    }

  cur->debuglevel = sos_debug_query(package, funname);

  if (dll_insert(fun_list, cur) != DICT_OK)
    {
      sos_debug_printf_and(32,"Could not insert current sos_fun_stuff into fun_list\n");
      free(cur);
      return(NULL);
    }
  sos_cur_fun = cur;

  sos_debug_printf_cand(1, fun_debug, "-- Entering function %s in the %s package",cur->funname,cur->package);
  sos_debug_printf_cand(3, fun_debug, "   Debug level: %d  Entry information: %s",cur->debuglevel,cur->other?cur->other:"(null)");

  return ((void *)cur);
}


/*
 * Update the stack of function's notions of debugging levels
 * (appropriate only if you change debugging and want it reflected)
 */
void sos_fun_upbug(void)
{
  dict_obj cur;
  struct sos_fun_stuff *fs;
  int fun_debug = sos_debug_query("sos_fun","sos_fun_upbug");

  if (!lots_of_fun) return;

  sos_debug_printf_cand(16, fun_debug, "Entering fun_upbug (%x/%x)\n",fun_list,sos_cur_fun);

  if (!fun_list) return;

  for(cur=dll_maximum(fun_list);cur;cur=dll_predecessor(fun_list, cur))
    {
      fs=cur;

      fs->debuglevel = sos_debug_query(fs->package, fs->funname);
      sos_debug_printf_cand(16, fun_debug, "Updating debugging level of %s/%s to %d\n",fs->package,fs->funname,fs->debuglevel);
    }
}


/*
 * Add a callback to run a function (usually a cleanup function)
 * when sos_fun_exit() indicates that this function is about to
 * exit.  Note that the return() has not yet run.
 */
int sos_fun_onreturn(void (*fn)(void *args), void *args)
{
  int fun_debug = sos_debug_query("sos_fun","sos_fun_onreturn");
  struct fun_callback *fc;

  if (!lots_of_fun) return(0);

  if (!fn)
    {
      sos_debug_printf_and(32,"sos_fun_onreturn called with invalid arguments\n");
      return(-1);
    }

  if (!sos_cur_fun)
    {
      sos_debug_printf_and(32,"Cannot add function to onreturn list for current function--sos_cur_fun is NULL\n");
      return(-1);
    }

  if (!sos_cur_fun->onreturns)
    {
      if ((sos_cur_fun->onreturns = dll_create(NULL, NULL, DICT_UNORDERED, NULL)) == NULL)
	{
	  sos_debug_printf_and(32,"Cannot create SOS function callbacks via dll_create in sos_fun_onreturn()\n");
	  return(-1);
	}
    }

  if ((fc = (struct fun_callback *)malloc(sizeof(struct fun_callback))) == NULL)
    {
      sos_debug_printf_and(32,"Cannot malloc %d bytes in sos_fun_onreturn\n");
      return(-1);
    }
  fc->fn = fn;
  fc->args = args;

  if (dll_insert(sos_cur_fun->onreturns, fc) != DICT_OK)
    {
      sos_debug_printf_and(32,"Could not insert current onreturn into curfun list\n");
      free(fc);
      return(-1);
    }

  sos_debug_printf_cand(8, fun_debug, "-- Adding function %x(%x) as an exit function for %s",fn,args,sos_cur_fun->funname);

  return(0);
}



/*
 * Indicate that the function is about ready to exit.  Run
 * any scheduled callbacks.  Clean up ``sos-stack''.
 */
void sos_fun_exit(sos_fun_handle fh)
{
  int fun_debug = sos_debug_query("sos_fun","sos_fun_exit");
  dict_obj cur;
  int keepgoing;

  if (!lots_of_fun) return;

  if (!fh || !fun_list) return;		/* Stupid users */

  /* Check to see if this handle is in our queue */
  for(cur=dll_minimum(fun_list);cur && cur!=fh;cur=dll_successor(fun_list, cur))
    ;

  if (!cur)
    {
      sos_debug_printf_and(32,"Could not find function handle %x\n",fh);
      return;
    }

  /* Delete all elements on queue up to and including the one we are interested in */
  keepgoing = 1;
  for(cur=dll_minimum(fun_list);cur && keepgoing;cur=dll_minimum(fun_list))
    {
      struct sos_fun_stuff *fs = (struct sos_fun_stuff *)cur;

      sos_cur_fun = cur;	/* Keep message debugging prints with accurate info */

      if (cur==fh)
	keepgoing = 0;

      /* Execute any callbacks that were scheduled */
      if (fs->onreturns)
	{
	  dict_obj curret;
	  for(curret=dll_maximum(fs->onreturns);curret;curret=dll_maximum(fs->onreturns))
	    {
	      struct fun_callback *fc = (struct fun_callback *)curret;

	      sos_debug_printf_cand(8, fun_debug, "-- Executing %x(%x) as an exit function for %s",fc->fn,fc->args,fs->funname);

	      (*fc->fn)(fc->args);
	      dll_delete(fs->onreturns, curret);
	    }
	  dll_destroy(fs->onreturns);
	}

      if (cur == fh)
	sos_debug_printf_cand(4, fun_debug, "-- Explicitly exiting function %s in package %s",fs->funname,fs->package);
      else
	sos_debug_printf_cand(4, fun_debug, "-- Automatically exiting function %s in package %s",fs->funname,fs->package);

      free(fs->package);
      free(fs->funname);
      if (fs->other) free(fs->other);

      dll_delete(fun_list, fs);
      free(fs);
    }

  sos_cur_fun = dll_minimum(fun_list);

  return;
}



/*
 * Figure out what function package and name we are in
 * (whatever the depth most recent call to _entry says)
 */
char *sos_fun_current(int depth, char **package, char **name)
{
  int fun_debug = sos_debug_query("sos_fun","sos_fun_current");
  dict_obj cur;
  struct sos_fun_stuff *fs;

  if (!lots_of_fun) return(NULL);

  if (!fun_list)
    return(NULL);

  if (depth == 0)
    {
      if (!sos_cur_fun)
	{
	  *package = NULL;
	  *name = NULL;
	  return(NULL);
	}

      if (package) *package = sos_cur_fun->package;
      if (name) *name = sos_cur_fun->funname;
      return(sos_cur_fun->funname);
    }

  for(cur = dll_minimum(fun_list);cur && depth;cur=dll_successor(fun_list, cur),depth--) ;

  if (cur)
    {
      fs = cur;

      if (package) *package = fs->package;
      if (name) *name = fs->funname;
      return(fs->funname);
    }

  *package = NULL;
  *name = NULL;
  return(NULL);
}



/*
 * Print out the current function trace.  If verbose
 * print out associated data and flags in long nice format.
 */
void sos_fun_ftrace(FILE *out, int verbose)
{
  int fun_debug = sos_debug_query("sos_fun","sos_fun_ftrace");
  dict_obj cur;
  struct sos_fun_stuff *fs;

  if (!lots_of_fun) return;

  if (!fun_list) sos_fun_init();

  if (verbose)
    fprintf(out,"\n\nVerbose function trace:\n----------------------------------------------------------------------\n");
  else
    fprintf(out,"Function trace:  ");

  for(cur=dll_maximum(fun_list);cur;cur=dll_predecessor(fun_list, cur))
    {
      fs=cur;
      if (verbose)
	{
	  fprintf(out, "Function %s in package %s\n",fs->funname,fs->package);
	  fprintf(out, "  Debug level: %d  Entry information: %s",fs->debuglevel,fs->other?fs->other:"(null)");
	  if (fs->onreturns)
	    {
	      dict_obj curret;

	      fprintf(out, "  On-return callbacks:\n");

	      for(curret=dll_maximum(fs->onreturns);curret;curret=dll_maximum(fs->onreturns))
		{
		  struct fun_callback *fc = (struct fun_callback *)curret;

		  fprintf(out, "    Function %x(%x)\n",fc->fn,fc->args);
		}
	    }
	  fprintf(out,"--------------------------------------------------\n");
	}
      else
	{
	  fprintf(out, "%s ",fs->funname);
	}
    }

  if (verbose)
    fprintf(out,"----------------------------------------------------------------------\n\n");
  else
    fprintf(out,"\n");

  fflush(out);

  return;
}



/*
 * Return a string containing stack trace.  If buf is supplied,
 * use storage there, otherwise use static space.
 */
char *sos_fun_strace(sos_string *sbuf)
{
  int fun_debug = sos_debug_query("sos_fun","sos_fun_strace");
  char staticbuf[2048];
  char *buf = staticbuf;
  int buflen = 2048;
  char *curstart = staticbuf;
  int curlen = 0;
  dict_obj cur;
  struct sos_fun_stuff *fs;

  if (!lots_of_fun) return(NULL);

  if (sbuf)
    {
      curstart = buf = sbuf->str;
      buflen = sbuf->len;
    }

  if (!fun_list) sos_fun_init();

  snprintf(curstart,(buflen-curlen),"Function trace: ");
  curlen = strlen(buf);
  curstart = buf+curlen;

  for(cur=dll_maximum(fun_list);cur;cur=dll_predecessor(fun_list, cur))
    {
      fs=cur;

      snprintf(curstart,(buflen-curlen)," %s",fs->funname);
      curlen = strlen(buf);
      curstart = buf+curlen;
    }

  return(buf);
}
