#if !defined(lint) && !defined(__INSIGHT__)
static char sos__rcsid[] = "$Id$";
static char sos__copyright[] = "Copyright (c) 1994, 1995, 1996 SOS Corporation";
static char sos__contact[] = "SOS Corporation <sos-info@soscorp.com> +1 800 SOS UNIX";
#endif /* not lint */

/*
 * ++Copyright Released Product++
 *
 * Copyright (c) 1994, 1995, 1996 Sources of Supply Corporation ("SOS").
 * All rights reserved.
 *
 * The SOS Released Product License Agreement specifies the terms and
 * conditions for redistribution.  You may find the License Agreement
 * in the file LICENSE.
 *
 * SOS Corporation
 * 461 5th Ave.; 16th floor
 * New York, NY 10017
 *
 * +1 800 SOS UNIX
 * <sos-info@soscorp.com>
 *
 * --Copyright Released Product--
 */

/*
 * General config-file reader.
 */

#include "sos.h"

static sos_config_key_t *sos_config_allocate_key(void);
static sos_config_value_t *sos_config_allocate_value(void);
static int sos_config_insert_key(sos_config_t config, sos_config_key key);
static sos_config_key_t *sos_config_find_key(sos_config_t config, sos_config_key key);
static int sos_config_remove_key(sos_config_t config, sos_config_key key);

/*
 * These defs should remain local to the config module.
 */

/* #define CONFIGBUFSIZ (MIN((MAXPATHLEN+2), 2048)) */
#define CONFIGBUFSIZ (MAXPATHLEN+2)
static char config_buf[CONFIGBUFSIZ];

/* I like these definitions (more accurate than 0 and 1) */
#ifndef FALSE
# define FALSE 0
#endif /* !FALSE */
#ifndef TRUE
# define TRUE (!FALSE)
#endif /* TRUE */


/*
 * Dynamically allocate and zero key structure.
 */
static sos_config_key_t *
sos_config_allocate_key(void)
{
  SOS_ENTRY("sos_config","sos_config_allocate_key",NULL);
  sos_config_key_t *new;

  if (!(new = (sos_config_key_t *) malloc(sizeof(sos_config_key_t))))
    {
      sos_error_printf("Could not malloc %d bytes for key structure: %s\n",sizeof(sos_config_key_t),strerror(errno));
      SOS_RETURN(NULL);
    }

  new->key = (char *)NULL;
  new->current = (sos_config_value_t *) NULL;
  new->first = (sos_config_value_t *) NULL;
  new->last = (sos_config_value_t *) NULL;
  new->no_direction_yet = TRUE;
  new->num_values = 0;

  new->next = (sos_config_key_t *) NULL;
  new->prev = (sos_config_key_t *) NULL;

  SOS_RETURN(new);
}



/* 
 * Dynamically allocate and zero value structure.
 */
static sos_config_value_t *
sos_config_allocate_value(void)
{
  SOS_ENTRY("sos_config","sos_config_allocate_value",NULL);
  sos_config_value_t *new;

  if (!(new = (sos_config_value_t *) malloc(sizeof(sos_config_value_t))))
    {
      sos_error_printf("Could not malloc %d bytes for value structure: %s\n",sizeof(sos_config_value_t), strerror(errno));
      SOS_RETURN(NULL);
    }

  new->value = (char *)NULL;

  new->next = (sos_config_value_t *) NULL;
  new->prev = (sos_config_value_t *) NULL;

  SOS_RETURN(new);
}



/*
 * Add key into config file structure
 */
static int sos_config_insert_key(sos_config_t config, sos_config_key key)
{
  SOS_ENTRY("sos_config","sos_config_insert_key",NULL);
  sos_config_key_t *new;

  if (!key || !config)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if (!(new = sos_config_allocate_key()))
    {
      sos_error_printf("Could not allocate key structure\n");
      SOS_RETURN(-1);
    }

  if (!(new->key = (sos_config_key) malloc(strlen(key) + 1)))
    {
      sos_error_printf("Could not allocate value structure\n");
      SOS_RETURN(-1);
    }

  memcpy(new->key, key, strlen(key) + 1);

  /* 
   * Link it up to the key list (FIFO).
   */
  if ( config->key_last )
    {
      config->key_last->next = new;
      new->prev=config->key_last;
      config->key_last = new;
    }
  else
    {
      /*
       * First in the list 
       */
      config->key_list = new;
      config->key_last = new;
    }

  SOS_RETURN(0);
}

/*
 * Delete key into config file structure
 */
static int sos_config_remove_key(sos_config_t config, sos_config_key key)
{
  SOS_ENTRY("sos_config","sos_config_remove_key",NULL);
  sos_config_key_t *old;

  if (!key || !config)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }


  if ( !(old = sos_config_find_key(config, key)) )
    {
      /*
       * If callee deleted a nonexistant key just return, no harm done.
       */
      SOS_RETURN(0);
    }
  
  if ( old == config->key_list )
    {
      /* First in list */
      config->key_list = old->next;
    }
  if ( old == config->key_last )
    {
      config->key_last = old->prev;
    }

  if ( old->prev )
    {
      old->prev->next=old->next;
      old->prev=NULL;
    }
  if ( old->next )
    {
      old->next->prev=old->prev;
      old->next=NULL;
    }
  
  free(old->key);
  free(old);
  old=NULL;
    
  SOS_RETURN(0);
}



/*
 * Look for a key in the config file structure
 */
static sos_config_key_t *
sos_config_find_key(sos_config_t config, sos_config_key key)
{
  SOS_ENTRY("sos_config","sos_config_find_key",NULL);
  sos_config_key_t *confkey=NULL;

  if (!key || !config)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  for (confkey=config->key_list; confkey; confkey = confkey->next)
    if (SOS_STREQ(confkey->key, key))
      break;

  SOS_RETURN(confkey);
}



/*
 * `config' is the main pointer to the whole kit and caboodle. 
 * 
 * Insert `value' into the config structure in the `key' bin.
 * If bin doesn't exist, insert it.
 *
 * return `config' since this function may well alter its value.
 */
sos_config_key_t *sos_config_insert_value(sos_config_t config,
			       const sos_config_key key,
			       const sos_config_value value)
{
  SOS_ENTRY("sos_config","sos_config_insert_value",NULL);
  sos_config_key_t *bin;	/* Keys act like hash bins and I need a ident */
  sos_config_value_t *new;	/* new value pointer */

  if (!key || !value || !config)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  if (!(bin = sos_config_find_key(config, key)))
    {
      if (sos_config_insert_key(config, key) < 0 ) 
	{
	  sos_error_printf("Could not insert key\n");
	  SOS_RETURN(NULL);
	}
      bin = config->key_last;
    }

  if (!(new = sos_config_allocate_value()))
    {
      sos_error_printf("Could not allocate value structure\n");
      SOS_RETURN(NULL);
    }

  if (!(new->value = (sos_config_value)strdup(value)))
    {
      sos_error_printf("Could not strdup %d bytes: %s\n",strlen(value)+1,strerror(errno));
      SOS_RETURN(NULL);
    }

  /*
   * Queue up new value structure. 
   */
  if (!bin->first)
    {
      /*
       * This bin has never had a value installed. Initialize the key structure
       * pointers to the new value. 
       */
      bin->current = new;
      bin->first = new;
      bin->last = new;
    }
  else
    {
      bin->last->next = new;
      new->prev = bin->last;
      bin->last = new;
    }

  bin->num_values++;		/* Count the values */

  SOS_RETURN(config->key_list);
}


/*
 * `config' is the main pointer to the whole kit and caboodle. 
 * 
 * Delete `value' from the config structure in the `key' bin.
 * If value is the only value, delete the key.
 *
 * return `config' since this function may well alter its value.
 * 
 * XXX How can we tel between an error and deleting the final key in the
 * list?? Sigh...
 */
sos_config_key_t *sos_config_delete_value(sos_config_t config,
			       const sos_config_key key,
			       const sos_config_value value)
{
  SOS_ENTRY("sos_config","sos_config_delete_value",NULL);
  sos_config_key_t *bin;	/* Keys act like hash bins and I need a ident */
  sos_config_value_t *old;	/* new value pointer */

  if (!key || !value || !config)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }


  /* 
   * We need to cache this sos_config_key_t so that we can adjust the
   * list.
   */
  if (!(bin = sos_config_find_key(config, key)))
    {
      sos_error_printf("Could not locate key\n");
      SOS_RETURN(config->key_list);
    }

  for (old = bin->first; old; old=old->next)
    {
      if ( SOS_STREQ(old->value,value)  )
	{
	  break;
	}   
    }

  if ( !old )
    {
      /*
       * Value not found, but that's OK. If the callee wished to delete a 
       * value that didn't exist, it and the data structures certainly 
       * agree on the current state.
       */
      SOS_RETURN(config->key_list);
    }
  
  /*
   * Dequeue old value.
   */
  if ( old->prev )
    {
      old->prev->next = old->next;
      old->prev=NULL;
    }
  else
    {
      /* 
       * This is first in the list. 
       */
      bin->first=old->next;
    }

  if ( old->next )
    {
      old->next->prev = old->prev;
      old->next = NULL;
    }
  else
    {
      /*
       * This is final in the list
       */
      bin->last=old->prev;
    }

  free(old->value);
  free(old);
  old=NULL;
      
  bin->num_values--;		/* Count the values */

  if ( bin->num_values == 0 )
    {
      sos_config_remove_key(config,key);
    }

  SOS_RETURN(config->key_list);
}



/* 
 * Open and read the specified config file <filename>
 * There is no intelligence for parsing out unknown keywords, only syntatic
 * errors can be caught
 *
 * ConfigH is the handle on the created dictionary.
 * Config handle is returned.
 */
sos_config_t
sos_config_read(char *filename)
{
  SOS_ENTRY("sos_config","sos_config_read",NULL);
  FILE *Config;			/* The config file we are opening */
  int config_line_count = 0;	/* Used for nice error messages */
  sos_config_t ConfigH = NULL;	/* Config file handle. Returned */
  sos_config_key_t *temp;	/* Used for good error checking */

  if (!filename || strlen(filename) > MAXPATHLEN)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  if ( ( ConfigH = (sos_config_t)malloc(sizeof *ConfigH)) == NULL )
    {
      sos_error_printf("Could not malloc config handle (%d bytes):%s\n",sizeof *ConfigH,strerror(errno));
      SOS_RETURN(NULL);
    }

  memset(ConfigH->filename, (char)0, MAXPATHLEN);
  ConfigH->key_list = NULL;
  ConfigH->key_last = NULL;
  
  snprintf(ConfigH->filename, MAXPATHLEN, "%s", filename);



  /*
   * Go ahead and open.
   */
  if ((Config = fopen(ConfigH->filename, "r")) == (FILE *) NULL)
    {
      sos_error_printf("Could not open config file %s: %s\n",ConfigH->filename,strerror(errno));
      SOS_RETURN(NULL);
    }

  memset(config_buf, (char)NULL, CONFIGBUFSIZ);

  while (fgets(config_buf, CONFIGBUFSIZ, Config))
    {
      char *value;		/* Will parse out the value portion of the input line */
      char *key;		/* Useless but makes the code slightly more readable */

      config_line_count++;	/* NB this will start the count at 1 */

#ifdef WANT_COMMENT_SYNTAX
      /* 
       * Strip comments and blank lines
       * Left margin-abutting white space indicates a comment.
       */
      if (*config_buf == '#' ||
	  *config_buf == ';' ||
	  strspn(config_buf, SOS_WHITESPACE) )
	continue;
#endif /* WANT_COMMENT_SYNTAX */

      sos_rip (config_buf);

      /*
       * Search for division between key and value
       */
      if ((value = strchr(config_buf, ' ')) == (char *)NULL)
	{
	  value = config_buf + strlen(config_buf);
	}
      else
	{
	  *value++ = '\0';
	}

      /*
       * Now it makes sense to use `key' (instead of config_buf)
       * (note key could be of length zero)
       */
      key = config_buf;

      /*
       * Go ahead and pass these values of `key' and `value'. They will
       * get copied to permenent storage by the insertion routines.
       */
      temp = sos_config_insert_value(ConfigH, key, value);
      if (temp && temp != ConfigH->key_list)
	/*
	 * Config structure got set/reset during this call so 
	 */
	ConfigH->key_list = temp;

    }				/* End of read loop */

  fclose(Config);

  SOS_RETURN(ConfigH);
}				/* End of sos_config_read() */




/* 
 * Get the next value of `key' from `config', moving in `direction', and
 * with the specified `semantic'
 * 
 * This function will oftern return NULL as normal, expected return value
 * (eg: if it is called with a LINEAR semantic and it already at the end 
 * of the list). NULL should *not* be considered an error.
 */
sos_config_value
sos_config_getnext(sos_config_t config,
		   sos_config_key key,
		   int direction,
		   int semantic)
{
  SOS_ENTRY("sos_config","sos_config_getnext",NULL);
  sos_config_key_t *bin;	/* Keys are like hash bins. Dont buy it do you? */
  sos_config_value_t *ret;	/* The structure containing the return value. */
  sos_config_key_t *config_key_list;
  
  if ( !config || !key)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  config_key_list = config->key_list;

  ret = (sos_config_value_t *) NULL;

  if (!(bin = sos_config_find_key(config, key)))
    {
      sos_error_printf("Could not find key \"%s\"\n",key);
      SOS_RETURN(NULL);
    }

  if (bin->no_direction_yet)
    {
      /*
       * Initialize the direction of the `value' traversal
       */
      if (direction == SOS_CONFIG_REVERSE)
	bin->current = bin->last;
      else
	/* 
	 * This heavily favors moving in the forward direction, but that's 
	 * probably what most folks want.
	 */
	bin->current = bin->first;

      bin->no_direction_yet = FALSE;
    }

  ret = bin->current;

  /* 
   * If callee asks for a static read, return before updating the `current'
   * pointer
   */
  if (semantic == SOS_CONFIG_STATIC)
    {
      if (ret)
	SOS_RETURN(ret->value);
      else
	SOS_RETURN(NULL);
    }

  /*
   * Update the `current' pointer based on the callee's desired direction
   * and semantic
   */
  if (ret)
    switch (direction)
      {
      case SOS_CONFIG_NULL:
	/* 
	 * If a client doesn't really know what it wants (thus uses NULL
	 * argument), advance the `current' pointer in the forward direction
	 */
      case SOS_CONFIG_FORWARD:
	bin->current = bin->current->next;
	break;

      case SOS_CONFIG_REVERSE:
	bin->current = bin->current->prev;
	break;

      default:
	break;
      }

  switch (semantic)
    {
    case SOS_CONFIG_NULL:
    case SOS_CONFIG_LINEAR:
      break;

    case SOS_CONFIG_AUTORESET:
      if (!ret)

    case SOS_CONFIG_CIRCULAR:
	if (!bin->current)
	  {
	    /* 
	     * If we have reached the end of the list and the callers wants
	     * circular sematics, reset the current appropriately
	     */
	    switch (direction)
	      {
	      case SOS_CONFIG_NULL:
	      case SOS_CONFIG_FORWARD:
		bin->current = bin->first;
		break;

	      case SOS_CONFIG_REVERSE:
		bin->current = bin->last;
		break;

	      default:
		break;
	      }
	  }
    }

  if (ret)
    SOS_RETURN(ret->value);
  else
    SOS_RETURN(NULL);
}



/* 
 * Print out the config structure
 */
void
sos_config_print(sos_config_t config)
{
  SOS_ENTRY("sos_config","sos_config_print",NULL);
  sos_config_value_t *value;
  sos_config_key_t *key_list;
  
  if (!config)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_VRETURN();
    }

  printf("************************************\n");
  printf("              CONFIG                \n");

  printf("filename: %s\n\t********\n", config->filename);

  for (key_list = config->key_list; key_list; key_list = key_list->next)
    {
      printf("%s: ", key_list->key);
      for (value = key_list->first; value; value = value->next)
	printf("**%s** ", value->value);
      printf("\n");
    }
  printf("************************************\n\n");
  SOS_VRETURN();
}



/*
 * Destroy a config file and all contents
 */
void 
sos_config_destroy_list(sos_config_t config)
{
  SOS_ENTRY("sos_config","sos_config_destroy_list",NULL);
  sos_config_key_t *k;
  sos_config_value_t *config_value;
  sos_config_key_t *key_list;
  sos_config_value_t *v;

  if (!config)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_VRETURN();
    }

  key_list = config->key_list;
  while (key_list)
    {
      memset(key_list->key, (char)0, strlen(key_list->key));
      free(key_list->key);

      config_value=key_list->first; 
      while ( config_value )
	{
	  memset (config_value->value, (char)0, strlen(config_value->value));
	  free(config_value->value);

	  v = config_value;
	  config_value = config_value->next;
	  free(v);
	}
      
      k = key_list;
      key_list = key_list->next;
      free(k);
    }
  
  memset(config->filename, (char)0, strlen(config->filename));
  free(config);
  SOS_VRETURN();
}



/*
 * Reload the config file pointed at by config from file name.
 * If filename is NULL, use the file name generated from the previous call
 * to sos_config_read.
 */
sos_config_t
sos_config_reload(sos_config_t config, char *filename)
{
  SOS_ENTRY("sos_config","sos_config_reload",NULL);
  char *f;
  int need_free=0;

  if (!config)
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  if ( filename )
    {
      f = filename;
    }
  else
    {
      if (!config->filename)
	{
	  sos_error_printf("No filename supplied and no filename saved in config file (how did that happen?)\n");
	  SOS_RETURN(NULL);
	}
      f = strdup(config->filename);
      need_free++;
    }
      
  if (config)
    {
      sos_config_destroy_list(config);
    }

  config = sos_config_read(f);

  if ( need_free ) 
    {
      memset(f, (char)0, strlen(f));
      free(f);
    }

  SOS_RETURN(config);
}


/* 
 * Write the configuration file to a file named by filename
 * If filename is NULL the file written will be that found in the config
 * struct. All writes are done to tmp file first and the tmpfile is renamed
 * to the new filename.
 */
int
sos_config_write_file (sos_config_t config, char *filename)
{
  SOS_ENTRY("sos_config","sos_config_write_file","filename: %s", filename);
  FILE *fp;			/* STREAM pointer to temp file */
  int fd;			/* File descriptor open on real file. */
  sos_config_value_t *value;
  sos_config_key_t *key_list;
  char buf[BUFSIZ];
  size_t nbytes;

  if ( !config )
    {
      sos_error_printf("Invalid argument\n");
      SOS_RETURN(-1);
    }

  if ( !filename )
    {
      if ( !config->filename )
	{
	  sos_error_printf("No filename supplied and no filename saved in config file (how did that happen?)\n");
	  SOS_RETURN(-1);
	}
      filename = config->filename;
    }

  
  /*
   * open filename subject to umask
   */
  if ( (fp=tmpfile()) == NULL )
    {
      sos_error_printf("Opening temporary  config file for write: %s\n", 
		       strerror(errno));
      SOS_RETURN (-1);
    }
  
  /* 
   * This scheme allows us to securely abort if something goes wrong
   * At this point there is nothing that can go wrong, so it's all a 
   * tad wasteful. But, hey, how often do you write out the config file?
   */
  for (key_list = config->key_list; key_list; key_list = key_list->next)
    {
      for (value = key_list->first; value; value = value->next)
	{
	  sos_debug_printf_and(64,"Writing out: %s %s\n", key_list->key, value->value);
	  fprintf(fp, "%s %s\n", key_list->key, value->value);
	}
    }

  if ((fd=open(filename, O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0 )
    {
      sos_error_printf("Error opening config file: %s\n", filename);
      SOS_RETURN(-1);
    }

  rewind(fp);
  while ( !feof(fp) )
    {
      /* 
       * And yes this works. You don't need to worry about the value of nbytes.
       */
      nbytes = fread(buf, 1, BUFSIZ, fp);
      sos_debug_printf_and(64,"Copying config file: %s\n", buf);
      if ( write (fd, buf, nbytes) != nbytes )
	{
	  sos_error_printf("Error copying temp file to config file \"%s\": %s\n",
			   filename, strerror(errno));
	}
    }
  close(fd);
  fclose (fp);

  SOS_RETURN (0);
}



/*
 * Delete a key and all its current values. Use with care :-) 
 * This routine is really ineffcient. But reuses code, so maintence is 
 * easier.
 */
int sos_config_delete_key(sos_config_t config, char *key)
{
  SOS_ENTRY("sos_config", "sos_config_delete_key", NULL);
  sos_config_value_t *value=NULL;
  sos_config_key_t *keyp=NULL;
  int ret=0;
  sos_config_value_t *old;

  if ( !config || ! key )
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(-1);
    }

  if ( (keyp=sos_config_find_key(config ,key)) == NULL)
    {
      /* If you can't find the key, so what? Caller wanted it deleted. */
      sos_debug_printf_and(64,"Could not find key: \"%s\" to delete");
      SOS_RETURN(0);
    }

  value = keyp->first;

  while (value)
    {
      old = value;
      value = value->next;
      free (old->value);
      free (old);
    }

  if ( sos_config_remove_key (config, key) < 0 )
    {
      SOS_RETURN(-1);
    }

  SOS_RETURN(0);
}



/*
 * sos_config_concat_key
 * Concatenate the values of a specified key using the supplied separator
 * Returns a private buffer which the user must free herself.
 * 
 * The caller may specify direction, but semantic is assumed linear.
 */
char *
sos_config_concat_key(sos_config_t config, sos_config_key key, char *separator,
		      int direction)
{
  SOS_ENTRY("sos_config","sos_config_allocate_key",NULL);
  static char *ret=NULL;
  sos_config_value value;
  int rlen=0;
  int vlen=0;
  int slen=0;

  if ( !config || !key )
    {
      sos_error_printf("Invalid arguments\n");
      SOS_RETURN(NULL);
    }

  if ( !(value=sos_config_getnext(config, key, direction, SOS_CONFIG_NULL)) )
    {
      SOS_RETURN(NULL);
    }

  slen=strlen(separator);
  
  if (ret)
    {
      free(ret);
      ret=NULL;
    }
  
  rlen=strlen(value);
  if ( !(ret=malloc(rlen)) )
    {
      sos_error_printf("Error allocating ret: %s", strerror(errno));
      SOS_RETURN(NULL);
    }

  strncpy(ret, value, rlen);

  while ( value=sos_config_getnext(config,key,direction, SOS_CONFIG_NULL))
    {
      vlen=strlen(value)+slen;
      
      if ( !(ret=realloc(ret,rlen+vlen)) )
	{
	  sos_error_printf("Error reallocating ret: %s", strerror(errno));
	  SOS_RETURN(NULL);
	}
      
      snprintf(ret+rlen, vlen+1, "%s%s", separator, value);
      rlen += vlen;
    }

  SOS_RETURN(ret);
}
