/*
 * License MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Gesso, an extension for Mozilla Firefox
 *
 * The Initial Developer of the Original Code is Ningjie (Jim) Chen.
 * Portions created by the Initial Developer are Copyright (C) 2007
 * the Initial Developer. All Rights Reserved.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the License, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the License, the GPL or the LGPL.
 *
 * See MPL.txt for terms of the Mozilla Public License Version 1.1
 * See GPL.txt for terms of the GNU General Public License Version 2.0
 * See LGPL.txt for terms of the GNU Lesser General Public License Version 2.1
 */

var vGesso, vGessoAccServ;


function GessoDebug(msg) {  
	var consoleService = Components.classes["@mozilla.org/consoleservice;1"].getService(Components.interfaces.nsIConsoleService);
	consoleService.logStringMessage(msg);
}

/**************************************************
 * 
 *  Some assumptions for GessoTextDocument:
 *   n, this.node = any interface that implements nsIDOMNSEditableElement
 *   	ex. XUL textbox, HTML input("text"), HTML textarea
 *   this.editor = nsIEditor that inherits from nsIPlaintextEditor, but NOT nsIHTMLEditor
 *   this.editor.rootElement = one HTML element (ex. div) that is the only child of its parent
 *   	and this element can have zero or more TEXT node(s) as its child(ren). 
 *   	no other node type can be its child, except maybe <br>
 *   	although <br> shouldn't exist in a "pre" style node (?)
 *   
 */
function GessoTextDocument(n, win) {
	this.node = n;
	this.editor = n.editor.QueryInterface(Components.interfaces.nsIPlaintextEditor);
	this.sel = this.editor.selection.QueryInterface(Components.interfaces.nsISelectionPrivate)
	this.sink = vGesso.addDocument(this);
	this.editing = false;
	
	this.eventHandler = function(o) {return function(e) {o.handleEvent(e);};}(this);
	win.addEventListener("unload", this.eventHandler, false);
	n.addEventListener("focus", this.eventHandler, true);
	n.addEventListener("blur", this.eventHandler, true);
	this.sel.addSelectionListener(this);
	this.editor.addEditActionListener(this);/**/
}
GessoTextDocument.prototype = {
	 // if nodeType != 3 assume <br> because that's the only other thing we allow in a plaintext editor
	GetACP: function(node, offset) {
		if (node == this.editor.rootElement && offset > 0) {
			node = node.childNodes.item(offset - 1);
			offset = node.length;
		}
		while ((node = node.previousSibling) != null)
			if (node.nodeType == 3) offset += node.length;
		return offset;
	},
	GetNodeOffset: function(acp) {
		var node = this.editor.rootElement, nodect = 0;
		if (acp == 0) return {node: node, offset: 0};
		node = node.firstChild;
		while ((acp -= (++nodect, (node.nodeType == 3 ? node.length : 0))) > 0)
			node = node.nextSibling;
		if (acp == 0) return {node: node.parentNode, offset: nodect};
		return {node: node, offset: acp + node.length};
	},
	GetTextNodeOffset: function(acp) {
		var pos = this.GetNodeOffset(acp), root = this.editor.rootElement;
		if (pos.node == root) {
			if (pos.offset > 0) {
				pos.node = root.childNodes.item(pos.offset - 1);
				pos.offset = pos.node.length;
			} else if (root.childNodes.length > 0)
				pos.node = root.childNodes.item(0);
		}
		return pos;
	},
	lock: function(locking) {
	},
	getStatus: function(stat) {
		stat.dynamicFlags = this.editor.isDocumentEditable ? 0 : stat.TS_SD_READONLY;
		stat.staticFlags = stat.TS_SS_NOHIDDENTEXT;
	},
	queryInsert: function(acpTestStart, acpTestEnd, cch, pacpResultStart, pacpResultEnd) {
		if (acpTestStart < 0 || acpTestStart > acpTestEnd || acpTestEnd > this.editor.textLength)
			throw Components.Exception("invalid position", vGesso.TS_E_INVALIDPOS);
		var end = acpTestStart + cch;
		pacpResultEnd.value = end;
		pacpResultStart.value = end; 
	},
	getSelection: function(ulIndex, pSelection) {
		if (ulIndex == 0) {
			var localsel = this.sel, ancNode = localsel.anchorNode, focNode = localsel.focusNode;
			if (!ancNode || !focNode) throw Components.Exception("no selection", vGesso.TS_E_NOSELECTION);
			var anchor = this.GetACP(ancNode, localsel.anchorOffset),
				focus = this.GetACP(focNode, localsel.focusOffset);
			if (anchor < focus) {
				pSelection.start = anchor;
				pSelection.end = focus;
				pSelection.selend = pSelection.TS_AE_END;
			}
			else {
				pSelection.start = focus;
				pSelection.end = anchor;
				pSelection.selend = anchor > focus ? pSelection.TS_AE_START : pSelection.TS_AE_NONE;
			}
		}
	},
	getSelectionCount: function(ulCount) {
		ulCount.value = 1;
	},
	setSelection: function(ulIndex, pSelection) {
		if (pSelection.start < 0 || pSelection.start > pSelection.end || pSelection.end > this.editor.textLength)
			throw Components.Exception("invalid position", vGesso.TS_E_INVALIDPOS);
		var wasediting = this.editing; this.editing = true;
		try {
			var sst = pSelection.start, sed = pSelection.end,
				st = this.GetNodeOffset(sst),
				ed = sed != sst ? this.GetNodeOffset(sed) : st,
				localsel = this.sel;
			if (pSelection.selstart == pSelection.TS_AE_END) {
				localsel.collapse(st.node, st.offset);
				localsel.extend(ed.node, ed.offset);
			} else if (sst != sed) {
				localsel.collapse(ed.node, ed.offset);
				localsel.extend(st.node, st.offset);
			}
			else localsel.collapse(st.node, st.offset);
		} catch (e) {
			throw Components.results.NS_ERROR_FAILURE;
		} finally {
			this.editing = wasediting;
		}
	},
	getText: function(acpStart, acpEnd, pchPlain, wantPlain, prgRunInfo, ulRunInfoReq, pulRunInfoOut) {
		if (acpEnd == -1) acpEnd = this.editor.textLength;
		if (acpStart < 0 || acpStart > acpEnd || acpEnd > this.editor.textLength)
			throw Components.Exception("invalid position", vGesso.TS_E_INVALIDPOS);
		var st = this.GetNodeOffset(acpStart),
			ed = this.GetNodeOffset(acpEnd);
		if (wantPlain) {
			var range = this.node.ownerDocument.createRange();
			range.setStart(st.node, st.offset);
			range.setEnd(ed.node, ed.offset);
			pchPlain.value = range.toString();
		}
		if (ulRunInfoReq > 0) {
			var run = prgRunInfo[0];
			run.count = acpEnd - acpStart;
			run.type = run.TS_RT_PLAIN;
			pulRunInfoOut.value = 1;
		}
	},
	getFormattedText: function(ulIndex, wFormat, pchFormattedText) {
		if (ulIndex == 0) {
			wFormat.value = 13; // CF_UNICODETEXT
			this.getText(this.formatStart, this.formatEnd, pchFormattedText, true, null, 0, null, 0);
		}
	},
	getFormattedTextCount: function(acpStart, acpEnd, ulCount) {
		if (acpStart < 0 || acpStart > acpEnd || acpEnd > this.editor.textLength)
			throw Components.Exception("invalid position", vGesso.TS_E_INVALIDPOS);
		this.formatStart = acpStart;
		this.formatEnd = acpEnd;
		ulCount.value = 1;
	},
	getEndACP: function(pacp) {
		pacp.value = this.editor.textLength;
	},
	requestSupportedAttrs: function(dwFlags, ulFilterAttrs, paFilterAttrs) {
		
	},
	requestAttrsAtPosition: function(acpPos, ulFilterAttrs, paFilterAttrs, dwFlags) {
		
	},
	requestAttrsTransitioningAtPosition: function(acpPos, ulFilterAttrs, paFilterAttrs, dwFlags) {
		
	},
	findNextAttrTransition: function(acpStart, acpHalt, ulFilterAttrs, paFilterAttrs, dwFlags, pacpNext, pfFound, plFoundOffset) {
		pacpNext.value = acpHalt; pfFound.value = FALSE;
	},
	retrieveRequestedAttrs: function(ulCount, paAttrVals, pcFetched) {
		pcFetched.value = 0;
	},
	getActiveView: function(pvcView) {
		pvcView.value = 1;
	},
	getACPFromPoint: function(vcView, pt, dwFlags, pacp) {
		if (vcView == 1) {
			if (acpStart < 0 || acpStart > acpEnd || acpEnd > this.editor.textLength)
				throw Components.Exception("invalid position", vGesso.TS_E_INVALIDPOS);
			var clip = new Object(), text, acc;
			try {
				text = this.editor.rootElement.firstChild;
				if (text.nodeType == 3) acc = vGessoAccServ.getAccessibleFor(text); 
				while (text.nextSibling != null) {
					if (text.nodeType == 3) {
						var rc = vGesso.accTextGetBounds(acc, 0, text.length, clip);
						if (pt.y < rc.top || (pt.y < rc.bottom && pt.x < rc.right)) break;
					}
					text = text.nextSibling;
					if (text.nodeType == 3) acc = vGessoAccServ.getAccessibleFor(text); 
				}
			} catch (e) {
				pacp.value = 0;
				return;
			}
			if (text && acc)
				pacp.value = this.GetACP(text, vGesso.accTextGetOffset(acc, pt.x, pt.y, dwFlags, text.length));
		}
	},
	// a nice little hack to make the day a little brighter, just a little
	//	- ISimpleDOMText has trouble returning bounds if the indices are the same.
	getTextExt: function(vcView, acpStart, acpEnd, prc, pfClipped) {
		if (vcView == 1) {
			var len = this.editor.textLength;
			if (len > 0) {
				if (acpStart < 0 || acpStart > acpEnd || acpEnd > len)
					throw Components.Exception("invalid position", vGesso.TS_E_INVALIDPOS);
				try {
					var l = Number.MAX_VALUE, t = l, r = -l, b = r, clip = false, clipone = new Object();
					var start = this.GetTextNodeOffset(acpStart), end = this.GetTextNodeOffset(acpEnd);
					if (end.node == start.node && end.offset == start.offset) 
						if (start.offset > 0) --start.offset;
						else ++end.offset;
					for (;;) {
						var ed = start.node == end.node ? end.offset : start.node.length - 1; 
						var rc = vGesso.accTextGetBounds(vGessoAccServ.getAccessibleFor(start.node), start.offset, ed, clipone);
						l = Math.min(l, rc.left); t = Math.min(t, rc.top);
						r = Math.max(r, rc.right); b = Math.max(b, rc.bottom);
						clip |= clipone.value;
						if (start.node == end.node) break;
						start.node = start.node.nextSibling; start.offset = 1;
					}
					prc.left = l; prc.right = r; prc.top = t; prc.bottom = b;
					pfClipped.value = clip;
					return;
				} catch (e) {
				}
			}
			this.getScreenExt(1, prc);
			pfClipped.value = true;
		}
	},
	getScreenExt: function(vcView, prc) {
		if (vcView == 1) {
			var box = this.node.ownerDocument.getBoxObjectFor(this.node);
			var x = box.screenX; prc.left = x; prc.right = x + box.width;
			var y = box.screenY; prc.top = y; prc.bottom = y + box.height; 
		}
	},
	getAccessNode: function(vcView) {
		if (vcView == 1)
			try {
				var n = this.node;
				if (n.accessible) return n.accessible.QueryInterface(Components.interfaces.nsIAccessNode);
				else return vGessoAccServ.getAccessibleFor(n).QueryInterface(Components.interfaces.nsIAccessNode);
			} catch (e) {
				throw Components.results.NS_ERROR_FAILURE;
			}
	},
	// fixed insertText() doesn't work with zero-length strings
	//	and deleteFromDocument() doesn't work if there's no selection, makes sense right?
	//		until you realize it doesn't just quit but instead throws at you a nasty exception.
	// insertText() also messes up the selection sometimes when we join the nodes in 
	//  DidInsertNode. therefore we manually fix the selection
	insertTextAtSelection: function(dwFlags, pchText, pacpStart, pacpEnd, pChange) {
		var wasediting = this.editing; this.editing = true;
		try {
			var localsel = this.sel;
			var st = this.GetACP(localsel.anchorNode, localsel.anchorOffset);
			var ed = this.GetACP(localsel.focusNode, localsel.focusOffset);
			if (st > ed) {var tmp = ed; ed = st; st = tmp;}
			if (dwFlags & vGesso.TS_IAS_QUERYONLY) {
				ed = st + pchText.length;
				pacpStart.value = ed;
				pacpEnd.value = ed;
				if (!pChange.isNull) {
					pChange.start = st;
					pChange.oldEnd = ed; 
					pChange.newEnd = st + pchText.length;
				}
			} else {
				pChange.start = st; pChange.oldEnd = ed;
				if (pchText.length > 0) {
					this.editor.insertText(pchText);
					ed = st + pchText.length;
					var sel = this.GetNodeOffset(ed);
					localsel.collapse(sel.node, sel.offset);
				}
				else if (st != ed) {
					localsel.deleteFromDocument();
					ed = st;
				}
				pChange.newEnd = ed;
				if ((dwFlags & vGesso.TF_IAS_NOQUERY) == 0) {
					pacpStart.value = this.GetACP(localsel.anchorNode, localsel.anchorOffset);
					pacpEnd.value = ed;
				}
			}
		} catch (e) {
			throw Components.results.NS_ERROR_FAILURE;
		} finally {
			this.editing = wasediting;
		}
	}, 
	handleEvent: function(e) {
		var targ = e.currentTarget;
		switch (e.type) {
		case "unload":
			targ.removeEventListener("unload", this.eventHandler, false);
			this.node.removeEventListener("focus", this.eventHandler, true);
			this.node.removeEventListener("blur", this.eventHandler, true);
			this.sel.removeSelectionListener(this);
			this.editor.removeEditActionListener(this);
			this.sink.removeDocument();
			this.sink = null; 
			this.editor = null; this.sel = null;
			GessoDebug(this.node.tagName + " unloaded");
			this.node = null;
			break;
		case "focus":
			this.sink.focus();
			break;
		case "blur":
			this.sink.unfocus();
			break;		
		}
	},
	DeleteWorker: function() {
		if (!this.editing) {
			var st = this.editStart, ed = this.editEnd;
			this.sink.onTextChange(0, {start: st, oldEnd: ed, newEnd: st});
		}
	},
	notifySelectionChanged: function(doc, sel, reason) {
		if (!this.editing) {
			this.sink.onSelectionChange();
		}
	},
	DidCreateNode: function (tag, node, parent, pos, res) {
		this.DidInsertNode(node, parent, pos, res);
	},
	DidDeleteNode: function (child, res) {
		this.DeleteWorker();
	},
	DidDeleteSelection: function(sel) {
		this.DeleteWorker();
	},
	DidDeleteText: function(node, offset, len, res) {
		this.DeleteWorker();
	},
	// insertText has this habit of creating extra text nodes
	// try fixing it here, otherwise GetACP and GetNodeText will become slower and slower
	DidInsertNode: function(node, parent, pos, res) {
		if (!this.editing) {
			if (pos >= parent.childNodes.length) pos = parent.childNodes.length - 1;
			var st = this.GetACP(parent, pos), ed = this.GetACP(parent, pos + 1); 
			this.sink.onTextChange(0, {start: st, oldEnd: st, newEnd: ed});
		} else if (node.nodeType == 3) {
			var wasediting = this.editing; this.editing = true;
			var prevsib = node.previousSibling, prevjoin = prevsib ? prevsib.nodeType == 3 : false,
				nextsib = node.nextSibling, nextjoin = nextsib ? nextsib.nodeType == 3 : false;
			if (prevjoin) this.editor.joinNodes(prevsib, node, parent);
			if (nextjoin) this.editor.joinNodes(node, nextsib, parent);
			this.editing = wasediting;
		}
	},
	DidInsertText: function(node, offset, str, res) {
		if (!this.editing) {
			var st = this.GetACP(node, offset), ed = st + str.length;
			this.sink.onTextChange(0, {start: st, oldEnd: st, newEnd: ed});
		}
	},
	DidJoinNodes: function(left, right, parent, res) {
	},
	DidSplitNode: function(right, offset, left, res) {
	},
	WillCreateNode: function(tag, parent, pos) {
	},
	WillDeleteNode: function(child) {
		if (!this.editing) {
			this.editStart = this.GetACP(child, 0);
			this.editEnd = this.editStart + child.length;
		}
	},
	WillDeleteSelection: function(sel) {
		if (!this.editing) {
			this.editStart = this.GetACP(sel.anchorNode, sel.anchorOffset);
			this.editEnd = this.GetACP(sel.focusNode, sel.focusOffset);
		}
	},
	WillDeleteText: function(node, offset, len) {
		if (!this.editing) {
			this.editStart = this.GetACP(node, offset);
			this.editEnd = this.editStart + len;
		}
	},
	WillInsertNode: function(node, parent, pos) {
	},
	WillInsertText: function(node, offset, str) {
	},
	WillJoinNodes: function(left, right, parent) {
	},
	WillSplitNode: function(right, offset) {
	}
};


const GessoObserver = {
	observe: function(subj, topic) {
		if (topic == "domwindowopened") {
			GessoDebug("Gesso: Window " + subj.location + " opened.");
			subj.addEventListener("focus", GessoWindowFocus, true);
		} else if (topic == "domwindowclosed") {
			subj.removeEventListener("focus", GessoWindowFocus, true);
			GessoDebug("Gesso: Window " + subj.location + " closed.");
		}
		else if (topic == "quit-application") {
			if (vGesso) vGesso.term();
		}
	}
};


function GessoWindowFocus(e) {
	var targ = e.currentTarget.document.commandDispatcher.focusedElement; 
	if (targ && !targ.gessoSearched) {
		targ.gessoSearched = true;
		try {
			if (targ.textbox) targ = targ.textbox; // searchbar hack
			if (!targ.editor) targ = targ.QueryInterface(Components.interfaces.nsIDOMNSEditableElement);
			new GessoTextDocument(targ, e.currentTarget);
			GessoDebug(targ.tagName + " added as text document.");
		}
		catch (e) {
		}
	}
}

function GessoWindowInit() {
	vGesso = Components.classes["@gesso.mozdev.org/gesso;1"].getService(Components.interfaces.iGesso);
	if (vGesso.init()) {
		var vWatcher = Components.classes["@mozilla.org/embedcomp/window-watcher;1"].getService(Components.interfaces.nsIWindowWatcher);
		vWatcher.registerNotification(GessoObserver);
		var observerService = Components.classes["@mozilla.org/observer-service;1"].getService(Components.interfaces.nsIObserverService);
		observerService.addObserver(GessoObserver,"quit-application",false);
		GessoObserver.observe(window, "domwindowopened");
	}
	vGessoAccServ = Components.classes["@mozilla.org/accessibilityService;1"].getService(Components.interfaces.nsIAccessibilityService);
}


// Initialize Gesso
GessoWindowInit();


