/* 
 * Copyright (C) 2003 David Bordoley
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomeui/gnome-about.h>
#include <libgnome/gnome-help.h>
#include <bonobo/bonobo-i18n.h>
#include <panel-applet.h>

typedef struct 
{
	PanelApplet base;
	
	GtkWidget *entry;
	GtkWidget *event_box;
} OpenApplet;

static gchar	*compute_url 			(const gchar *input);
static void 	 open_applet_about_cb 		(BonoboUIComponent *uic,
						 OpenApplet *applet);
static void 	 open_applet_help_cb 		(BonoboUIComponent *uic,
						 OpenApplet *applet);

static const BonoboUIVerb open_applet_menu_verbs [] = {
	BONOBO_UI_UNSAFE_VERB ("OpenAppletHelp", open_applet_help_cb),
	BONOBO_UI_UNSAFE_VERB ("OpenAppletAbout", open_applet_about_cb),
	BONOBO_UI_VERB_END
};

static GType
open_applet_get_type (void)
{
	static GType type = 0;

	if (!type) 
	{
		static const GTypeInfo info = 
			{sizeof (PanelAppletClass),
			 NULL, NULL, NULL, NULL, NULL,
			 sizeof (OpenApplet),
			 0, NULL, NULL};

		type = g_type_register_static (PANEL_TYPE_APPLET, "Open_Applet", &info, 0);
	}

	return type;
}

/************************************ Accessory Functions ************************************/

static void
open_applet_help (GtkWindow  *parent,
		  const char *file_name,
		  const char *link_id)
{
	GError *err = NULL;

	gnome_help_display (file_name, link_id, &err);

	if (err != NULL)
	{
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new (parent,
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _("Could not display help: %s"), err->message);
		g_signal_connect (G_OBJECT (dialog), "response",
				  G_CALLBACK (gtk_widget_destroy),
				  NULL);
		gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
		gtk_widget_show (dialog);
		g_error_free (err);
	}
}

/************************************ Callbacks ************************************/

static gboolean
button_pressed_event_box_cb (GtkWidget		*widget,
			     GdkEventButton 	*event,
			     OpenApplet		*applet)
{
	gtk_widget_grab_focus (applet->entry);
	
	/* FIXME: Implement me please :) */
	/* Show the entry context menu if the user throws the
	 * mouse at the screen edge.
	 */
	if (event->button == 3 && event->type == GDK_BUTTON_PRESS)
	{
		return TRUE;
	}
	return FALSE;
} 
  
static void
activate_entry_cb (GtkWidget *entry, OpenApplet *applet)
{
	gchar *entry_text;
	gchar *url;

	entry_text = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
	gtk_entry_set_text (GTK_ENTRY (entry), "");

	url = compute_url (entry_text);
	
	if (url)
	{
		gnome_vfs_url_show (url);
		g_free (url);
	}

	g_free (entry_text);
}

static gboolean
open_applet_destroy_cb (OpenApplet		*applet,
		        gpointer		 data)
{
	return FALSE;
}

static void
open_applet_help_cb (BonoboUIComponent 	*uic, 
		     OpenApplet 	*applet)
{
	open_applet_help (NULL, "open-applet-help", NULL);
}

static void
open_applet_about_cb (BonoboUIComponent	*uic, 
		      OpenApplet 	*applet)
{
	static GtkWidget *about = NULL;
	GdkPixbuf *pixbuf;
	char *file;
	const gchar *authors[] = {
		"David Bordoley <bordoley@msu.edu>",
		NULL};
	const gchar *documenters[] = { NULL };
	const gchar *translator_credits = _("translator_credits");

	if (about != NULL)
	{
		gtk_widget_show (about);
		gtk_window_present (GTK_WINDOW (about));
		return;
	}

	file = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP,
			"open-applet/open-applet-about.png", FALSE, NULL);
	pixbuf = gdk_pixbuf_new_from_file (file, NULL);
	g_free (file);

	about = gnome_about_new (
			_("Open Applet"),
			VERSION,
			_("(C) 2003 David Bordoley "),
			_("A text box to open stuff with"),
			authors,
			documenters,
			strcmp (translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
			pixbuf);

	g_object_unref (pixbuf);
	
	gtk_widget_show (about);

	g_object_add_weak_pointer (G_OBJECT (about),
			(void**)&about);

	return;
}

/************************************ Build the GUI ************************************/

static void
open_applet_connect_signals (OpenApplet *applet)
{
	g_signal_connect (G_OBJECT (applet->event_box), "button-press-event",
			  G_CALLBACK (button_pressed_event_box_cb), applet);
	g_signal_connect (G_OBJECT (applet->entry), "activate",
			  G_CALLBACK (activate_entry_cb), applet);
}

/* FIXME: I would like to add a slight border around the text entry 
 * so that it doesn't rub up against the screen edge. However, its 
 * important to ensure that you can still give focus to the entry by
 * slamming the mouse on the edge of the screen and by clicking the 
 * grey area.
 */
/* FIXME: Wrap the open label in an event box, so that when you click on it,
 * the entry gets focus. It should be a seperate event box though, because right
 * clicking on open should give you the applet/panel menu not the entry context 
 * menu.
 */
/* FIXME: Make the applet a drop destination for text. */
/* FIXME: Support middle click pasting on Open: */
static void
open_applet_build (OpenApplet *applet)
{	
	GtkWidget *separator;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *event_box;
	GtkWidget *hbox;
	
	separator = gtk_vseparator_new ();
	label = gtk_label_new (_("Open:"));
	entry = gtk_entry_new ();
	event_box = gtk_event_box_new ();
	hbox = gtk_hbox_new (FALSE, 3);

	gtk_container_add (GTK_CONTAINER (event_box), entry);

	gtk_box_pack_start (GTK_BOX (hbox), separator, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), event_box, FALSE, FALSE, 0);
	
	gtk_container_add (GTK_CONTAINER (applet), hbox);

	gtk_widget_show_all (GTK_WIDGET (applet));

	applet->entry = entry;
	applet->event_box = event_box;

	open_applet_connect_signals (applet);
}

static gboolean
open_applet_new (OpenApplet *applet)
{
	open_applet_build (applet);
	return TRUE;
}

static gboolean
open_applet_factory (OpenApplet		*applet,
		     const gchar	*iid,
		     gpointer		 data)
{
	gboolean retval = FALSE;
	
	panel_applet_set_flags (PANEL_APPLET (applet), PANEL_APPLET_EXPAND_MINOR);
	
	if (!strcmp (iid, "OAFIID:GNOME_Open_Applet"))
		retval = open_applet_new (applet);
	
	g_signal_connect (G_OBJECT (applet), "destroy",
			  G_CALLBACK (open_applet_destroy_cb), NULL);

	panel_applet_setup_menu_from_file (PANEL_APPLET (applet), 
					   NULL,
					   "GNOME_Open_Applet.xml",
					   NULL,
					   open_applet_menu_verbs,
					   applet);

	return retval;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_Open_Applet_Factory",
			     open_applet_get_type (),
			     "Open_Applet",
			     "0",
			     (PanelAppletFactoryCallback) open_applet_factory,
			     NULL)

/************************************ URL Parsing Functions ************************************/

static gboolean
is_url (const gchar *input)
{
	gchar *scheme;

	scheme = gnome_vfs_get_uri_scheme (input);

	/* We assume that if the user provides input 
	 * with a scheme than it's a url. 
	 */
	if (scheme)
	{
		g_free (scheme);
		return TRUE;
	}

	return FALSE;
}

/* Really crude hack.
 * If the input has any ' ' characters fail.
 * If the input has no '.' characters fail.
 */
static gboolean
is_web_address (const gchar *input)
{
	int length = strlen (input);
	int i;
	int period_count = 0;

	for (i = 0; i < length; i++)
	{
		if (input[i] == ' ')
		{
			return FALSE;
		}
		else if (input[i] == '.')
		{
			period_count++;
		}
	}

	if (period_count > 0)
	{
		return TRUE;
	}

	return FALSE;
}

static gboolean
is_email_address (const gchar *input_copy)
{
	int length = strlen (input_copy);
	int i;
	int amphersand_count = 0;

	if (is_web_address (input_copy))
	{
		for (i = 0; i < length; i++)
		{
			if (input_copy[i] == '@')
			{
				amphersand_count++;
			}

			if (amphersand_count > 1)
			{
				return FALSE;
			}
		}
	}

	if (amphersand_count < 1)
	{
		return FALSE;
	}
	return TRUE;
}

/* Guaranteed to return a valid url, otherwise will return NULL */
static gchar*
compute_url (const gchar *input)
{
	gchar *input_copy = g_strdup (input);
	gchar *url = NULL;		
	gchar *retval = NULL;	/* Return NULL if we can't compute a URL */

	/* Remove any leading or trailing white space */
	g_strstrip (input_copy);

	/* Check if the input is a url */
	if (is_url (input_copy))
	{
		url = g_strdup (input_copy);
	}

	/* If the first character is '/', 
	 * we assume that its a local path 
	 */
	else if (input_copy[0] == '/')
	{
		url = gnome_vfs_get_uri_from_local_path (input_copy);
	}

	/* If the first character is '~', 
	 * we assume that its a local path 
	 * starting in the homedir. 
	 */
	else if (input_copy[0] == '~')
	{ 
		gchar *temp;
	
		temp = gnome_vfs_expand_initial_tilde (input_copy);
		url = gnome_vfs_get_uri_from_local_path (temp);
	
		g_free (temp);
	}

	/* Check for common prefixes */
	else if (g_str_has_prefix (input_copy, "irc."))
	{
		url = g_strconcat("irc://", input_copy, NULL); 
	}
	else if (g_str_has_prefix (input_copy, "ftp."))
	{
		url = g_strconcat("ftp://", input_copy, NULL); 
	}
	else if (g_str_has_prefix (input_copy, "www."))
	{
		url = g_strconcat("http://", input_copy, NULL); 
	}

	/* Check if the input is an email address. */
	else if (is_email_address (input_copy))
	{
		url = g_strconcat("mailto://", input_copy, NULL);
	}	

	/* Check if the input is something resembling 
	 * a web address. If so append http:// to it and
	 * hope it works.
	 */ 
	else if (is_web_address(input_copy))
	{
		url = g_strconcat("http://", input_copy, NULL);
	}

	/* Worst comes to worst search with google, as long as you don't have an empty string */
	/* NOTE: Or if medusa supports searching natural language phrases, 
	 * we could do that instead. One option would be to support keywords
	 * such that "google 'phrase'" searches google while by default we search
	 * the filesystem (smart bookmarks might work here too). 
	 * This is all moot until medusa actually works though. 
	 */
	else if (strcmp (input_copy, "") > 0)
	{
		url = g_strdup_printf ("http://www.google.com/search?q=%s&ie=UTF-8&oe=UTF-8",
					  input_copy);
	}
	
	g_free (input_copy);

	/* canonicalized url */
	if (url)
	{
		retval = gnome_vfs_make_uri_canonical (url);
		g_free (url);
	}

	return retval;
}
