/*-*- Mode: C; c-basic-offset: 8; indent-tabs-mode: nil -*-*/

/***
  This file is part of systemd.

  Copyright 2010 Lennart Poettering

  systemd is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 2.1 of the License, or
  (at your option) any later version.

  systemd is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with systemd; If not, see <http://www.gnu.org/licenses/>.
***/

#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <ftw.h>

#include "cgroup-util.h"
#include "cgmanager.h"
#include "log.h"
#include "set.h"
#include "macro.h"
#include "util.h"
#include "mkdir.h"

int cg_create(const char *controller, const char *path, const char *suffix) {
        _cleanup_free_ char *fs = NULL;
        int r;

        #ifdef HAVE_CGMANAGER
        /* CGManager support */
        int existed;
        if (cgm_dbus_connect()) {
                if (!controller) {
                    cgm_dbus_disconnect();
                    return -1;
                }

                r = cgm_create(normalize_controller(controller),
                                         path, &existed);

                cgm_dbus_disconnect();

                if (!r)
                        return -1;

                return 1;
        }
        #endif

        r = cg_get_path_and_check(controller, path, suffix, &fs);
        if (r < 0)
                return r;

        r = mkdir_parents_label(fs, 0755);
        if (r < 0)
                return r;

        if (mkdir(fs, 0755) < 0) {

                if (errno == EEXIST)
                        return 0;

                return -errno;
        }

        return 1;
}

int cg_create_and_attach(const char *controller, const char *path, pid_t pid) {
        int r, q;

        assert(pid >= 0);

        r = cg_create(controller, path, NULL);
        if (r < 0)
                return r;

        q = cg_attach(controller, path, pid);
        if (q < 0)
                return q;

        /* This does not remove the cgroup on failure */
        return r;
}
